import React, { useState, useEffect, useMemo, useRef } from "react";
import { useApolloClient, ApolloQueryResult } from "@apollo/client";
import { Segment, Button, Tab, Dropdown, Message, Header, Icon } from "semantic-ui-react";
import {
  useDevicesByAspectTypesQuery,
  useStoragePoolsQuery,
  useZonesQuery,
  DeviceFunctionalAspectType,
  useDiscoverDevicesLazyQuery,
  DiscoverDevicesQuery,
  DeviceBaseConfigType,
  CreateDeviceMutation,
  CreateDeviceMutationVariables,
  CreateDeviceDocument,
  DeviceInput,
  DeviceBaseConfigInput,
  MediaArchiveOption,
  ProbeDeviceQuery,
  ProbeDeviceQueryVariables,
  ProbeDeviceDocument, useDevicesByAspectTypesShortQuery
} from "@generated/graphql";
import WithQueryStatus from "components/WithQueryStatus";
import Loading from "components/Loading";
import CameraOnvif from "./CameraOnvif";
import CameraOnboard from "./CameraOnboard";
import CameraDemo from "./CameraDemo";
import { PropType, queryToInput } from "utils";
import {aspectToAspectInput, Device, useDeviceActions} from "@core/actions";
import { AutoForm, AutoLayout, FieldValues, FormSchema, DataSourceItem } from "components/AutoForm";
import {Log} from "@solid/libs/log";
import {__} from "@solid/libs/i18n";
import { useAccessability } from "@core/store/actions/accessability";

import "./style.css";

type CameraDiscoveryProps = {
  linkId?: string;
  onBack?: (createdDeviceIds?: string[]) => void;
};

type DeviceProbe = PropType<DiscoverDevicesQuery, "discoverDevices">[0];

export type DiscoveredDevice = DeviceProbe & {
  device?: Device; // Reference to already registered device if available
  userInput: DeviceUserInput;
};

export type DeviceUserInput = {
  name: string;
  userName: string;
  password: string;
  isSelected: boolean;
  streamName?: string;
  audio?: string | boolean;
  validationError?: string;
  isProbed?: boolean;
  isLoading?: boolean;
};

const archiveOptions: DataSourceItem[] = [
  { id: MediaArchiveOption.None, name: __("None") },
  { id: MediaArchiveOption.Continuous, name: __("Continuous") },
  { id: MediaArchiveOption.Events, name: __("Events") },
  { id: MediaArchiveOption.Schedule, name: __("Schedule") },
];

const CameraDiscovery = ({ onBack, ...props }: CameraDiscoveryProps) => {
  const { data: linkData, error: linkError, loading: linkLoading } = useDevicesByAspectTypesShortQuery({ variables: { types: [{ type: DeviceFunctionalAspectType.Avatar }] }, fetchPolicy: "no-cache" });
  const { data: devData, error: devError, loading: devLoading, refetch: devRefetch } = useDevicesByAspectTypesQuery({ variables: { types: [{ type: DeviceFunctionalAspectType.Media }] }, fetchPolicy: "no-cache" });
  const { data: zoneData, error: zoneError, loading: zoneLoading } = useZonesQuery();
  const { data: storageData, error: storageError, loading: storageLoading } = useStoragePoolsQuery();
  const [discoverDevices, { data: discData, error: discError, loading: discLoading }] = useDiscoverDevicesLazyQuery({ fetchPolicy: "no-cache" });
  const { config: { limitedZonesAccess } } = useAccessability();
  const { updateCachedDevice } = useDeviceActions({skipSubscription: true});
  const links = useMemo(() => linkData?.devicesByAspectTypes
    .filter(avatar => avatar.aspects.some(aspect => aspect.__typename === "DFA_Avatar" && aspect.isLink))
    .sort((a, b) => a.name.localeCompare(b.name, undefined, { sensitivity: "base" })) ?? [], [linkData]);
  const linkOptions = useMemo(() => links.map(({ id, name }) => ({ key: id, value: id, text: name })), [links]);
  const [linkId, setLinkId] = useState("");
  const [discoveredLinkId, setDiscoveredLinkId] = useState("");
  const [devices, setDevices] = useState<DiscoveredDevice[]>([]);
  const devicesByTypes = useMemo(() => {
    const map: Record<DeviceBaseConfigType, DiscoveredDevice[]> = {
      [DeviceBaseConfigType.Onvif]: devices.filter(dev => dev.config.configType === DeviceBaseConfigType.Onvif),
      [DeviceBaseConfigType.Onboard]: devices.filter(dev => dev.config.configType === DeviceBaseConfigType.Onboard),
      [DeviceBaseConfigType.Demo]: devices.filter(dev => dev.config.configType === DeviceBaseConfigType.Demo),
      [DeviceBaseConfigType.Url]: [],
      [DeviceBaseConfigType.Sensor]: [],
      [DeviceBaseConfigType.Gateway]: []
    };
    return map;
  }, [devices]);
  const [activeTab, setActiveTab] = useState(0);
  const client = useApolloClient();
  const [updating, setUpdating] = useState(false);

  const propsSchema = useMemo(() => {
    const schema: FormSchema = [
      {
        name: "location",
        label: __("Location"),
      },
      !limitedZonesAccess ?
        {
          name: "zoneId",
          label: __("Zone"),
          type: "dropdown",
          dataSource: Array.from(zoneData?.zones ?? []).sort((a, b) => a.name.localeCompare(b.name, undefined, { sensitivity: "base" }))
        } :
        undefined,
      {
        name: "edgeArchive",
        label: __("Edge Archive"),
        type: "dropdown",
        required: true,
        dataSource: archiveOptions
      },
      {
        name: "cloudArchive",
        label: __("Cloud Archive"),
        type: "dropdown",
        required: true,
        dataSource: archiveOptions
      },
      {
        name: "storagePoolId",
        label: __("Storage Pool"),
        type: "dropdown",
        required: true,
        dataSource: Array.from(storageData?.storagePools ?? []).sort((a, b) => a.name.localeCompare(b.name, undefined, { sensitivity: "base" }))
      },
    ];
    return schema;
  }, [zoneData, storageData, limitedZonesAccess]);

  const propsFormRef = useRef<AutoForm>(null);

  useEffect(() => {
    if (props.linkId) {
      setLinkId(props.linkId);
    }
    else {
      setLinkId(linkOptions.length > 0 ? linkOptions[0].value : "");
    }
  }, [linkOptions, props.linkId]);

  useEffect(() => {
    if (!discData) {
      setDevices([]);
      return;
    }
    setDevices(discData.discoverDevices.map(probe => {
      const device = getAddedDevice(probe);
      return {
        ...probe,
        device,
        userInput: {
          name: probe.name || probe.config.model || "",
          userName: device?.config.connect.user ?? "",
          password: device?.config.connect.pass ?? "",
          streamName: probe.aspects.find(aspect => aspect.template.id === "video")?.name,
          audio: probe.aspects.find(aspect => aspect.template.id === "audio")?.name || false,
          isSelected: false,
        }
      };
    }));
  }, [discData, devData]);

  function getAddedDevice(probe: DeviceProbe): Device | undefined {
    if (!devData) {
      return undefined;
    }
    switch (probe.config.configType) {
      case DeviceBaseConfigType.Onvif:
        return devData.devicesByAspectTypes.find(dev =>
          dev.platform?.id === linkId &&
          probe.config.configType === dev.config.configType &&
          probe.config.connect.host &&
          dev.config.connect.host === probe.config.connect.host &&
          (dev.config.connect.port ?? 80) === (probe.config.connect.port ?? 80));

      case DeviceBaseConfigType.Onboard:
        return devData.devicesByAspectTypes.find(dev =>
          dev.platform?.id === linkId &&
          probe.config.configType === dev.config.configType &&
          probe.config.model &&
          dev.config.model === probe.config.model);

      case DeviceBaseConfigType.Demo:
        return devData.devicesByAspectTypes.find(dev =>
          dev.platform?.id === linkId &&
          probe.config.configType === dev.config.configType &&
          probe.name &&
          dev.config.model === probe.name);

      default:
        return undefined;
    }
  }

  function onDiscoverClick(): void {
    setActiveTab(0);
    setDiscoveredLinkId(linkId);
    discoverDevices({ variables: { platformId: linkId }});
  }

  function onUserInput(device: DiscoveredDevice, input: DeviceUserInput): void {
    setDevices(devices => {
      const index = devices.indexOf(device);
      if (index < 0) {
        return devices;
      }

      const currentInput = { ...input };
      if (devices[index].userInput.streamName !== currentInput.streamName) {
        const videoAspect = devices[index].aspects.find(aspect =>
          aspect.__typename === "DFA_Media" &&
          aspect.template.id === "video" &&
          currentInput.streamName === aspect.name &&
          aspect.onvifProfile
        );

        const audioAspect = devices[index].aspects.find(aspect =>
          aspect.__typename === "DFA_Media" &&
          aspect.template.id === "audio" &&
          //@ts-ignore
          aspect.onvifProfile === videoAspect?.onvifProfile
        );

        currentInput.audio = audioAspect ? audioAspect.name : false;
      }

      const newDevices = Array.from(devices);
      const userInput = { ...currentInput };
      if (userInput.isSelected && !userInput.name) {
        userInput.validationError = __("Name should be not empty");
      }
      else {
        userInput.validationError = undefined;
      }
      newDevices[index] = { ...newDevices[index], userInput };
      const newDevice = newDevices[index];
      if (newDevice.config.configType === DeviceBaseConfigType.Onvif &&
          newDevice.userInput.isSelected &&
          !device.userInput.isSelected &&
          !newDevice.userInput.isProbed) {
        probeOnvifCamera(newDevice);
      }
      return newDevices;
    });
  }

  async function probeOnvifCamera(device: DiscoveredDevice): Promise<void> {
    setDevices(devices => {
      const index = devices.findIndex(dev =>
        dev.config.configType === device.config.configType &&
        dev.config.connect.host === device.config.connect.host &&
        (dev.config.connect.port ?? 80) === (device.config.connect.port ?? 80));
      if (index < 0) {
        return devices;
      }
      const newDevices = Array.from(devices);
      newDevices[index] = { ...newDevices[index], userInput: { ...newDevices[index].userInput, isLoading: true } };
      return newDevices;
    });

    const { configType, connect } = device.config;
    const { userName, password } = device.userInput;

    let result: ApolloQueryResult<ProbeDeviceQuery> | undefined;
    try {
      result = await client.query<ProbeDeviceQuery, ProbeDeviceQueryVariables>({
        query: ProbeDeviceDocument,
        variables: {
          input: {
            configType,
            connect: {
              ...queryToInput(connect),
              user: userName,
              pass: password
            },
            platformId: discoveredLinkId
          }
        },
        fetchPolicy: "no-cache",
        errorPolicy: "all"
      });
    }
    catch (e: any) {
      Log.error(e.message);
    }
    finally {
      setDevices(devices => {
        const index = devices.findIndex(dev =>
          dev.config.configType === device.config.configType &&
          dev.config.connect.host === device.config.connect.host &&
          (dev.config.connect.port ?? 80) === (device.config.connect.port ?? 80));
        if (index < 0) {
          return devices;
        }
        const newDevices = Array.from(devices);
        if (result?.data) {
          newDevices[index] = {
            ...result.data.probeDevice,
            device: newDevices[index].device,
            userInput: {
              ...newDevices[index].userInput,
              streamName: result.data.probeDevice.aspects.find(aspect => aspect.template.id === "video" && aspect.name === newDevices[index].userInput.streamName)?.name ??
                result.data.probeDevice.aspects.find(aspect => aspect.template.id === "video")?.name,
              audio: result.data.probeDevice.aspects.find(aspect => aspect.template.id === "audio" && aspect.name === newDevices[index].userInput.audio)?.name ??
                result.data.probeDevice.aspects.find(aspect => aspect.template.id === "audio")?.name,
              isProbed: true,
              isLoading: false
            }};
        }
        else {
          if (result?.errors && result.errors.length > 0) {
            Log.error(result.errors[0].message);
          }
          newDevices[index] = {
            ...newDevices[index],
            userInput: { ...newDevices[index].userInput, isSelected: false, isProbed: false, isLoading: false }};
        }
        return newDevices;
      });
    }
  }

  function onPropsFormChange(name: string, value: any, values: FieldValues, form: AutoForm): void {
    const edgeArchive: MediaArchiveOption = values["edgeArchive"];
    const cloudArchive: MediaArchiveOption = values["cloudArchive"];
    if (name === "edgeArchive") {
      switch (edgeArchive) {
        case MediaArchiveOption.None:
          form.setValue("cloudArchive", MediaArchiveOption.None);
          break;
        case MediaArchiveOption.Events:
        case MediaArchiveOption.Schedule:
          if (cloudArchive !== MediaArchiveOption.None) {
            form.setValue("cloudArchive", edgeArchive);
          }
          break;
      }
    }
    if (name === "cloudArchive") {
      switch (cloudArchive) {
        case MediaArchiveOption.Continuous:
          form.setValue("edgeArchive", MediaArchiveOption.Continuous);
          break;
        case MediaArchiveOption.Events:
        case MediaArchiveOption.Schedule:
          if (edgeArchive !== MediaArchiveOption.Continuous) {
            form.setValue("edgeArchive", cloudArchive);
          }
          break;
      }
    }
  }

  function DeviceToInput(device: DiscoveredDevice): DeviceInput {
    if (!propsFormRef.current) {
      throw new Error("Properties form reference is NULL");
    }

    const config: DeviceBaseConfigInput = queryToInput(device.config);
    if (device.config.configType === DeviceBaseConfigType.Onvif) {
      config.connect.user = device.userInput.userName;
      config.connect.pass = device.userInput.password;
      config.connect.port = config.connect.port || 80;
      config.connect.rtspPort = config.connect.rtspPort || 554;
    }
    if (device.config.configType === DeviceBaseConfigType.Demo) {
      config.model = config.model || device.name;
      config.connect.port = config.connect.port || 80;
      config.connect.rtspPort = config.connect.rtspPort || 554;
      config.connect.host = config.connect.URL;
    }

    const propsValues = propsFormRef.current.getValues();

    const videoAspects = device.aspects
      .filter(aspect => aspect.template.id === "video")
      .map(aspect => aspectToAspectInput(aspect));

    const otherAspects = device.aspects
      .filter(aspect => aspect.template.id !== "video" && aspect.template.id !== "audio")
      .map(aspect => aspectToAspectInput(aspect));

    for (let i = 0; i < videoAspects.length; i++) {
      const aspect = videoAspects[i];
      aspect.enabled = aspect.name === device.userInput.streamName || (i === 0 && !device.userInput.streamName);
      if (aspect.enabled) {
        aspect.edgeArchive = propsValues["edgeArchive"];
        aspect.cloudArchive = propsValues["cloudArchive"];
      }
    }

    let aspects = videoAspects.concat(otherAspects);

    const audioAspects = device.aspects
      .filter(aspect => aspect.__typename === "DFA_Media" && aspect.template.id === "audio");

    if (audioAspects.length > 0) {
      const audioAspectsInput = audioAspects
        .map(aspect => {
          aspect.enabled = !!device.userInput.audio;
          return aspectToAspectInput(aspect);
        });
      aspects = aspects.concat(audioAspectsInput);
    }

    const input: DeviceInput = {
      name: device.userInput.name,
      enabled: true,
      platformId: discoveredLinkId,
      location: propsValues["location"],
      zoneId: propsValues["zoneId"],
      config,
      storageConfig: {
        storagePoolId: propsValues["storagePoolId"],
      },
      aspects,
    };
    return input;
  }

  async function onAddSelected(): Promise<void> {
    const devToAdd = devices.filter(dev => dev.userInput.isSelected && !dev.userInput.validationError);
    if (devToAdd.length === 0) {
      onBack && onBack([]);
      return;
    }

    let canAdd = true;
    for (const device of devices) {
      if (device.userInput.isSelected && !device.userInput.streamName) {
        Log.error(__("Camera '{{name}}' does not have a video stream.", {name: device.userInput.name}));
        canAdd = false;
      }
    }
    if (!canAdd) {
      return;
    }

    if (!propsFormRef.current) {
      Log.error(__("Properties form reference is NULL"));
      return;
    }

    if (!propsFormRef.current.validate()) {
      setActiveTab(1);
      return;
    }

    const createdIds: string[] = [];
    setUpdating(true);
    try {
      const results = await Promise.all(devToAdd.map(device => client.mutate<CreateDeviceMutation, CreateDeviceMutationVariables>({
        mutation: CreateDeviceDocument,
        variables: { device: DeviceToInput(device) },
        errorPolicy: "all"
      })));
      for (let i = 0; i < results.length; i++) {
        const result = results[i];
        if (result.data) {
          await updateCachedDevice(result.data.createDevice.id);
          createdIds.push(result.data.createDevice.id);
        }
        if (result.errors && result.errors.length > 0) {
          Log.error(__("Camera '{{name}}' could not be added: {{message}}", {name: devToAdd[i].userInput.name, message: result.errors[0].message}));
        }
        if (result.data && result.data.createDevice.warning) {
          Log.error(__("Camera '{{name}}' warning: {{message}}", {name: devToAdd[i].userInput.name, message: result.data.createDevice.warning}));
        }
      }
      await devRefetch();
    }
    catch (e: any) {
      Log.error(e.message);
    }
    finally {
      setUpdating(false);
      onBack && onBack(createdIds);
    }
  }

  const linkName = links.find(link => link.id === discoveredLinkId)?.name;

  /* eslint-disable react/jsx-indent */
  const panes = [
    {
      menuItem: __("Discovered Cameras") + (linkName ? ` - ${linkName}` : ""),
      pane:
        <Tab.Pane key="discovery" className="CameraDiscovery-DiscoveryPane">
          {!!discError && <Message error content={discError.message}/>}

          <div className="CameraDiscovery-Content">
            {devicesByTypes[DeviceBaseConfigType.Onvif].length > 0 &&
            <>
              <Header>{__("ONVIF Cameras")}</Header>
              <div className="CameraDiscovery-CameraList">
                {devicesByTypes[DeviceBaseConfigType.Onvif].map((device, index) =>
                  <CameraOnvif
                    key={`${device.config.model}_${index}`}
                    device={device}
                    onUserInput={onUserInput}
                    onTestClick={device => probeOnvifCamera(device)}
                  />)}
              </div>
            </>}
            {devicesByTypes[DeviceBaseConfigType.Onboard].length > 0 &&
            <>
              <Header>{__("Attached Cameras")}</Header>
              <div className="CameraDiscovery-CameraList">
                {devicesByTypes[DeviceBaseConfigType.Onboard].map((device, index) =>
                  <CameraOnboard key={`${device.config.model}_${index}`} device={device} onUserInput={onUserInput}/>)}
              </div>
            </>}
            {devicesByTypes[DeviceBaseConfigType.Demo].length > 0 &&
            <>
              <Header>{__("DEMO Cameras")}</Header>
              <div className="CameraDiscovery-CameraList">
                {devicesByTypes[DeviceBaseConfigType.Demo].map((device, index) =>
                  <CameraDemo key={`${device.name}_${index}`} device={device} onUserInput={onUserInput}/>)}
              </div>
            </>}
          </div>
        </Tab.Pane>
    },
    {
      menuItem: __("Cameras Properties"),
      pane:
        <Tab.Pane key="props" className="CameraDiscovery-PropertiesPane">
          <Message info content={__("Specify properties to be applied to all cameras you select for addition")}/>

          <AutoForm ref={propsFormRef} schema={propsSchema} onChange={onPropsFormChange}>
            <AutoLayout/>
          </AutoForm>
        </Tab.Pane>
    }
  ];
  /* eslint-enable react/jsx-indent */

  return (
    <Segment className="CameraDiscovery">
      <WithQueryStatus
        loading={linkLoading || devLoading || storageLoading || zoneLoading}
        error={linkError || devError || storageError || zoneError}>

        <div className="CameraDiscovery-LinkSelection">
          <div className="CameraDiscovery-LinkLabel">{__("Avatar")}</div>
          <Dropdown
            fluid
            selection
            search
            options={linkOptions}
            value={linkId}
            disabled={!!props.linkId}
            onChange={(e, { value }) => setLinkId(typeof value === "string" ? value : "")}
          />
        </div>
        <div className="CameraDiscovery-TopButtons">
          <Button onClick={() => onBack && onBack()}>
            <Icon name="arrow left"/>{__("Back")}
          </Button>
          <Button
            icon="globe"
            content={__("Discover")}
            disabled={!linkId || devices.some(dev => dev.userInput.isLoading)}
            onClick={onDiscoverClick}
          />
          <Button
            positive
            icon="plus"
            content={__("Add Selected")}
            disabled={!devices.some(dev => dev.userInput.isSelected) ||
              devices.some(dev => (dev.userInput.isSelected && dev.userInput.validationError) || dev.userInput.isLoading)}
            onClick={onAddSelected}
          />
        </div>

        <Tab
          panes={panes}
          className="CameraDiscovery-Tab"
          renderActiveOnly={false}
          activeIndex={activeTab}
          onTabChange={(e, { activeIndex }) => typeof activeIndex === "number" && setActiveTab(activeIndex)}
        />

        {discLoading && <Loading/>}
        {updating && <Loading text={__("Updating...")}/>}

      </WithQueryStatus>
    </Segment>
  );
};

export default CameraDiscovery;
