import fs from "fs";
import dotenv, {DotenvParseOutput} from "dotenv";
import * as logger from "./src/electron/logger";
import path from "path";
import {MSEMediaPlayerConnection} from "@solid/types/player";
import * as os from "os";

export interface ENV {
  ELECTRON_ENABLE_LOGGING: "true" | "false",
  START_URL: string,
  DATA_DIR: string,
  API_USER: string
  API_PASS: string
  API_GRAPHQL_CODEGEN: string,
  API_BACKEND: string,
  API_GRAPHQL: string,
  API_MINIDUMP: string,
  CREDENTIALS: "same-origin",
  FORCE_EGL: "yes" | "no",
  FORCE_WEBGL: "yes" | "no",
  NODE_ENV: "development" | "production",
  PLAYER: "mse" | "native",
  PLAYER_CONNECTION: MSEMediaPlayerConnection,
  PLAYER_LOG: "INFO" | "DEBUG",
  PLAYER_DEFAULT_SPEED: string,
  THEME: "purple",
  BUILD_TARGET: "browser" | "electron",
  OPEN_WEB_TOOLS: "yes" | "no"
}

export const ENVDefault: ENV = {
  ELECTRON_ENABLE_LOGGING: "true",
  START_URL: "",
  DATA_DIR: path.join(os.tmpdir(), "solid_storage"),
  API_USER: "",
  API_PASS: "",
  API_GRAPHQL_CODEGEN: "",
  API_BACKEND: "",
  API_GRAPHQL: "/api/graphql",
  API_MINIDUMP: "",
  CREDENTIALS: "same-origin",
  FORCE_EGL: "no",
  FORCE_WEBGL: "no",
  NODE_ENV: "development",
  PLAYER: "native",
  PLAYER_CONNECTION: MSEMediaPlayerConnection.AUTO,
  PLAYER_LOG: "INFO",
  PLAYER_DEFAULT_SPEED: "1",
  THEME: "purple",
  BUILD_TARGET: "electron",
  OPEN_WEB_TOOLS: "no"
};

declare global {
  namespace NodeJS {
    interface ProcessEnv extends ENV {}
  }
}

let isEnvLoaded = false;
export function loadENV(path = ".env") {
  if (isEnvLoaded) {
    return;
  }
  try {
    loadDefaultEnv();
    load(path);
    isEnvLoaded = true;
  }
  catch (e) {
    console.error(e);
  }
}

export function loadDefaultEnv() {
  for (const key in ENVDefault) {
    if (!(key in process.env)) {
      process.env[key] = ENVDefault[key];
    }
  }
}

export function load(envFile: string): DotenvParseOutput | undefined {
  envFile = path.resolve(__dirname, envFile);
  try {
    if (!fs.existsSync(envFile)) {
      logger.warn(`environment file '${envFile}' is not found, using defaults and ENV vars`);
      return {};
    }

    const envConfig = dotenv.parse(fs.readFileSync(envFile));
    for (const key in envConfig) {
      process.env[key] = envConfig[key];
    }

    console.log("environment loaded from", envFile);
    console.table(envConfig);

    return envConfig;
  }
  catch (e: any) {
    console.warn(e.message);
  }

  return undefined;
}
