import React, {useCallback, useEffect, useRef} from "react";
import {TimelineOptions} from "vis-timeline";
import {TimeLineEventInfo, TimeLineTrack, TimeLineType, VisTimeLine} from "@solid/timeline/vistimeline";
import {Granularity} from "@solid/types/coverage";
import {UUID} from "@solid/types";

type TimeLineProps = {
  type?: TimeLineType,
  isDebug?: boolean,
  className?: string,
  options?: TimelineOptions,
  hideSelection?: boolean,
  showTime?: boolean,
  onLoad: (timeline: VisTimeLine) => void,
  onGetData: (beginTime: number, endTime: number, setData: (list: TimeLineTrack[]) => void) => void,
  onGetEventInfo?: (startTime: number, endTime: number, obj: UUID) => Promise<TimeLineEventInfo>,
  onEventClick?: (startTime: number, endTime: number, obj: UUID) => void,
  onTimeChange?: (time: number, changedByUser: boolean) => void,
  onTrackSelected?: (index: number) => void,
  onSelectionChange?: (startTime?: number, endTime?: number) => void
};

const TimeLine = (props: TimeLineProps) => {
  const timeLineNodeRef = useRef<HTMLDivElement>(null);
  const timeLineRef = useRef<VisTimeLine | null>(null);

  const setEventHandlers = useCallback(() => {
    const timeLine = timeLineRef.current;
    if (!timeLine) {
      return;
    }

    timeLine.onGetData = (line: string, beginTime: number, endTime: number) => {
      props.onGetData(beginTime, endTime, (data) => {
        timeLine.setData(line, data);
      });
    };

    if (props.onGetEventInfo) {
      timeLine.onGetEventInfo = props.onGetEventInfo;
    }

    if (props.onEventClick) {
      timeLine.onEventClick = props.onEventClick;
    }

    if (props.onTimeChange) {
      timeLine.onTimeChange = props.onTimeChange;
    }

    if (props.onTrackSelected) {
      timeLine.onTrackSelected = props.onTrackSelected;
    }

    // if (props.onSelectionChange) {
    //   timeLine.onSelectionChange = (line: string, startTime?: number, endTime?: number) =>
    //     props.onSelectionChange && props.onSelectionChange(startTime, endTime);
    // }
  }, [props]);

  useEffect(() => {
    if (!timeLineNodeRef.current) {
      return;
    }

    // init
    timeLineRef.current = new VisTimeLine({
      node: timeLineNodeRef.current,
      type: props.type,
      options: props.options,
      isDebug: !!props.isDebug,
      hideSelection: props.hideSelection,
      showTime: props.showTime
    });

    const timeLine = timeLineRef.current;
    timeLine.init().then(() => {
      timeLine.setParameters({
        selection: {fixed: false},
        scale: {granularity: Granularity.min}
      });

      setEventHandlers();

      timeLineRef.current && props.onLoad(timeLineRef.current);
    });
  }, []);

  useEffect(useCallback(() => {
    setEventHandlers();
  }, [props]), [props]);

  return (
    <div ref={timeLineNodeRef} className={props.className}/>
  );
};

export default TimeLine;
