import {bufferToBase64, Logger} from "@solid/libs";
import type {MSEMediaPlayer} from "./msemediaplayer";

type ImagePlayerParameters = {
  logger?: Logger | Console
};

type ImagePlayerInitParameters = {
  node: HTMLElement,
  mediaplayer: MSEMediaPlayer,
  frame: (time: number) => void
};

type Media = {
  time: number,
  data: Uint8Array
};

export class ImagePlayer {
  _logger: Logger | Console = console;
  _node: HTMLElement | null = null;

  _mediaQueue: Media[] = [];

 _currentMedia: Media | null = null;
 _lastRenderedTime = 0;

  _img: HTMLImageElement = document.createElement("img");

  _forceRedraw = false;

  _lastTimestampList: number[] = [];

  _intervalID: number = 0;
  _interval = 33.3; // 30 fps (1000 / 30 = 33.3)

  mediaplayer: MSEMediaPlayer | null = null;
  frame: (time: number) => void = () => {};

  _skipChunks = false;

  _inFrameLoop = false;
  _diff = 0;

  constructor(parameters: ImagePlayerParameters = {}) {
    if (typeof parameters.logger !== "undefined") {
      this._logger = parameters.logger;
    }
  }

  init(parameters: ImagePlayerInitParameters): Promise<void> {
    this._node = parameters.node;
    this.mediaplayer = parameters.mediaplayer;
    this.frame = parameters.frame ?? (() => {});

    this.initPlayer();

    return new Promise((resolve, reject) => {
      resolve();
    });
  }

  initPlayer() {
  }

  addMedia(media: Media) {
    // add timestamps
    for (let i = 0; i < this._lastTimestampList.length; i++) {
      // this._videoFrame.addTime(this._lastTimestampList[i]);
    }

    // add media chunk
    this._mediaQueue.push(media);

    this._skipChunks = false;
  }

  addTime(timestampList: number[], isKey: boolean = false) {
    if (this._skipChunks) {
      // this._logger.log("skip chunk");
    }
    this._lastTimestampList = timestampList.slice();
    this._skipChunks = true;
  }

  _frameLoop() {
    if (this._inFrameLoop) {
      return;
    }
    this._inFrameLoop = true;

    let isSkip = true;
    while (isSkip && this._mediaQueue.length > 0) {
      let media = this._mediaQueue.shift();
      if (!media) {
        continue;
      }

      if (this._diff === 0) {
        this._diff = Date.now() - media.time;
        this._logger.log("new diff", this._diff);
      }
      let diff = Math.abs((Date.now() - media.time) - this._diff);
      isSkip = diff > 2000;
      let isNotify = diff > 1000;
      if (isNotify) {
        this._logger.log(`the player behind for ${diff}ms`);
      }

      // check this logic
      /*
      if (isSkip) {
        this._logger.log("isSkip", diff, (Date.now() - media.time));
        continue;
      }
      */

      media.time && this.frame(media.time);

      this._currentMedia = media;
    }

    this._inFrameLoop = false;
  }

  muteAudio(muted: boolean): void {
    // imageplayer has no sound
  }

  getAudio(): boolean {
    return false;
  }

  draw(imgData: ArrayBuffer) {
    if (!this._node) {
      this._logger.warn("this._node is undefined");
      return;
    }

    let b64imgData = bufferToBase64(imgData); // Binary to ASCII, where it probably stands for
    // let img = new Image();
    // img.src = "data:image/jpeg;base64," + b64imgData;
    const imageNode = this._node.querySelector("img");
    if (imageNode) {
      imageNode.src = "data:image/jpeg;base64," + b64imgData;
    }
  }

  async play() {
    this._logger.log("play");

    this._diff = 0;

    this._intervalID = window.setInterval(() => this._frameLoop(), this._interval);
  }

  pause() {
    this._logger.log("pause");

    clearInterval(this._intervalID);
  }

  stop() {
    this._logger.log("stop");

    clearInterval(this._intervalID);

    this._mediaQueue = [];
  }

  setStartTime(startTime: number) {
  }

  render(source: CanvasImageSource, target: CanvasDrawImage): HTMLImageElement | null {
    if (!this._currentMedia) { return null; }

    if (!this._forceRedraw && this._lastRenderedTime === this._currentMedia.time) { return this._img; }
    this._forceRedraw = false;
    this._lastRenderedTime = this._currentMedia.time;

    this._drawImageBlob(target, this._currentMedia.data);
    // this._drawImageBase64(context, this._currentMedia.data);

    return this._img;
  }

  redraw() {
    this._forceRedraw = true;
  }

  _drawImageBlob(context: CanvasDrawImage, data: ArrayBuffer) {
    const blob = new Blob([data], {'type': 'image/jpeg'});
    this._img.src = URL.createObjectURL(blob);
    this._img.onload = () => {
      context.drawImage(this._img, 0, 0);
    };
  };

  _drawImageBase64(context: CanvasDrawImage, data: ArrayBuffer) {
    let b64imgData = bufferToBase64(data);
    this._img.src = "data:image/jpeg;base64," + b64imgData;
    this._img.onload = () => {
      context.drawImage(this._img, 0, 0);
    };
  }

  destroy() {
  }

  setStatFrameCall(isCall: boolean = false) {
  }

  getStat() {
    return undefined;
  }
}
