import { __, API, isElectron, Log, Logger } from "@solid/libs";
import { MSEMediaPlayer } from "@solid/player/msemediaplayer";
import { YUVPlayer, YUVPlayerInitParameters } from "@libs/player-native";
import { TimeLineBlockWidth, TimeLineError, VisTimeLine } from "@solid/timeline/vistimeline";
import { AODStatus, AODWait } from "@solid/libs/avatar/aod";
import { getCoverage, isChunkExist } from "@solid/libs/avatar/coverage";
import {
  CHUNK_SIZE,
  Granularity,
  MSEMediaPlayerConnection,
  MSEMediaPlayerError,
  MSEMediaPlayerInitParameters,
  MSEMediaPlayerMetadataType,
  MSEMediaPlayerStream,
  StreamParams,
  TimeLineData,
  UUID
} from "@solid/types";
import moment from "moment-timezone";
import rome, { RomeDatePicker } from "@bevacqua/rome";
import { DigitalZoom } from "@solid/joystick/digitalzoom";
import { CameraCellPTZ, PtzConfig } from "@solid/joystick/ptzOnPlayer";
import m from "mithril";
import { PlayerMode } from "@solid/types/player_widget";

export type StreamGeoPosition = [number, number, number?];

export type CameraCellParameters = {
  isShowHeader?: boolean,
  isTimeLineVisible?: boolean,
  isControlsHoverable?: boolean,
  isPlayOnStart?: boolean,
  isPTZVisible?: boolean,
  isDigitalZoomVisible?: boolean,
  isFullScreenControlVisible?: boolean,
  metadataType?: MSEMediaPlayerMetadataType,
  connectionType?: MSEMediaPlayerConnection,
  isShowPauseButton?: boolean,
  disableLocalTransport?: boolean,
  idleTimeout: number,
  limitedSpeedList: boolean,
  muted?: boolean,
  jitterBufferLength?: number,
  onChangeMode?: (mode: PlayerMode, time?: number) => void,
  onAddDownloadJob?: (start: number) => void,
  onArchiveLengthChange?: (hours: number) => void,
  onGeoPosition?: (geo: StreamGeoPosition) => void
}

class CameraCellModel {
  player!: MSEMediaPlayer | YUVPlayer;
  obj!: UUID;
  name: string = "";

  isArchive = false;
  isLowResolution = false;
  isPlay = false;

  isSelected = false;
  isShowCameraCellPTZControl = false;
  isExpandable = false;
  isExpanded = false;
  speedUp = 10;
  speedList = [-30, -10, -5, -2, -1, 0.125, 0.25, 0.5, 1, 2, 5, 10, 30];
  speed = 1;
  jumpStep = 30; // sec
  fromTime?: number;
  directStream?: boolean;
  headerMouseDown: (e: MouseEvent) => void = () => {}
  oncontract: () => void = () => {}

  external: CameraCellParameters = {
    isShowHeader: true,
    isTimeLineVisible: true,
    isControlsHoverable: true,
    isPlayOnStart: true,
    isPTZVisible: true,
    isDigitalZoomVisible: true,
    isFullScreenControlVisible: true,
    metadataType: MSEMediaPlayerMetadataType.TRIGGERED,
    connectionType: MSEMediaPlayerConnection.CLOUD,
    isShowPauseButton: false,
    disableLocalTransport: false,
    idleTimeout: 0,
    limitedSpeedList: false,
    muted: true,
    onChangeMode: (mode: PlayerMode, time?: number) => {},
    onGeoPosition: (geo: StreamGeoPosition) => {}
  }

  select = {
    start: null,
    end: null
  };

  timeline!: VisTimeLine;
  aodWait: AODWait = new AODWait();

  digitalZoom = new DigitalZoom();

  isPTZ = false;
  isClickOnScreen = false;

  isSimple = false;

  datapicker: RomeDatePicker | null = null;
  coverageList: TimeLineData[] = [];

  cachedJitterBufferLength = 0;

  constructor() {

  }

  turnJitterBufferLength() {
    if (isElectron() && this.player instanceof YUVPlayer) {
      this.isShowCameraCellPTZControl && this.player.setJitterBufferLen(0);
      !this.isShowCameraCellPTZControl && this.player.setJitterBufferLen(this.cachedJitterBufferLength);
    }
  }

  hidePtzControls() {
    if (this.isShowCameraCellPTZControl) {
      if (isElectron() && this.player instanceof YUVPlayer) {
        this.player.setJitterBufferLen(this.cachedJitterBufferLength);
      }
      this.isShowCameraCellPTZControl = false;
      this.external.isTimeLineVisible = true;
    }
  }

  async getAOD(deviceId: UUID, start: number, end: number) {
    try {
      await this.aodWait.add(deviceId, start, end, "1", false);
      this.timeline.updateData();
      this.onaoddone();
    }
    catch (e) {
      console.error("aod submit error:", e);
    }
  }

  exportIntervalExists(start: number, end: number) {
    for (let chunk = start; chunk < end; chunk += CHUNK_SIZE) {
      if (!isChunkExist(chunk, this.coverageList, MSEMediaPlayerStream.FULL)) {
        return false;
      }
    }
    return true;
  }

  setTime(time: number, changedByUser: boolean = false, isPlay: boolean = false) {
    if (changedByUser) {
      // cameraCellModel.pause(true, false);

      this.isArchive = true;

      if (changedByUser) {
        this.timeline.setTime(time, true, changedByUser);
      }

      this.player.setTime(this.obj, time, isPlay);

      this.player.printErrorMessage("");
      this.player.hideProgressIndicator();
    }
  }

  setSpeed(speed: number, isPlay: boolean = false) {
    if (!this.player) {
      console.error("this.player is not defined");
      return;
    }

    this.speed = speed;
    this.player.setSpeed(speed);

    if (isPlay && !this.isPlay) {
      // this.player.pause(this.player instanceof MSEMediaPlayer);
      this.player.play();
    }
  }

  async setNextSpeed(isPlay: boolean = false) {
    if (this.player instanceof YUVPlayer) {
      const speed = this.player.getSpeed();
      let speedIndex = this.speedList.indexOf(speed);
      if (speedIndex + 1 <= this.speedList.length - 1) {
        speedIndex++;
      }

      const newSpeed = this.speedList[speedIndex];
      this.speed = newSpeed;
      this.player.setSpeed(newSpeed);
    }
  }

  async setPrevSpeed(isPlay: boolean = false) {
    if (this.player instanceof YUVPlayer) {
      const speed = this.player.getSpeed();
      let speedIndex = this.speedList.indexOf(speed);
      if (speedIndex - 1 >= 0) {
        speedIndex--;
      }

      const newSpeed = this.speedList[speedIndex];
      this.speed = newSpeed;
      this.player.setSpeed(newSpeed);
    }
  }

  async play(obj?: UUID, fromTime?: number, toTime?: number, doNotStart: boolean = false): Promise<boolean> {
    // stop player before going to Live from Archive
    if (this.isArchive && obj && !fromTime) {
      this.player.stop();
    }
    return this.player.play(obj, fromTime, toTime, doNotStart);
  }

  pause(byUser: boolean = false, closeTransport: boolean = true) {
    if (!this.isPlay && typeof byUser !== "undefined") {
      this.onpause(byUser);
    }

    this.player.pause(false);

    this.player.printErrorMessage("");
    this.player.hideProgressIndicator();
  }

  playerStop() {
    if (this.player) {
      this.player.stop();
    }
  }

  setPreferredConnectionType(type: MSEMediaPlayerConnection) {
    this.player.setPreferredConnectionType(type, this.isPlay);
  }

  onload(): void {}
  onclose(): void {}
  onselect(): void {}
  // onstatuschange() {},
  onrecord(): void {}
  onplay(): void {}
  onstop(): void {}
  onpause(byUser: boolean): void {}
  onframe(timestamp: number): void {}
  onaoddone(): void {}
  onadddownloadjob(start: number): void {}
  onarchivelengthchange(hours: number): void {}
}

type PlayerCellCallbacks = Pick<Partial<CameraCellModel>, "onload" | "onclose" | "onrecord" | "onplay" | "onstop" | "onpause" | "onframe" | "onaoddone" | "onadddownloadjob" | "onarchivelengthchange"> & {oninit?: (model: CameraCellModel) => void};
export class PlayerCell {
  cameraCellModel = new CameraCellModel();
  node: HTMLElement | null = null;

  constructor() {
  }

  unselect() {
    this.cameraCellModel.isSelected = false;
    this.cameraCellModel.isShowCameraCellPTZControl = false;
    m.redraw();
  }

  init(node: HTMLElement, obj: UUID, header: string, time: number | undefined, parameters: CameraCellParameters, {onload, onclose, onrecord, onplay, onstop, onpause, onframe, onaoddone, onadddownloadjob, onarchivelengthchange}: PlayerCellCallbacks) {
    for (const parameter in parameters) {
      this.cameraCellModel.external[parameter] = parameters[parameter];
    }

    if (onload) {
      this.cameraCellModel.onload = onload;
    }
    if (onclose) {
      this.cameraCellModel.onclose = onclose;
    }
    if (onrecord) {
      this.cameraCellModel.onrecord = onrecord;
    }
    if (onplay) {
      this.cameraCellModel.onplay = onplay;
    }
    if (onstop) {
      this.cameraCellModel.onstop = onstop;
    }
    if (onpause) {
      this.cameraCellModel.onpause = onpause;
    }
    if (onframe) {
      this.cameraCellModel.onframe = onframe;
    }
    if (onaoddone) {
      this.cameraCellModel.onaoddone = onaoddone;
    }
    if (onadddownloadjob) {
      this.cameraCellModel.onadddownloadjob = onadddownloadjob;
    }
    if (onarchivelengthchange) {
      this.cameraCellModel.onarchivelengthchange = onarchivelengthchange;
    }

    if (parameters.limitedSpeedList) {
      this.cameraCellModel.speedList = [-5, -2, -1, 0.125, 0.25, 0.5, 1, 2, 5];
    }

    this.node = node;

    const api = new API();
    const {joystick, CameraCellPTZControl} = CameraCellPTZ();
    const cameraCellModel = this.cameraCellModel;

    type CameraCellHeaderAttributes = {
      obj: UUID,
      title: string
    }
    type CameraCellHeaderState = {}
    const CameraCellHeader: m.Component<CameraCellHeaderAttributes, CameraCellHeaderState> = {
      view: (vnode) => {
        return [
          m(".left", {
            'onmousedown': (e: MouseEvent) => {
              // if not LEFT mouse button
              if (e.button !== 0) {
                return;
              }

              e.preventDefault();

              if (cameraCellModel.isExpandable && !cameraCellModel.isExpanded) {
                cameraCellModel.isExpandable && cameraCellModel.headerMouseDown(e);
                // @ts-ignore
                let triggerMouseEvent = (node: HTMLElement, eventType: string) => {
                  let clickEvent = document.createEvent('MouseEvents');
                  clickEvent.initEvent(eventType, true, true);
                  node.dispatchEvent(clickEvent);
                };
              }
              return false;
            },
            'ondragstart': (e: DragEvent) => {
              e.preventDefault();
              return false;
            }
          }, [
            m("span.player_stream_name", {class: cameraCellModel.isArchive ? "archive" : "live"}, cameraCellModel.isArchive ? __("Archive") : __("Live")),
            m("span.cell_dev_name", vnode.attrs.title)
          ]),
          m(".right", [
            !cameraCellModel.isSimple && m("i.fas.fa-times", {
              onclick: (e: MouseEvent) => {
                cameraCellModel.onclose();
                e.stopPropagation();
              }
            })
          ])
        ];
      }
    };

    type CameraCellContentAttributes = {
      obj: UUID,
      toTime?: number,
      fromTime?: number
    }
    type CameraCellContentState = {
      timestamp: number,
      isShowRecord: boolean,
      isShowDatePicker: boolean,
      isFullscreen: boolean,
      startArchiveTime: number,
      ptzConfig: PtzConfig,
      attributes: {[attribute: string]: string},
      isFoldableMenuOpen: boolean,
      leftMouseButtonDown: boolean,
      leftMouseButtonIsHeldDown: boolean,
      isPlayBeforeLeftMouseButtonIsHeldDown: boolean,
      speedBeforeLeftMouseButtonIsHeldDown: number,
      minPlayerControlsWidth: number,
      isMoreMinWidth: boolean
    }
    const CameraCellContent: m.Component<CameraCellContentAttributes, CameraCellContentState> = {
      oninit: ({state}) => {
        state.timestamp = 0;
        state.isShowRecord = false;
        state.isShowDatePicker = false;
        state.isFullscreen = false;
        state.startArchiveTime = 0;
        state.attributes = {};
        state.isFoldableMenuOpen = false;

        state.leftMouseButtonDown = false;
        state.leftMouseButtonIsHeldDown = false;

        state.minPlayerControlsWidth = 500;
        state.isMoreMinWidth = false;
      },
      oncreate: (vnode) => {
        vnode.dom.addEventListener("fullscreenchange", () => {
          vnode.state.isFullscreen = !!document.fullscreenElement;
          m.redraw();
        });

        const timelineNode: HTMLDivElement | null = vnode.dom.querySelector(".timeline");
        cameraCellModel.timeline = new VisTimeLine({
          node: timelineNode,
          isDebug: false
        });
        const timeline = cameraCellModel.timeline;

        const playerNode: HTMLElement | null = vnode.dom.querySelector(".player");
        if (!playerNode) {
          console.error("player node is undefined");
          return;
        }

        if (playerNode.clientWidth < vnode.state.minPlayerControlsWidth) {
          vnode.state.isMoreMinWidth = true;
        }

        let connectionType = cameraCellModel.directStream || cameraCellModel.external.disableLocalTransport ? MSEMediaPlayerConnection.CLOUD : undefined;
        if (typeof parameters.connectionType !== "undefined") {
          connectionType = parameters.connectionType;
        }
        if (isElectron() && process.env?.PLAYER !== "mse") {
          cameraCellModel.player = new YUVPlayer({
            node: playerNode,
            logger: new Logger(),
            connectionType: connectionType
          });
        } else {
          cameraCellModel.player = new MSEMediaPlayer({
            node: playerNode,
            logger: new Logger(),
            connectionType: connectionType
          });
        }

        const player = cameraCellModel.player;

        const initParameters: MSEMediaPlayerInitParameters | YUVPlayerInitParameters = {
          obj: vnode.attrs.obj,
          time: vnode.attrs.fromTime,
          streamid: cameraCellModel.isLowResolution ? MSEMediaPlayerStream.DIGEST : MSEMediaPlayerStream.FULL,
          metadataType: cameraCellModel.external.metadataType,
          connectionType: cameraCellModel.external.connectionType,
          showPauseButton: cameraCellModel.external.isShowPauseButton,
          idleTimeout: cameraCellModel.external.idleTimeout,
          muted: cameraCellModel.external.muted,
          jitter_buffer_len: cameraCellModel.external.jitterBufferLength,
          disableLocalTransport: cameraCellModel.external.disableLocalTransport
        };

        Promise.all([
          timeline.init(),
          player.init(initParameters)
        ]).then(() => {
          timeline.setParameters({
            selection: {
              fixed: false
            },
            scale: {granularity: Granularity.min}
          });

          if (cameraCellModel.fromTime) {
            timeline.setTime(cameraCellModel.fromTime);
          }

          timeline.onGetData = (line, beginTime, endTime, granularity) => {
            let maxMinutes = timeline.getWidth() / TimeLineBlockWidth.MINUTE;
            beginTime -= maxMinutes * 60 * 1000;
            endTime += maxMinutes * 60 * 1000;

            /*
            let start = new Date();
            start.setTime(beginTime);
            let end = new Date();
            end.setTime(endTime);
            console.log("get data " + start.getHours() + ":" + start.getMinutes() + ":" + start.getSeconds() + " " + end.getHours() + ":" + end.getMinutes() + ":" + end.getSeconds());
            */

            timeline.createThumb(vnode.attrs.obj);

            const withLow = player instanceof MSEMediaPlayer;
            cameraCellModel.external.isTimeLineVisible && getCoverage(vnode.attrs.obj, beginTime, endTime, undefined, Number(MSEMediaPlayerStream.FULL), process.env.DATA_DIR, withLow)
              .then(({
                list,
                avatar_coverage,
                ttl_hours
              }) => {
                ttl_hours && onarchivelengthchange && onarchivelengthchange(ttl_hours[vnode.attrs.obj])
                cameraCellModel.coverageList = list;
                timeline.setData(line, [
                  {
                    name: "",
                    rows: list
                  }
                ]);
              });
          };

          timeline.onTimeChange = (time, changedByUser) => {
            if (changedByUser) {
              vnode.state.timestamp = time;

              cameraCellModel.aodWait.clear();

              m.redraw();
            }

            cameraCellModel.setTime(time, changedByUser);
          };

          // timeline.onSelectionChange = (line, beginTime, endTime) => {
          //   cameraCellModel.pause();
          //   vnode.state.isShowRecord = !!beginTime && !!endTime;
          //   m.redraw();
          // };

          parameters.metadataType && this.cameraCellModel.player?.setMetadataDrawType(parameters.metadataType);

          player.subscribe("resize", () => {
            const cellWidth = this.node?.clientWidth;
            if (cellWidth && cellWidth < vnode.state.minPlayerControlsWidth) {
              vnode.state.isMoreMinWidth = true;
              m.redraw();
            }
            if (cellWidth && cellWidth > vnode.state.minPlayerControlsWidth && vnode.state.isMoreMinWidth) {
              vnode.state.isMoreMinWidth = false;
              m.redraw();
            }
            if (vnode.state.isFullscreen && vnode.state.isMoreMinWidth) {
              vnode.state.isMoreMinWidth = false;
              m.redraw();
            }

            cameraCellModel.digitalZoom.changeSize();
          });

          player.subscribe("initializing", () => {
            cameraCellModel.isPlay = true;
          });

          player.subscribe("play", (stream: StreamParams) => {
            cameraCellModel.isPlay = true;

            if (cameraCellModel.external.onChangeMode) {
              const time = player.getTime();
              cameraCellModel.isArchive
                ? cameraCellModel.external.onChangeMode(PlayerMode.Archive, time)
                : cameraCellModel.external.onChangeMode(PlayerMode.Live)
            }

            const drawNode = player.getDrawNode();
            const metaDataNode = player.getMetaDataNode();
            if (drawNode && metaDataNode) {
              cameraCellModel.digitalZoom.init({
                node: drawNode,
                metaDataNode,
                width: stream.width,
                height: stream.height
              });
              cameraCellModel.digitalZoom.showMiniMap();
            }

            cameraCellModel.aodWait.clear();

            cameraCellModel.onplay();
          });

          player.subscribe("pause", (byUser: boolean) => {
            cameraCellModel.isPlay = false;

            cameraCellModel.digitalZoom.hideMiniMap();

            cameraCellModel.onpause(byUser);

            cameraCellModel.external.onChangeMode && cameraCellModel.external.onChangeMode(PlayerMode.Stopped, cameraCellModel.isArchive ? player.getTime() : undefined);
          });
          player.subscribe("click", (width: number, height: number, x: number, y: number) => {
            if (cameraCellModel.isClickOnScreen) {
              joystick.gotoXY(Math.round(x), Math.round(y), width, height);
            }
          });

          player.subscribe("stop", async (code?: number, message?: string) => {
            cameraCellModel.isPlay = false;
            cameraCellModel.external.onChangeMode && cameraCellModel.external.onChangeMode(PlayerMode.Stopped, cameraCellModel.isArchive ? player.getTime(): undefined);
            const aodRequest = (start: number, streamid: string) => {
              player.printErrorMessage(__("Request archive from avatar to cloud"));
              player.showProgressIndicator();

              cameraCellModel.aodWait.add(vnode.attrs.obj, start, undefined, streamid)
                .then((request) => {
                  timeline.updateData();
                  onaoddone && onaoddone();

                  if (request.status === AODStatus.COMPLETED) {
                    timeline.setTime(start, true);
                    cameraCellModel.play(vnode.attrs.obj, start);
                  } else
                  if (![AODStatus.CANCELED, AODStatus.FAILED].includes(request.status)) {
                    cameraCellModel.aodWait.once(`request.${request.requestid}`, ({status}: {status: string}) => {
                      timeline.updateData();
                      onaoddone && onaoddone();

                      if (!cameraCellModel.isPlay) {
                        return;
                      }

                      if (!status) {
                        player.printErrorMessage(__("Cannot process AOD request"));
                        player.hideProgressIndicator();
                        return;
                      }

                      timeline.setTime(start, true);
                      cameraCellModel.play(vnode.attrs.obj, start);
                    });
                  }
                })
                .catch((e) => {
                  Log.error(e.message);
                  player.printErrorMessage(e.message);
                });
            };

            let streamid = cameraCellModel.isLowResolution ? MSEMediaPlayerStream.DIGEST : MSEMediaPlayerStream.FULL;

            const connectionType = player.getConnectionType();

            // const diffLarge = 30 * 1000; // ms

            if (cameraCellModel.isArchive
                &&
                !cameraCellModel.directStream
                &&
                connectionType === MSEMediaPlayerConnection.CLOUD
                &&
                (code === MSEMediaPlayerError.VIDEO_UNAVAILABLE
                 || code === MSEMediaPlayerError.END_OF_ARCHIVE)
            ) {
              if (code === MSEMediaPlayerError.END_OF_ARCHIVE) {
                // check next chunk after current for scenario when information about chunk in archive is incorrect
                let chunkExist = isChunkExist(timeline.getTime() + CHUNK_SIZE, cameraCellModel.coverageList, streamid);
                if (chunkExist) {
                  player.printErrorMessage("[" + code + "] " + __("Can not retrieve content from the video archive"));
                  return;
                }
              }

              let start = timeline.getTime();

              aodRequest(start, streamid);
            }

            cameraCellModel.onstop();
          });

          player.subscribe("frame", (timestamp: number, width: number, height: number) => {
            if (!timestamp) return;
            if (!cameraCellModel.isPlay) return;

            timeline.setTime(timestamp, false)
              .then(() => {
                // TODO: check for local archive
                const connectionType = player.getConnectionType();
                if (cameraCellModel.isArchive
                    &&
                    !cameraCellModel.directStream
                    &&
                    connectionType === MSEMediaPlayerConnection.CLOUD
                ) {
                  // check when current interval is ending and create request for archive if it is not requested yet
                  const checkDiff = CHUNK_SIZE + 10 * 1000; // ms
                  const timestampToCheck = timestamp + checkDiff * Math.sign(cameraCellModel.speed);
                  let streamid = cameraCellModel.isLowResolution ? MSEMediaPlayerStream.DIGEST : MSEMediaPlayerStream.FULL;
                  if (timestamp + CHUNK_SIZE < Date.now()
                      && !isChunkExist(timestampToCheck, cameraCellModel.coverageList, streamid)
                  ) {
                    // search and request one chunk only
                    if (!cameraCellModel.aodWait.searchRequest(vnode.attrs.obj, timestampToCheck, 1, streamid)) {
                      cameraCellModel.aodWait.add(vnode.attrs.obj, timestampToCheck, 60, streamid)
                        .then((request) => {
                          timeline.updateData();
                          onaoddone && onaoddone();

                          cameraCellModel.aodWait.once(`request.${request.requestid}`, () => {
                            timeline.updateData();
                            onaoddone && onaoddone();
                          });
                        });
                    }
                  }
                }

                vnode.state.timestamp = timestamp;
                m.redraw();
              })
              .catch((e) => {
                if (e instanceof TimeLineError) {
                  cameraCellModel.aodWait.clear();

                  // maybe we set time not in user selection (if defined)
                  player.pause((cameraCellModel.player instanceof MSEMediaPlayer) || !cameraCellModel.isArchive);
                } else {
                  throw e;
                }
              });

            cameraCellModel.onframe(timestamp);
          });

          player.subscribe("live", () => {
            cameraCellModel.isArchive = false;

            m.redraw();
          });

          player.subscribe("archive", () => {
            cameraCellModel.isArchive = true;

            m.redraw();
          });

          player.subscribe("geoPosition", (geo: StreamGeoPosition) => {
            cameraCellModel.external.onGeoPosition?.(geo);
          })

          if (vnode.attrs.fromTime && vnode.attrs.toTime) {
            let then = () => {
              // if (vnode.attrs.fromTime && vnode.attrs.toTime) {
              //   timeline.setSelection(vnode.attrs.fromTime, vnode.attrs.toTime, true);
              // }
              if (cameraCellModel.isSimple) {
                cameraCellModel.play(vnode.attrs.obj, vnode.attrs.fromTime, undefined, !cameraCellModel.external.isPlayOnStart);
              }
            };

            timeline.setTime(vnode.attrs.fromTime, true)
              .then(() => {
                then();
              })
              .catch(() => {
                then();
              });
          } else
          if (vnode.attrs.fromTime) {
            timeline.setTime(vnode.attrs.fromTime, true)
              .then(() => {
                cameraCellModel.play(vnode.attrs.obj, vnode.attrs.fromTime, undefined, !cameraCellModel.external.isPlayOnStart);
              });
          } else {
            timeline.setTime(Date.now(), true);
            cameraCellModel.play(vnode.attrs.obj, undefined, undefined, !cameraCellModel.external.isPlayOnStart);
          }

          // TODO: rethink this onload logic
          player.once("play", () => cameraCellModel.onload());

          if (isElectron() && player instanceof YUVPlayer) {
            this.cameraCellModel.cachedJitterBufferLength = player.getJitterBufferLen();
          }
        });

        api.getAttributes({obj: vnode.attrs.obj})
          .then((response) => {
            vnode.state.attributes = response.list;
            if (response.list.PTZ_CONFIG) {
              vnode.state.ptzConfig = JSON.parse(response.list.PTZ_CONFIG);
            }

            let STORAGE_BUCKETS = JSON.parse(response.list.STORAGE_BUCKETS);
            if (STORAGE_BUCKETS[0].ttlHrs) {
              vnode.state.startArchiveTime = Date.now() - STORAGE_BUCKETS[0].ttlHrs * 60 * 60 * 1000;
            }

            const dtp = vnode.dom.querySelector('.dtp');
            if (!dtp) {
              console.error("datePicker node is undefined");
              return;
            }

            cameraCellModel.datapicker = rome(dtp, {
              min: moment(vnode.state.startArchiveTime).format('YYYY-MM-DD HH:mm'),
              max: moment(Date.now()).format('YYYY-MM-DD HH:mm')
            });
            cameraCellModel.datapicker.hide();

            let rd_time = vnode.dom.querySelectorAll('.rd-time-option');
            for (let i = 0; i < rd_time.length; i++) {
              rd_time[i].addEventListener('click', () => {
                if (!cameraCellModel.datapicker) {
                  console.error("datePicker node is undefined");
                  return;
                }

                cameraCellModel.datapicker.on('data', (timeValue: string) => {
                  if (!vnode.state.isShowDatePicker) {
                    let selectTime = moment(timeValue).format('YYYY-MM-DD HH:mm:ss');
                    let time = moment(selectTime).valueOf();
                    cameraCellModel.pause(true, true);
                    timeline.setTime(time, true)
                      .then(() => cameraCellModel.play(vnode.attrs.obj, time));
                  }
                });

                vnode.state.isShowDatePicker = false;
                cameraCellModel.datapicker.hide();
              });
            }
          });
      },
      view: (vnode) => {
        let timestampString = "";
        if (vnode.state.timestamp) {
          let date = new Date();
          date.setTime(vnode.state.timestamp);
          timestampString = `${String(date.getHours()).padStart(2, "0")}:${String(date.getMinutes()).padStart(2, "0")}:${String(date.getSeconds()).padStart(2, "0")}${Math.abs(cameraCellModel.speed) < 1 ? `.${String(date.getUTCMilliseconds()).padStart(3, "0")}` : ""}`;
        }

        let metadataDrawType = cameraCellModel.player?.getMetadataDrawType();
        // @ts-ignore
        let metadataDrawTypeList = {
          "triggered": {
            selector: "i.fas.fa-exclamation-circle",
            title: __("Alert")
          },
          "all": {
            selector: "i.fas.fa-info-circle",
            title: __("All")
          },
          "none": {
            selector: "i.fas.fa-ban",
            title: __("None")
          }
        };

        let connectionType = cameraCellModel.player?.getConnectionType();
        let preferredConnectionType = cameraCellModel.player?.getPreferredConnectionType();
        let connectionTypeSelector = "i.fas" + (preferredConnectionType === MSEMediaPlayerConnection.AUTO ? ".connection-auto" : "");
        if (preferredConnectionType === MSEMediaPlayerConnection.AUTO) {
          connectionTypeSelector += (connectionType === MSEMediaPlayerConnection.CLOUD ? ".fa-cloud" : ".fa-home");
        } else {
          connectionTypeSelector += (preferredConnectionType === MSEMediaPlayerConnection.CLOUD ? ".fa-cloud" : ".fa-home");
        }

        return m(".content", [
          m(".player"),
          cameraCellModel.external.isPTZVisible && cameraCellModel.isShowCameraCellPTZControl && vnode.state.ptzConfig.optical && m(CameraCellPTZControl, {
            obj: vnode.attrs.obj,
            ptz_config: vnode.state.ptzConfig,
            onHide: () => {
              cameraCellModel.isShowCameraCellPTZControl = false;
              cameraCellModel.external.isTimeLineVisible = true;
              cameraCellModel.turnJitterBufferLength();
            },
            // setClickOnScreen: (enable: boolean) => {
            //   cameraCellModel.isClickOnScreen = enable;
            // }
          }),
          m(".controls", {
            class: (cameraCellModel.external.isShowPauseButton ? "controls-hidden" : "")
                   + " "
                   + (cameraCellModel.external.isTimeLineVisible ? "" : "controls-without-timeline")
                   + " "
                   + (cameraCellModel.external.isControlsHoverable ? "controls-hoverable" : "controls-not-hoverable")
                   + " "
                   + (cameraCellModel.isArchive ? "player-archive" : ""),
            style: {top: +(cameraCellModel.isArchive ? "calc(100% - 76px)" : "100%")}
          }, [
            m(".player-controls-background"),
            m(".legend", [
              m("i.far.fa-question-circle"),
              m("div", [
                m(".legend-item", [
                  m(".legend-img", [
                    m("div", {
                      style: {
                        background: "#4CAF50"
                      }
                    })
                  ]),
                  m(".legend-descr", __("Full-res video"))
                ]),
                m(".legend-item", [
                  m(".legend-img", [
                    m(".legend-event-marker"),
                  ]),
                  m(".legend-descr", __("Event marker"))
                ]),
                m(".legend-item", [
                  m(".legend-img", [
                    m("div", {
                      style: {
                        backgroundImage: "repeating-linear-gradient(135deg, #FFEF07 3px, #FFEF07 5px, #00000000 6px, #00000000 9.5px)"
                      }
                    })
                  ]),
                  m(".legend-descr", __("Partial video"))
                ]),
                m(".legend-item", [
                  m(".legend-img", [
                    m("div", {
                      style: {
                        background: "#EF6C00"
                      }
                    })
                  ]),
                  m(".legend-descr", __("Active download"))
                ])
              ])
            ]),
            m(".timeline"),
            m(".player-controls", [
              m(".left", [
                /*!!vnode.state.timestamp && m("i.fas.fa-step-backward", {
                  onclick: () => {
                    CameraCellModel.player.stepBackward();
                  }
                }),*/
                !vnode.state.isMoreMinWidth && cameraCellModel.isArchive && (cameraCellModel.player instanceof YUVPlayer) && m("span.backward", {
                  title: __('Backward'),
                  onmousedown: () => {
                    vnode.state.leftMouseButtonDown = true;
                    setTimeout(() => {
                      if (!vnode.state.leftMouseButtonDown) {
                        return;
                      }

                      vnode.state.isPlayBeforeLeftMouseButtonIsHeldDown = cameraCellModel.isPlay;
                      vnode.state.speedBeforeLeftMouseButtonIsHeldDown = cameraCellModel.speed;
                      vnode.state.leftMouseButtonIsHeldDown = true;
                      cameraCellModel.setSpeed(-cameraCellModel.speedUp, true);

                      m.redraw();
                    }, 1000);
                  },
                  onmouseup: () => {
                    vnode.state.leftMouseButtonDown = false;
                    if (vnode.state.leftMouseButtonIsHeldDown) {
                      if (!vnode.state.isPlayBeforeLeftMouseButtonIsHeldDown) {
                        cameraCellModel.pause();
                      }
                      cameraCellModel.setSpeed(vnode.state.speedBeforeLeftMouseButtonIsHeldDown, vnode.state.isPlayBeforeLeftMouseButtonIsHeldDown);
                    }
                  },
                  onclick: () => {
                    if (vnode.state.leftMouseButtonIsHeldDown) {
                      vnode.state.leftMouseButtonIsHeldDown = false;
                      return;
                    }

                    const newSpeed = -Math.ceil(Math.abs(cameraCellModel.speed));
                    if (newSpeed != cameraCellModel.speed) {
                      cameraCellModel.setSpeed(newSpeed, true);
                    } else
                    if (!cameraCellModel.isPlay) {
                      cameraCellModel.play();
                    }
                  }
                }, [
                  !vnode.state.isMoreMinWidth && m("i.fas.fa-backward")
                ]),
                !vnode.state.isMoreMinWidth && cameraCellModel.isArchive && (cameraCellModel.player instanceof YUVPlayer) && m("span.backward-step", {
                  title: __("Jump {{number}}sec Backward", {number: cameraCellModel.jumpStep}),
                  onclick: () => {
                    if (!cameraCellModel.isPlay) {
                      return;
                    }

                    cameraCellModel.player.setTime(vnode.attrs.obj, cameraCellModel.timeline.getTime() - cameraCellModel.jumpStep * 1000, true);
                  }
                }, [
                  !vnode.state.isMoreMinWidth && m("i.fas.fa-arrow-left")
                ]),
                !cameraCellModel.isPlay && m("i.fas.fa-play", {
                  title: __("Play"),
                  onclick: () => {
                    cameraCellModel.play();
                  }
                }),
                cameraCellModel.isPlay && m("i.fas.fa-pause", {
                  title: __("Pause"),
                  onclick: () => {
                    cameraCellModel.pause(true, (cameraCellModel.player instanceof MSEMediaPlayer));
                  }
                }),
                !vnode.state.isMoreMinWidth && cameraCellModel.isArchive && (cameraCellModel.player instanceof YUVPlayer) && m("select.speed", {
                  value: cameraCellModel.player.getSpeed(),
                  onchange: (e: UIEvent) => {
                    if (!(cameraCellModel.player instanceof YUVPlayer)) {
                      return;
                    }
                    const speed = Number((e.target as HTMLSelectElement).value);
                    cameraCellModel.setSpeed(speed, cameraCellModel.isPlay);
                  }
                }, cameraCellModel.speedList.map((speed) => m("option", {value: speed}, `${Math.abs(speed) > 1 ? Math.ceil(speed) : speed}x`))),
                !vnode.state.isMoreMinWidth && cameraCellModel.isArchive && (cameraCellModel.player instanceof YUVPlayer) && m("span.forward-step", {
                  title: __("Jump {{number}}sec Forward", {number: cameraCellModel.jumpStep}),
                  onclick: () => {
                    if (!cameraCellModel.isPlay) {
                      return;
                    }

                    cameraCellModel.player.setTime(vnode.attrs.obj, cameraCellModel.timeline.getTime() + cameraCellModel.jumpStep * 1000, true);
                  }
                }, [
                  m("i.fas.fa-arrow-right")
                ]),
                !vnode.state.isMoreMinWidth && cameraCellModel.isArchive && (cameraCellModel.player instanceof YUVPlayer) && m("span.forward", {
                  title: __('Forward'),
                  onmousedown: () => {
                    vnode.state.leftMouseButtonDown = true;
                    setTimeout(() => {
                      if (!vnode.state.leftMouseButtonDown) {
                        return;
                      }

                      vnode.state.isPlayBeforeLeftMouseButtonIsHeldDown = cameraCellModel.isPlay;
                      vnode.state.speedBeforeLeftMouseButtonIsHeldDown = cameraCellModel.speed;
                      vnode.state.leftMouseButtonIsHeldDown = true;
                      cameraCellModel.setSpeed(cameraCellModel.speedUp, true);

                      m.redraw();
                    }, 1000);
                  },
                  onmouseup: () => {
                    vnode.state.leftMouseButtonDown = false;
                    if (vnode.state.leftMouseButtonIsHeldDown) {
                      if (!vnode.state.isPlayBeforeLeftMouseButtonIsHeldDown) {
                        cameraCellModel.pause();
                      }
                      cameraCellModel.setSpeed(vnode.state.speedBeforeLeftMouseButtonIsHeldDown, vnode.state.isPlayBeforeLeftMouseButtonIsHeldDown);
                    }
                  },
                  onclick: () => {
                    if (vnode.state.leftMouseButtonIsHeldDown) {
                      vnode.state.leftMouseButtonIsHeldDown = false;
                      return;
                    }

                    const newSpeed = Math.abs(cameraCellModel.speed);
                    if (newSpeed != cameraCellModel.speed) {
                      cameraCellModel.setSpeed(newSpeed, true);
                    } else
                    if (!cameraCellModel.isPlay) {
                      cameraCellModel.play();
                    }
                  }
                }, [
                  !vnode.state.isMoreMinWidth && m("i.fas.fa-forward")
                ]),
                /*!!vnode.state.timestamp && m("i.fas.fa-step-forward", {
                  onclick: () => {
                    CameraCellModel.player.stepForward();
                  }
                }),*/
                m("span.timestamp", timestampString),
              ]),
              m(".dtp"),
              m(".right", [
                m("button.player-controls-button.live-badge", {
                  disabled: !cameraCellModel.isArchive,
                  onclick: () => {
                    cameraCellModel.setSpeed(1);
                    cameraCellModel.play(vnode.attrs.obj);
                  }
                }, __("Live")),
                !vnode.state.isMoreMinWidth && m(connectionTypeSelector, {
                  title: __('Connection type'),
                  onclick: () => {
                    if (!cameraCellModel.directStream && !cameraCellModel.external.disableLocalTransport) {
                      if (preferredConnectionType === MSEMediaPlayerConnection.AUTO) {
                        preferredConnectionType = MSEMediaPlayerConnection.CLOUD;
                      } else
                      if (preferredConnectionType === MSEMediaPlayerConnection.CLOUD) {
                        preferredConnectionType = MSEMediaPlayerConnection.LOCAL;
                      } else
                      if (preferredConnectionType === MSEMediaPlayerConnection.LOCAL) {
                        preferredConnectionType = MSEMediaPlayerConnection.AUTO;
                      }
                      cameraCellModel.player.setPreferredConnectionType(preferredConnectionType, cameraCellModel.isPlay);
                    }
                  }
                }),
                // !CameraCellModel.isSimple && m("div.record", {
                // 	title: __("Record"),
                // 	width: 22,
                // 	height: 22,
                // 	onclick: () => {
                // 		let selection = CameraCellModel.timeline.getSelection();
                // 		if (selection === null) {
                // 			let time = Math.round(CameraCellModel.timeline.getTime());
                // 			selection = {};
                // 			selection.start = time - 2 * 60 * 1000; // 2 minutes from current
                // 			selection.end = time;
                // 			if (!CameraCellModel.isArchive) {
                // 				let minute = 60 * 1000;
                // 				selection.start -= minute;
                // 				selection.end -= minute;
                // 			}
                // 		}
                // 		selection && CameraCellModel.onrecord(selection.start, selection.end);
                // 	}
                // }, "[R]"),
                !!vnode.state.timestamp && !vnode.state.isMoreMinWidth && m("i.fas.fa-camera", {
                  title: __('Save Snapshot'),
                  onclick: () => {
                    let canvas: HTMLCanvasElement | null = null;
                    let isVideo = true;
                    let element = cameraCellModel.player.getDrawNode();
                    if (!element) {
                      console.error('no element for snapshot');
                      return;
                    }
                    let currDate = new Date(Math.round(cameraCellModel.timeline.getTime()));
                    let fileName = cameraCellModel.name + ' ';
                    fileName += currDate.getFullYear() + '-';
                    fileName += String(currDate.getMonth() + 1).padStart(2, "0") + '-';
                    fileName += String(currDate.getDate()).padStart(2, "0") + '--';
                    fileName += String(currDate.getHours()).padStart(2, "0") + '-';
                    fileName += String(currDate.getMinutes()).padStart(2, "0") + '-';
                    fileName += String(currDate.getSeconds()).padStart(2, "0");
                    fileName += '.png';
                    let elementType = element.nodeName.toLowerCase();

                    if (elementType == 'video') {
                      const video = (element as HTMLVideoElement);
                      canvas = document.createElement("canvas");
                      canvas.width = video.videoWidth;
                      canvas.height = video.videoHeight;
                      canvas.getContext('2d')?.drawImage(element, 0, 0, video.videoWidth, video.videoHeight);
                    } else {
                      canvas = element as HTMLCanvasElement;
                      isVideo = false;
                    }
                    let tag = document.createElement('a');
                    tag.href = canvas.toDataURL();
                    tag.download = fileName;
                    document.body.appendChild(tag);
                    tag.click();
                    document.body.removeChild(tag);
                    if (isVideo) {
                      canvas.remove();
                    }
                  }
                }),
                !vnode.state.isShowDatePicker && !vnode.state.isMoreMinWidth && m("i.far.fa-calendar", {
                    title: __('Calendar'),
                    onclick: () => {
                      vnode.state.isShowDatePicker = true;
                      cameraCellModel.datapicker?.show();
                      vnode.state.isFoldableMenuOpen = false;
                      cameraCellModel.hidePtzControls();
                    }
                  }
                ),
                vnode.state.isShowDatePicker && !vnode.state.isMoreMinWidth && m(".control-dtp", [
                  m(".close-dtp", m("i.fas.fa-times", {
                    onclick: () => {
                      vnode.state.isShowDatePicker = false;
                      cameraCellModel.datapicker?.hide();
                    }
                  }))
                ]),
                !cameraCellModel.isArchive && cameraCellModel.isPTZ && !vnode.state.isMoreMinWidth && m("i.fas.fa-gamepad", {
                  title: __("PTZ control"),
                  onclick: (e: MouseEvent) => {
                    !cameraCellModel.isShowCameraCellPTZControl && cameraCellModel.digitalZoom.enabled && cameraCellModel.digitalZoom.switchZoom();
                    cameraCellModel.isShowCameraCellPTZControl = !cameraCellModel.isShowCameraCellPTZControl;
                    cameraCellModel.external.isTimeLineVisible = !cameraCellModel.external.isTimeLineVisible;
                    cameraCellModel.turnJitterBufferLength();
                    vnode.state.isFoldableMenuOpen = false;
                    cameraCellModel.datapicker?.hide();
                    vnode.state.isShowDatePicker = false;
                    e.stopPropagation();
                  }
                }),
                /*!vnode.state.isFullscreen && !CameraCellModel.isExpanded && CameraCellModel.isExpandable && m("i.fas.fa-expand-arrows-alt", {
                  title: __('Expand'),
                  onclick: () => {
                    CameraCellModel.onexpand();
                  }
                }),*/
                !vnode.state.isFullscreen && !vnode.state.isMoreMinWidth && cameraCellModel.isExpanded && m("i.fas.fa-compress-arrows-alt", {
                  title: __('Collapse'),
                  onclick: () => {
                    cameraCellModel.oncontract();
                  }
                }),
                !vnode.state.isMoreMinWidth && m("i.fas.fa-cog", {
                  title: __('Menu'),
                  onclick: (e: MouseEvent) => {
                    cameraCellModel.datapicker?.hide();
                    vnode.state.isShowDatePicker = false;
                    vnode.state.isFoldableMenuOpen = !vnode.state.isFoldableMenuOpen;
                    cameraCellModel.hidePtzControls();
                    e.stopPropagation();
                  }
                }),
                cameraCellModel.external.isFullScreenControlVisible && !vnode.state.isFullscreen && m("i.fas.fa-expand", {
                  title: vnode.state.isMoreMinWidth ? __('Enter Fullscreen to show all controls') : __('Enter Fullscreen'),
                  onclick: () => {
                    vnode.dom.requestFullscreen();
                  }
                }),
                cameraCellModel.external.isFullScreenControlVisible && vnode.state.isFullscreen && m("i.fas.fa-compress", {
                  title: __('Exit Fullscreen'),
                  onclick: () => {
                    document.exitFullscreen();
                  }
                }),
                vnode.state.isFoldableMenuOpen && m("span.foldable_menu", [
                  !!cameraCellModel.digitalZoom.controls && !cameraCellModel.digitalZoom.disable && m(".foldable_menu_item", {
                      onclick: (e: MouseEvent) => {
                        e.stopPropagation();

                        cameraCellModel.digitalZoom.switchZoom();
                        vnode.state.isFoldableMenuOpen = false;
                      }
                    },
                    [
                      m("i.icon"),
                      m(".menu_item_icon", [
                        m("i.fas.fa-search-plus")
                      ]),
                      m(".description", cameraCellModel.digitalZoom.enabled ? __("Disable Digital Zoom") : __("Enable Digital Zoom"))
                    ]
                  ),
                  m(".foldable_menu_item", [
                    m("i.icon.fas.fa-caret-left"),
                    m(".menu_item_icon", [
                      m("i.fas.fa-lightbulb")
                    ]),
                    m(".description", __("Analytics Markup")),
                    m("span.sub_menu", [
                      m(".sub_menu_item", {
                        onclick: (e: MouseEvent) => {
                          if (metadataDrawType !== MSEMediaPlayerMetadataType.TRIGGERED) {
                            metadataDrawType = MSEMediaPlayerMetadataType.TRIGGERED;
                            cameraCellModel.player.setMetadataDrawType(metadataDrawType);
                            e.stopPropagation();
                          }
                          vnode.state.isFoldableMenuOpen = false;
                        }
                      }, [
                        m(".sub_menu_item_label", [
                          m("i.icon.fas.fa-exclamation-circle"),
                          __('Alert')
                        ]),
                        metadataDrawType === MSEMediaPlayerMetadataType.TRIGGERED ? m('i.fas.fa-check') : null
                      ]),
                      m(".sub_menu_item", {
                        onclick: (e: MouseEvent) => {
                          if (metadataDrawType !== MSEMediaPlayerMetadataType.ALL) {
                            metadataDrawType = MSEMediaPlayerMetadataType.ALL;
                            cameraCellModel.player.setMetadataDrawType(metadataDrawType);
                            e.stopPropagation();
                          }
                          vnode.state.isFoldableMenuOpen = false;
                        }
                      }, [
                        m(".sub_menu_item_label", [
                          m("i.icon.fas.fa-circle"),
                          __('All')
                        ]),
                        metadataDrawType === MSEMediaPlayerMetadataType.ALL ? m('i.fas.fa-check') : null
                      ]),
                      m(".sub_menu_item", {
                        onclick: (e: MouseEvent) => {
                          if (metadataDrawType !== MSEMediaPlayerMetadataType.NONE) {
                            metadataDrawType = MSEMediaPlayerMetadataType.NONE;
                            cameraCellModel.player.setMetadataDrawType(metadataDrawType);
                            e.stopPropagation();
                          }
                          vnode.state.isFoldableMenuOpen = false;
                        }
                      }, [
                        m(".sub_menu_item_label", [
                          m("i.icon.fas.fa-ban"),
                          __('None')
                        ]),
                        metadataDrawType === MSEMediaPlayerMetadataType.NONE ? m('i.fas.fa-check') : null
                      ])
                    ])
                  ]),
                  vnode.state.isFoldableMenuOpen && cameraCellModel.isArchive && !cameraCellModel.directStream && m(".foldable_menu_item", [
                    m("i.icon"),
                    m(".menu_item_icon", [
                      m("i.fas.fa-cloud-download-alt")
                    ]),
                    m(".description", {
                      onclick: (e: MouseEvent) => {
                        e.stopPropagation();
                        let start = cameraCellModel.timeline.getTime();
                        onadddownloadjob && onadddownloadjob(start);
                        vnode.state.isFoldableMenuOpen = false;
                      }
                    },
                    __("Add download job")),
                  ]),
                ])
              ])
            ])
          ])
        ]);
      },
      onremove: (vnode) => {
        this.cameraCellModel.coverageList = [];
        this.cameraCellModel.timeline.destroy();
        this.cameraCellModel.player.destroy();
        this.cameraCellModel.aodWait.clear();
        this.cameraCellModel.datapicker?.destroy();
        this.cameraCellModel.datapicker = null;
      }
    };

    type CameraCellAttributes = {
      obj: UUID,
      name: string,
      fromTime?: number,
      toTime?: number
    }
    type CameraCellState = {}
    const CameraCell: m.Component<CameraCellAttributes, CameraCellState> = {
      oncreate: ({attrs}) => {
        api.getAttributes({obj: attrs.obj})
          .then((response) => {
            if (response.list.POSITIONCTL !== "none") {
              cameraCellModel.isPTZ = true;
              joystick.setObj(attrs.obj);
            }
          });
      },
      view: (vnode) => {
        return m(".cell", [
          cameraCellModel.external.isShowHeader && m("header", {
            class: cameraCellModel.isSelected ? "selected" : "",
            onclick: () => {
              cameraCellModel.isSelected = true;
              cameraCellModel.onselect();
            }
          }, m(CameraCellHeader, {
            title: vnode.attrs.name,
            obj: vnode.attrs.obj
          })),
          m(CameraCellContent, {
            obj: vnode.attrs.obj,
            fromTime: vnode.attrs.fromTime,
            toTime: vnode.attrs.toTime
          })
        ]);
      }
    };

    type CellAttributes = {
      obj: UUID,
      name: string,
      time?: number,
      fromTime?: number,
      toTime?: number,
      stream?: "digest" | "full",
      isSimple: boolean
    }
    type CellState = {}
    const Cell: m.Component<CellAttributes, CellState> = {
      oninit: ({attrs}) => {
        cameraCellModel.obj = attrs.obj;
        cameraCellModel.name = attrs.name.replace("&#47;", "/");
        cameraCellModel.isArchive = !!attrs.fromTime || !!attrs.fromTime && !!attrs.toTime;
        cameraCellModel.isLowResolution = attrs.stream === "digest";

        cameraCellModel.fromTime = attrs.time;

        cameraCellModel.isSimple = attrs.isSimple;

        cameraCellModel.isSelected = true;
        cameraCellModel.onselect();
      },
      view: ({attrs}) => {
        return m(CameraCell, {
          obj: cameraCellModel.obj,
          name: cameraCellModel.name,
          fromTime: attrs.fromTime,
          toTime: attrs.toTime
        });
      }
    };

    // @ts-ignore
    const EmptyCell: m.Component = {
      view: () => {
        return m("div.empty", "Empty");
      }
    };

    /*
    m.route(document.querySelector(".cell_wrapper"), "/", {
      "/": EmptyCell,
      "/simple/:obj/:fromTime/:name/:stream": {render: ({attrs}) => m(Cell, Object.assign({isSimple: true}, attrs))},
      "/simple/:obj/:fromTime/:toTime/:name:/:stream": {render: ({attrs}) => m(Cell, Object.assign({isSimple: true}, attrs))},
      "/:obj/:name/:stream": Cell,
      "/:obj/:fromTime/:name/:stream": Cell,
      "/:obj/:fromTime/:toTime/:name:/:stream": Cell
    });
    */

    m.mount(this.node, {
      view: () => {
        return m(Cell, {
          obj: obj,
          name: header,
          fromTime: time,
          isSimple: false
        });
      }
    });
  };

  destroy() {
    if (!this.node) {
      return;
    }

    m.mount(this.node, null);
  };

  getPlayer() {
    return this.cameraCellModel.player;
  };

  getModel() {
    return this.cameraCellModel;
  }
}
