/**
 * @file: vanilla.idle.js
 * @title: Vanilla Idle.
 * @description: A simple VanillaJS port of this jQuery plugin.
 * @author @hugohil
 * @version 1.2.7
 * @license https://opensource.org/licenses/MIT
 * @link https://github.com/henriqueboaventura/jquery.idle
 */

declare global {
   var idle: typeof Idle
}

type IdleOptions = {
  onActive: () => void,
  onShow: () => void,
  startAtIdle: boolean,
  idle: number,
  keepTracking: boolean,
  recurIdleCall: boolean,
  onIdle: () => void,
  onHide: () => void,
  events: string[]
};

let Idle = function (options: Partial<IdleOptions>) {
  const defaults: IdleOptions = {
    idle: 60000, // idle time in ms
    events: ["mousemove", "keydown", "mousedown", "touchstart"], // events that will trigger the idle resetter
    onIdle: function () {}, // callback function to be executed after idle time
    onActive: function () {}, // callback function to be executed after back form idleness
    onHide: function () {}, // callback function to be executed when window become hidden
    onShow: function () {}, // callback function to be executed when window become visible
    keepTracking: true, // set it to false of you want to track only once
    startAtIdle: false, // set it to true if you want to start in the idle state
    recurIdleCall: false
  };
  const settings = extend<IdleOptions>({}, defaults, options);
  let idle = settings.startAtIdle;
  let visible = !settings.startAtIdle;
  const visibilityEvents = ["visibilitychange", "webkitvisibilitychange", "mozvisibilitychange", "msvisibilitychange"];
  let lastId: NodeJS.Timeout | null  = null;

  const resetTimeout = (id: NodeJS.Timeout | null, settings: IdleOptions): NodeJS.Timeout | null => {
    if (idle) {
      idle = false;
      settings.onActive();
    }
    id && clearTimeout(id);
    if (settings.keepTracking) {
      return timeout(settings);
    }
    return null;
  };

  const timeout = function timeout(settings: IdleOptions): NodeJS.Timeout {
    const timer = (settings.recurIdleCall) ? setInterval : setTimeout;
    let id;
    id = timer(function () {
      idle = true;
      settings.onIdle();
    }, settings.idle);
    return id;
  };

  return {
    start: function () {
      lastId = timeout(settings);
      bulkAddEventListener(window, settings.events, (event: any) => {
        lastId = resetTimeout(lastId, settings);
      });
      if (settings.onShow || settings.onHide) {
        bulkAddEventListener(document, visibilityEvents, (event: any) => {
          // @ts-ignore
          if (document.hidden || document.webkitHidden || document.mozHidden || document.msHidden) {
            if (visible) {
              visible = false;
              settings.onHide();
            }
          } else {
            if (!visible) {
              visible = true;
              settings.onShow();
            }
          }
        });
      }
    },
    stop: function () {
      bulkRemoveEventListener(window, settings.events);
      settings.keepTracking = false;
      resetTimeout(lastId, settings);
    }
  };
};
globalThis.idle = Idle;

const bulkAddEventListener = (object: EventTarget, events: any[], callback: Function) => {
  events.forEach((event) => {
    object.addEventListener(event, (event: any) => {
      callback(event);
    });
  });
};

const bulkRemoveEventListener = (object: EventTarget, events: any[]) => {
  events.forEach((event) => {
    object.removeEventListener(event, null);
  });
};

// Thanks to http://youmightnotneedjquery.com/
const extend = function extend<T>(out: Partial<T>, defaults: T, ...args: Partial<T>[]): T {
  for (let i = 1; i < arguments.length; i++) {
    if (!arguments[i]) {
      continue;
    }
    for (const key in arguments[i]) {
      if (arguments[i].hasOwnProperty(key)) {
        out[key] = arguments[i][key];
      }
    }
  }
  // @ts-ignore
  return out;
};

export default Idle;
