import {__} from "@solid/libs/i18n";
import {API} from "@solid/libs/api";
import Joystick, { Ptz_Mode } from "./joystick";
import {VirtualJoystick} from "./virtualjoystick";
import rangeSlider from "rangeslider-pure";
import m from "mithril";
import {Log} from "@solid/libs/log";
import {UUID} from "@solid/types";
import { Ptz_StepFocus, Ptz_StepIris, Ptz_StepMove, Ptz_StepZoom } from "@solid/graphql";

const api = new API();

interface RangeSliderInput extends HTMLInputElement {
  rangeSlider: {
    update: ({ value }: { value: number } ) => void
  }
}

type CameraCellPTZModelParameters = {
  isShowCameraCellPTZ: boolean,
  isShowPtzMenuOptical: boolean,
  isShowPtzMenuSettings: boolean,
  isShowPtzMenuPresets: boolean,
  isShowPtzMenuOpticalEdit: boolean,
  isShowPtzPresetAdd: boolean,
  isShowPtzPresetEdit: boolean,
  isShowPtzPresetsCountEdit: boolean,
  isShowPtzSettingsAdd: boolean,
  isShowPtzSettingsEdit: boolean,
  isShowPtzSettingsCommandEdit: boolean,
  isShowPtzSettingsCommandAdd: boolean,
  isClickOnScreen: boolean
}

export type PtzConfig = {
  ver: number,
  kind: string,
  optical: { [name: string]: boolean },
  optical_settings: { [name: string]: number },
  preset_settings: { [name: string]: number },
  presets: any[],
  settings: PtzConfigSettings[],
}

type PtzConfigSettings = {
  name: string,
  param: {
    name: string,
    cmds: PtzConfigSettingsCmds[]
  }
}

type PtzConfigSettingsCmds = {
  name: string,
  cmd: {
    mode: "settings",
    cmd: "gain" | "autolevel" | "digitalzoom" | "shutter" | "polarity" | "autofocus" | "autoiris" | "whitebalance" | "thermal-hot",
    param: "auto" | "reset" | "up" | "down" | "direct" | "on" | "off" | "manual" | "white" | "black" | "indoor" | "outdoor" | "onepushwb" | "atw" | "onepushtrigger"
  }
}

export function CameraCellPTZ() {
  const joystick = new Joystick();

  const CameraCellPTZModel: CameraCellPTZModelParameters = {
    isShowCameraCellPTZ: true,
    isShowPtzMenuOptical: true,
    isShowPtzMenuSettings: false,
    isShowPtzMenuPresets: false,
    isShowPtzMenuOpticalEdit: false,
    isShowPtzPresetAdd: false,
    isShowPtzPresetEdit: false,
    isShowPtzPresetsCountEdit: false,
    isShowPtzSettingsAdd: false,
    isShowPtzSettingsEdit: false,
    isShowPtzSettingsCommandEdit: false,
    isShowPtzSettingsCommandAdd: false,
    isClickOnScreen: false
  };

  /*
  const ptzConfigDefault: PtzConfig = {
    "ver": 1,
    "kind": "PTZ controls config",
    "optical": {
      "click-to-reposition": true,
      "speed-vector": true,
      "pt-steps": false,
      "zoom-steps": true,
      "zoom-speed": true,
      "iris-steps": true,
      "iris-speed": true,
      "focus-steps": true,
      "focus-speed": true
    },
    "optical_settings": {
      "zoom": 0,
      "focus": 0,
      "iris": 0
    },
    "settings": [],
    "preset_settings": {
      "allow_preset": 1,
      "count": 10
    },
    "presets": []
  };
  */

  const default_settings: PtzConfigSettings[] = [
    {
      "name": "gain",
      "param": {
        "name": "Gain",
        "cmds": [
          {
            "name": "Auto",
            "cmd": {
              "mode": "settings",
              "cmd": "gain",
              "param": "auto"
            }
          },
          {
            "name": "Reset",
            "cmd": {
              "mode": "settings",
              "cmd": "gain",
              "param": "reset"
            }
          },
          {
            "name": "Up",
            "cmd": {
              "mode": "settings",
              "cmd": "gain",
              "param": "up"
            }
          },
          {
            "name": "Down",
            "cmd": {
              "mode": "settings",
              "cmd": "gain",
              "param": "down"
            }
          },
          {
            "name": "Direct",
            "cmd": {
              "mode": "settings",
              "cmd": "gain",
              "param": "direct"
            }
          }
        ]
      }
    },
    {
      "name": "autolevel",
      "param": {
        "name": "Auto level",
        "cmds": [
          {
            "name": "On",
            "cmd": {
              "mode": "settings",
              "cmd": "autolevel",
              "param": "on"
            }
          },
          {
            "name": "Off",
            "cmd": {
              "mode": "settings",
              "cmd": "autolevel",
              "param": "off"
            }
          }
        ]
      }
    },
    {
      "name": "digitalzoom",
      "param": {
        "name": "Digitalzoom",
        "cmds": [
          {
            "name": "On",
            "cmd": {
              "mode": "settings",
              "cmd": "digitalzoom",
              "param": "on"
            }
          },
          {
            "name": "Off",
            "cmd": {
              "mode": "settings",
              "cmd": "digitalzoom",
              "param": "off"
            }
          }
        ]
      }
    },
    {
      "name": "shutter",
      "param": {
        "name": "Shutter",
        "cmds": [
          {
            "name": "Auto",
            "cmd": {
              "mode": "settings",
              "cmd": "shutter",
              "param": "auto"
            }
          },
          {
            "name": "Manual",
            "cmd": {
              "mode": "settings",
              "cmd": "shutter",
              "param": "manual"
            }
          }
        ]
      }
    },
    {
      "name": "polarity",
      "param": {
        "name": "Polarity",
        "cmds": [
          {
            "name": "White",
            "cmd": {
              "mode": "settings",
              "cmd": "polarity",
              "param": "white"
            }
          },
          {
            "name": "Black",
            "cmd": {
              "mode": "settings",
              "cmd": "polarity",
              "param": "black"
            }
          }
        ]
      }
    },
    {
      "name": "autofocus",
      "param": {
        "name": "Autofocus",
        "cmds": [
          {
            "name": "On",
            "cmd": {
              "mode": "settings",
              "cmd": "autofocus",
              "param": "on"
            }
          },
          {
            "name": "Off",
            "cmd": {
              "mode": "settings",
              "cmd": "autofocus",
              "param": "off"
            }
          },
          {
            "name": "Auto",
            "cmd": {
              "mode": "settings",
              "cmd": "autofocus",
              "param": "auto"
            }
          }
        ]
      }
    },
    {
      "name": "autoiris",
      "param": {
        "name": "Autoiris",
        "cmds": [
          {
            "name": "On",
            "cmd": {
              "mode": "settings",
              "cmd": "autoiris",
              "param": "on"
            }
          },
          {
            "name": "Off",
            "cmd": {
              "mode": "settings",
              "cmd": "autoiris",
              "param": "off"
            }
          },
          {
            "name": "Auto",
            "cmd": {
              "mode": "settings",
              "cmd": "autoiris",
              "param": "auto"
            }
          }
        ]
      }
    },
    {
      "name": "whitebalance",
      "param": {
        "name": "White balance",
        "cmds": [
          {
            "name": "Auto",
            "cmd": {
              "mode": "settings",
              "cmd": "whitebalance",
              "param": "auto"
            }
          },
          {
            "name": "Indoor",
            "cmd": {
              "mode": "settings",
              "cmd": "whitebalance",
              "param": "indoor"
            }
          },
          {
            "name": "Outdoor",
            "cmd": {
              "mode": "settings",
              "cmd": "whitebalance",
              "param": "outdoor"
            }
          },
          {
            "name": "Off",
            "cmd": {
              "mode": "settings",
              "cmd": "whitebalance",
              "param": "off"
            }
          },
          {
            "name": "Onepushwb",
            "cmd": {
              "mode": "settings",
              "cmd": "whitebalance",
              "param": "onepushwb"
            }
          },
          {
            "name": "Atw",
            "cmd": {
              "mode": "settings",
              "cmd": "whitebalance",
              "param": "atw"
            }
          },
          {
            "name": "Manual",
            "cmd": {
              "mode": "settings",
              "cmd": "whitebalance",
              "param": "manual"
            }
          },
          {
            "name": "Onepushtrigger",
            "cmd": {
              "mode": "settings",
              "cmd": "whitebalance",
              "param": "onepushtrigger"
            }
          }
        ]
      }
    }
  ];

  const searchInJson = (obj: any, key: string, val: string) => {
    var objects: any[] = [];
    for (var i in obj) {
      if (!obj.hasOwnProperty(i)) continue;
      if (typeof obj[i] == 'object') {
        objects = objects.concat(searchInJson(obj[i], key, val));
      } else
      if (i == key && obj[i] == val || i == key && val == '') {
        objects.push(obj);
      } else
      if (obj[i] == val && key == '') {
        if (objects.lastIndexOf(obj) == -1) {
          objects.push(obj);
        }
      }
    }
    return objects;
  };

  type CameraCellPTZOpticalAttributes = {
    obj: UUID,
    ptz_config: PtzConfig
  }
  type CameraCellPTZOpticalState = {
    getJoystickPostionLoopId: number | undefined
  }
  const CameraCellPTZOptical: m.Component<CameraCellPTZOpticalAttributes, CameraCellPTZOpticalState> = {
    oninit: ({state}) => {
      state.getJoystickPostionLoopId = undefined;
    },
    oncreate: function (vnode) {
      const radius = 400; // px
      const joystickNode = vnode.dom.querySelector(".ptz-pad-box");
      if (!joystickNode) {
        throw new Error(".ptz-pad-box is undefined")
      }

      let virtualJoystick = new VirtualJoystick({
        container: joystickNode,
        mouseSupport: true,
        stationaryBase: true,
        baseX: joystickNode.scrollWidth / 2,
        baseY: joystickNode.scrollHeight / 2,
        limitStickTravel: true,
        stickRadius: Math.min(joystickNode.scrollWidth, joystickNode.scrollHeight) - 50,
        strokeStyle: '#fff'
      });
      virtualJoystick.init();

      const interval = 100; // ms
      let axis = {
        x: 0,
        y: 0
      };
      vnode.state.getJoystickPostionLoopId = window.setInterval(() => {
        if (CameraCellPTZModel.isClickOnScreen) return;

        const X = virtualJoystick.deltaX();
        const Y = virtualJoystick.deltaY();
        const relX = X / radius * 100;
        const relY = Y / radius * 100;

        if (relX !== axis.x) {
          axis.x = relX;
          joystick.onAxis(0, axis.x);
        }
        if (relY !== axis.y) {
          axis.y = relY;
          joystick.onAxis(1, axis.y);
        }
      }, interval);

      let slider = vnode.dom.querySelectorAll<RangeSliderInput>('input[type="range"]');
      if (slider.length > 0) {
        rangeSlider.create(slider, {
          polyfill: true,
          rangeClass: 'rangeSlider',
          disabledClass: 'rangeSlider--disabled',
          fillClass: 'rangeSlider__fill',
          bufferClass: 'rangeSlider__buffer',
          handleClass: 'rangeSlider__handle',
          startEvent: ['mousedown', 'touchstart', 'pointerdown'],
          moveEvent: ['mousemove', 'touchmoveptz-preset-popup', 'pointermove'],
          endEvent: ['mouseup', 'touchend', 'pointerup'],
          vertical: false,
          min: -100,
          max: 100,
          step: 1,
          value: 0,
          buffer: null,
          stick: null,
          borderRadius: 10,

          /*
            Move camera step by step
            Перемещение камеры шаг за шагом
            zoom	in, out
            focus	near, far
            iris	open, close
            gain	up, down
            //dev=1&mode=step&zoom=out

            Move camera smoothly until stop (or any other command) is sent
            Перемещайте камеру плавно до тех пор, пока не будет отправлена остановка (или любая другая команда)
            zoom	in, out, stop
            focus	near, far, stop
            iris	open, close, stop
            gain	up, down, stop
            //dev=3&mode=smooth&zoom=stop

           */
          //

          onSlide: function (value: number, percent: number, position: number) {
            joystick.onAxis(3, value);
          },

          onSlideEnd: function (value: number, percent: number, position: number) {
            slider[0].rangeSlider.update({ value: 0 });
          }
        });
      }
    },
    onremove: function ({state}) {
      clearInterval(state.getJoystickPostionLoopId);
    },
    view: function (vnode) {
      return m(".ptz-pad", [
        m(".ptz-joystick"),
        m(".ptz-pad-box", {
          // onclick: function(event: MouseEvent) {
          //   if (!CameraCellPTZModel.isClickOnScreen) return;
          //   joystick.gotoXY(Math.round(event.offsetX), Math.round(event.offsetY), (event.target as Element)?.clientWidth, (event.target as Element)?.clientHeight);
          // }
        },
        [
          CameraCellPTZModel.isShowPtzMenuOptical && m(".left", {
              onmousedown: (e: MouseEvent) => {
                e.stopPropagation();
              },
              onclick: function (e: MouseEvent) {
                joystick.ptz(Ptz_Mode.STEP, {
                  name: "move",
                  value: Ptz_StepMove.Left
                })
              }
            },
            m("i.fas.fa-angle-left")
          ),
          CameraCellPTZModel.isShowPtzMenuOptical && m(".right", {
              onmousedown: (e: MouseEvent) => {
                e.stopPropagation();
              },
              onclick: function (e: MouseEvent) {
                joystick.ptz(Ptz_Mode.STEP, {
                  name: "move",
                  value: Ptz_StepMove.Right
                })
              }
            },
            m("i.fas.fa-angle-right")
          ),
          CameraCellPTZModel.isShowPtzMenuOptical && m(".up", {
              onmousedown: (e: MouseEvent) => {
                e.stopPropagation();
              },
              onclick: function (e: MouseEvent) {
                joystick.ptz(Ptz_Mode.STEP, {
                  name: "move",
                  value: Ptz_StepMove.Up
                })
              }
            },
            m("i.fas.fa-angle-up")
          ),
          CameraCellPTZModel.isShowPtzMenuOptical && m(".down", {
              onmousedown: (e: MouseEvent) => {
                e.stopPropagation();
              },
              onclick: function (e: MouseEvent) {
                joystick.ptz(Ptz_Mode.STEP, {
                  name: "move",
                  value: Ptz_StepMove.Down
                })
              }
            },
            m("i.fas.fa-angle-down")
          )
          
        ]),

        /*
         PAN 	 Left  	Step left Direct Pan  Step right 	Right
         ZOOM 	 Wide  	Zoom Out Direct Zoom  Zoom In 		Tele
         FOCUS 	 Near  	Focus Near Smooth Focus Focus Far 	Far
         IRIS 	 Close  Iris Close Smooth Iris 	Iris Open 	Open
        * */
        m(".range_box", [
          vnode.attrs.ptz_config.optical["zoom-speed"] && m(".zoom", [
            m("label[for='range-zoom']", __("Zoom")),
            m(".range-item",
              vnode.attrs.ptz_config.optical["zoom-steps"] &&
              m("i.fas.fa-minus", {
                onclick: function () {
                  joystick.ptz(Ptz_Mode.STEP, {
                    name: "zoom",
                    value: Ptz_StepZoom.Out
                  })
                }
              }),
              m("input[data-rangeslider='']", {
                id: "range-zoom",
                name: "range-zoom",
                max: "100",
                min: "-100",
                step: "0.1",
                stick: "10 0.1",
                type: "range",
                // value: 0,
              }),
              vnode.attrs.ptz_config.optical["zoom-steps"] &&
              m("i.fas.fa-plus", {
                onclick: function () {
                  joystick.ptz(Ptz_Mode.STEP, {
                    name: "zoom",
                    value: Ptz_StepZoom.In
                  })
                }
              })
            )
          ]),
          vnode.attrs.ptz_config.optical["focus-speed"] && m(".focus", [
            m("label[for='range-focus']",
              __("Focus")
            ),
            m(".range-item",
              vnode.attrs.ptz_config.optical["focus-steps"] &&
              m("i.fas.fa-minus", {
                onclick: function () {
                  joystick.ptz(Ptz_Mode.STEP, {
                    name: "focus",
                    value: Ptz_StepFocus.Near
                  })
                }
              }),
              /*
              m("input][data-rangeslider='']",{
                id:"range-focus",
                name:"range-focus",
                max:"100",
                min:"-100",
                step:"0.1",
                stick:"10 0.1",
                type:"range",
                value: 0,
              }),
              */
              vnode.attrs.ptz_config.optical["focus-steps"] &&
              m("i.fas.fa-plus", {
                onclick: function () {
                  joystick.ptz(Ptz_Mode.STEP, {
                    name: "focus",
                    value: Ptz_StepFocus.Far
                  })
                }
              })
            )
          ]),
          vnode.attrs.ptz_config.optical["iris-speed"] && m(".iris", [
            m("label[for='range-iris']",
              __("Iris")
            ),
            m(".range-item",
              vnode.attrs.ptz_config.optical["iris-speed"] &&
              m("i.fas.fa-minus", {
                onclick: function () {
                  joystick.ptz(Ptz_Mode.STEP, {
                    name: "iris",
                    value: Ptz_StepIris.Close
                  })
                }
              }),
              /*
              m("input[data-rangeslider='']",{
                id:"range-iris",
                name:"range-iris",
                max:"100",
                min:"-100",
                step:"0.1",
                stick:"10 0.1",
                type:"range",
                value: 0,
              }),
              */
              vnode.attrs.ptz_config.optical["iris-speed"] &&
              m("i.fas.fa-plus", {
                onclick: function () {
                  joystick.ptz(Ptz_Mode.STEP, {
                    name: "iris",
                    value: Ptz_StepIris.Open
                  })
                }
              })
            )
          ])
        ])
      ]);
    }
  };

  type CameraCellPTZOpticalEditAttributes = {
    obj: UUID,
    ptz_config: PtzConfig
  }
  type CameraCellPTZOpticalEditState = {}
  const CameraCellPTZOpticalEdit: m.Component<CameraCellPTZOpticalEditAttributes, CameraCellPTZOpticalEditState> = {
    view: function (vnode) {
      return m(".ptz-menu.ptz-optical-edit", [
        m("h4", [
          __("PTZ Optical Edit"),
          m("i.far.fa-eye-slash", {
            onclick: function () {
              CameraCellPTZModel.isShowPtzMenuOpticalEdit = false;
            }
          })
        ]),
        m("ul", [
          m("li", [
            m("input[name='click-to-reposition'][type='checkbox']", {
              checked: vnode.attrs.ptz_config.optical["click-to-reposition"] ? "checked" : "",
              onchange: function () {
                vnode.attrs.ptz_config.optical["click-to-reposition"]
                  ? vnode.attrs.ptz_config.optical["click-to-reposition"] = false
                  : vnode.attrs.ptz_config.optical["click-to-reposition"] = true;
              }
            }),
            m("label[for='click-to-reposition']", __("Reposition by mouse click on video"))
          ]),
          m("li", [
            m("input[name='speed-vector'][type='checkbox']", {
              checked: vnode.attrs.ptz_config.optical["speed-vector"] ? "checked" : "",
              onchange: function () {
                vnode.attrs.ptz_config.optical["speed-vector"]
                  ? vnode.attrs.ptz_config.optical["speed-vector"] = false
                  : vnode.attrs.ptz_config.optical["speed-vector"] = true;
              }
            }),
            m("label[for='speed-vector']", __("Vector-of-motion control"))
          ]),
          m("li", [
            m("input[name='pt-steps'][type='checkbox']", {
              checked: vnode.attrs.ptz_config.optical["pt-steps"] ? "checked" : "",
              onchange: function () {
                vnode.attrs.ptz_config.optical["pt-steps"]
                  ? vnode.attrs.ptz_config.optical["pt-steps"] = false
                  : vnode.attrs.ptz_config.optical["pt-steps"] = true;
              }
            }),
            m("label[for='pt-steps']", __("Pan-tilt step control"))
          ]),
          m("li", [
            m("input[name='zoom-steps'][type='checkbox']", {
              checked: vnode.attrs.ptz_config.optical["zoom-steps"] ? "checked" : "",
              onchange: function () {
                vnode.attrs.ptz_config.optical["zoom-steps"]
                  ? vnode.attrs.ptz_config.optical["zoom-steps"] = false
                  : vnode.attrs.ptz_config.optical["zoom-steps"] = true;
              }
            }),
            m("label[for='zoom-steps']", __("Zoom step control"))
          ]),
          m("li", [
            m("input[name='zoom-speed'][type='checkbox']", {
              checked: vnode.attrs.ptz_config.optical["zoom-speed"] ? "checked" : "",
              onchange: function () {
                vnode.attrs.ptz_config.optical["zoom-speed"]
                  ? vnode.attrs.ptz_config.optical["zoom-speed"] = false
                  : vnode.attrs.ptz_config.optical["zoom-speed"] = true;
              }
            }),
            m("label[for='zoom-speed']", __("Zoom speed control"))
          ]),
          m("li", [
            m("input[name='focus-steps'][type='checkbox']", {
              checked: vnode.attrs.ptz_config.optical["focus-steps"] ? "checked" : "",
              onchange: function () {
                vnode.attrs.ptz_config.optical["focus-steps"]
                  ? vnode.attrs.ptz_config.optical["focus-steps"] = false
                  : vnode.attrs.ptz_config.optical["focus-steps"] = true;
              }
            }),
            m("label[for='focus-steps']", __("Focus step control"))
          ]),
          m("li", [
            m("input[name='focus-speed'][type='checkbox']", {
              checked: vnode.attrs.ptz_config.optical["focus-speed"] ? "checked" : "",
              onchange: function () {
                vnode.attrs.ptz_config.optical["focus-speed"]
                  ? vnode.attrs.ptz_config.optical["focus-speed"] = false
                  : vnode.attrs.ptz_config.optical["focus-speed"] = true;
              }
            }),
            m("label[for='focus-speed']", __("Focus speed control"))
          ]),
          m("li", [
            m("input[name='iris-steps'][type='checkbox']", {
              checked: vnode.attrs.ptz_config.optical["iris-steps"] ? "checked" : "",
              onchange: function () {
                vnode.attrs.ptz_config.optical["iris-steps"]
                  ? vnode.attrs.ptz_config.optical["iris-steps"] = false
                  : vnode.attrs.ptz_config.optical["iris-steps"] = true;
              }
            }),
            m("label[for='iris-steps']", __("Iris step control"))
          ]),
          m("li", [
            m("input[name='iris-speed'][type='checkbox']", {
              checked: vnode.attrs.ptz_config.optical["iris-speed"] ? "checked" : "",
              onchange: function () {
                vnode.attrs.ptz_config.optical["iris-speed"]
                  ? vnode.attrs.ptz_config.optical["iris-speed"] = false
                  : vnode.attrs.ptz_config.optical["iris-speed"] = true;
              }
            }),
            m("label[for='iris-speed']", __("Iris speed control"))
          ])
        ]),
        m(".menu-btn", [
          m("input[value='Apply'][type='button']", {
            value: __("Apply"),
            type: "button",
            onclick: function () {
              var arr = vnode.dom.querySelectorAll("input[type=checkbox]");
              let ptzConfig = vnode.attrs.ptz_config;
              [].map.call(arr, function (param: HTMLInputElement) {
                ptzConfig.optical["" + param.name + ""] = param.checked;
              });

              api.setAttributes({
                obj: vnode.attrs.obj,
                attributes: {PTZ_CONFIG: JSON.stringify(ptzConfig)}
              }).then((response) => {
                Log.info(__("Optical settings save"));
              });
            }
          }),
          m("input", {
            value: __("Cancel"),
            type: "button",
            onclick: function () {
              CameraCellPTZModel.isShowPtzMenuOpticalEdit = false;
            }
          })
        ])
      ]);
    }
  };

  type CameraCellPTZSettingsAttributes = {
    obj: UUID,
    ptz_config: PtzConfig
  }
  type CameraCellPTZSettingsState = {
    setting: PtzConfigSettings
  }
  const CameraCellPTZSettings: m.Component<CameraCellPTZSettingsAttributes, CameraCellPTZSettingsState> = {
    oninit: ({state}) => {
    },
    view: function (vnode) {
      return [
        m(".ptz-menu.ptz-preset-menu", [
          m("h4", [
            __("PTZ Settings"),
            m("i.far.fa-eye-slash", {
              onclick: function () {
                CameraCellPTZModel.isShowPtzMenuSettings = false;
              }
            })
          ]),
          m("table",
            m("thead", [
              m("tr", [
                m("th", __("Setting")),
                m("th", __("Commands")),
                m("th",
                  m("i.far.fa-plus-square", {
                    onclick: function () {
                      CameraCellPTZModel.isShowPtzSettingsAdd = true;
                    }
                  })
                )
              ]),
            ]),
            m("tbody", [
              default_settings.length > 0
                ? default_settings.map((conf) => {
                  var search_setting = searchInJson(vnode.attrs.ptz_config.settings, 'name', conf.name);
                  return m("tr", [
                    m("td", conf.name),
                    m("td[class='settings-value']",
                      conf.param.cmds.length > 0 ?
                        conf.param.cmds.map((c) => {
                          return m("input", {
                            value: c.name,
                            type: "button",
                            class: (!search_setting.length) ? '' : (search_setting[0].param.cmds[0].cmd.param == c.cmd.param) ? 'active-setting' : '',
                            onclick: function () {
                              var setting_command = {
                                "name": conf.name,
                                "param": {
                                  "name": conf.param.name,
                                  "cmds": [
                                    {
                                      "name": "" + c.name + "",
                                      "cmd": {
                                        "mode": c.cmd.mode,
                                        "cmd": c.cmd.cmd,
                                        "param": c.cmd.param
                                      }
                                    }
                                  ]
                                }
                              };

                              if (!search_setting.length) {
                                vnode.attrs.ptz_config.settings.push(setting_command);
                                joystick.sendCommand("mode=settings&" + conf.name + "=" + c.cmd.param + "");
                              } else
                              if (search_setting[0].param.cmds[0].cmd.param != c.cmd.param) {
                                vnode.attrs.ptz_config.settings.map(function (item, index, arr) {
                                  if (item.name == conf.name) {
                                    arr.splice(index, 1);
                                  }
                                });
                                vnode.attrs.ptz_config.settings.push(setting_command);
                                joystick.sendCommand("mode=settings&" + conf.name + "=" + c.cmd.param + "");
                              }
                            }
                          });
                        }) : "-"
                    ),
                    m("td",
                      m("i.fas.fa-edit", {
                        onclick: function () {
                          vnode.state.setting = conf;
                          CameraCellPTZModel.isShowPtzSettingsEdit = true;
                        }
                      })
                    )
                  ]);
                })
                : m("tr", [m("td[colspan='3']", __("no settings"))])
            ])
          ),
          CameraCellPTZModel.isShowPtzSettingsEdit && m(CameraPTZSettingEdit, {
            obj: vnode.attrs.obj,
            ptz_config: vnode.attrs.ptz_config,
            setting: vnode.state.setting
          }),
          CameraCellPTZModel.isShowPtzSettingsAdd && m(CameraPTZSettingAdd, {
            obj: vnode.attrs.obj,
            ptz_config: vnode.attrs.ptz_config
          }),
          m(".menu-btn", [
            m("input", {
              value: __("Apply"),
              type: "button",
              onclick: function () {
                api.setAttributes({
                  obj: vnode.attrs.obj,
                  attributes: {PTZ_CONFIG: JSON.stringify(vnode.attrs.ptz_config)}
                }).then((response) => {
                  Log.info(__("Setting save"));
                });
              }
            }),
            m("input", {
              value: __("Cancel"),
              type: "button",
              onclick: function () {
                CameraCellPTZModel.isShowPtzMenuSettings = false;
              }
            }),
            m("input", {
              value: __("Copy from"),
              type: "button",
              onclick: function () {
              }
            })
          ])
        ])
      ];
    }
  };

  type CameraPTZSettingAddAttributes = {
    obj: UUID,
    ptz_config: PtzConfig
  }
  type CameraPTZSettingAddState = {
    settings: string
  }
  const CameraPTZSettingAdd: m.Component<CameraPTZSettingAddAttributes, CameraPTZSettingAddState> = {
    oninit: ({state}) => {
      state.settings = "";
    },
    view: function (vnode) {
      return m(".ptz-menu.ptz-preset-popup.ptz-preset-popup-add", [
        m("h4", [
          __("PTZ Setting Add"),
          m("i.far.fa-eye-slash", {
            onclick: function () {
              CameraCellPTZModel.isShowPtzSettingsAdd = false;
            }
          })
        ]),
        m(".menu-btn", [
          m("label[for='option']", __("Setting Name")),
          m("input[name='name_setting'][type='text']"),
        ]),
        m(".menu-btn", [
          m("input", {
            value: __("Apply"),
            type: "button",
            onclick: function () {
              const nameSettingNode: HTMLInputElement | null = vnode.dom.querySelector("input[name='name_setting']");
              if (!nameSettingNode) {
                return;
              }

              var name_setting = nameSettingNode.value;
              var slug = name_setting.replace(/[.,\/#!$%\^&\*;:{}=\-_`~()]/g, "");

              var setting = {
                "name": "" + slug.replace(/\s+/g, "-").toLowerCase() + "",
                "param": {
                  "name": "" + name_setting + "",
                  "cmds": []
                }
              };

              if (name_setting) {
                vnode.attrs.ptz_config.settings.push(setting);

                api.setAttributes({
                  obj: vnode.attrs.obj,
                  attributes: {PTZ_CONFIG: JSON.stringify(vnode.attrs.ptz_config)}
                }).then((response) => {
                  CameraCellPTZModel.isShowPtzSettingsAdd = false;
                  Log.info(__("Setting create"));
                });
              } else {
                Log.error(__("Enter setting name!"));
              }
            }
          }),
          m("input", {
            value: __("Cancel"),
            type: "button",
            onclick: function () {
              CameraCellPTZModel.isShowPtzSettingsAdd = false;
            }
          })
        ])
      ]);
    }
  };

  type CameraPTZSettingEditAttribute = {
    obj: UUID,
    ptz_config: PtzConfig,
    setting: PtzConfigSettings
  }
  type CameraPTZSettingEditState = {
    setting: PtzConfigSettingsCmds,
    setting_name: string,
  }
  const CameraPTZSettingEdit: m.Component<CameraPTZSettingEditAttribute, CameraPTZSettingEditState> = {
    oninit: ({state}) => {
      state.setting_name = "";
    },
    view: function (vnode) {
      return m(".ptz-menu.ptz-preset-popup", [
        m("h4", [
          "PTZ Setting " + vnode.attrs.setting.param.name + " Edit",
          m("i.far.fa-eye-slash", {
            onclick: function () {
              CameraCellPTZModel.isShowPtzSettingsEdit = false;
            }
          })
        ]),
        m("label[for='name_setting']", __("Name")),
        m("input[data-name='" + vnode.attrs.setting.param.name + "']", {
          value: vnode.attrs.setting.param.name,
          name: "name_setting",
          placeholder: "setting name",
          type: "text"
        }),
        m("table",
          m("thead", [

            m("tr", [
              m("th", __("Name")),
              m("th", __("Command")),
              m("th", __("Value")),
              m("th", __("Option")),
              m("th", m("i.far.fa-plus-square", {
                onclick: function () {
                  CameraCellPTZModel.isShowPtzSettingsCommandEdit = false;
                  CameraCellPTZModel.isShowPtzSettingsCommandAdd = true;
                }
              })),
            ])
          ]),
          m("tbody", [
            vnode.attrs.setting.param.cmds.length > 0
              ? vnode.attrs.setting.param.cmds.map((val) => {
                return m("tr", [
                  m("td", val.name),
                  m("td", val.cmd.mode + "/" + val.cmd.cmd),
                  m("td", val.cmd.param),
                  m("td", "-"),
                  m("td",
                    m("i.fas.fa-edit", {
                      onclick: function () {
                        vnode.state.setting = val;
                        CameraCellPTZModel.isShowPtzSettingsCommandAdd = false;
                        CameraCellPTZModel.isShowPtzSettingsCommandEdit = true;
                      }
                    })
                  )
                ]);
              })
              : "-"
          ]),
        ),
        m(".menu-btn", [
          m("input", {
            value: __("Apply"),
            type: "button",
            onclick: function () {
              const nameSettingNode: HTMLInputElement | null = vnode.dom.querySelector("input[name='name_setting']");
              if (!nameSettingNode) {
                return;
              }

              vnode.attrs.ptz_config.settings.map(function (item, index, arr) {
                if (item.param.name == nameSettingNode.getAttribute("data-name")) {
                  item.param.name = nameSettingNode.value;

                  api.setAttributes({
                    obj: vnode.attrs.obj,
                    attributes: {PTZ_CONFIG: JSON.stringify(vnode.attrs.ptz_config)}
                  }).then((response) => {
                    Log.info(__("Setting save"));
                  });
                }
              });
            }
          }),
          m("input", {
            value: __("Cancel"),
            type: "button",
            onclick: function () {
              CameraCellPTZModel.isShowPtzSettingsEdit = false;
            }
          })
        ]),
        CameraCellPTZModel.isShowPtzSettingsCommandAdd && m(CameraPTZSettingCommandAdd, {
          setting_name: vnode.attrs.setting.param.name,
          obj: vnode.attrs.obj,
          ptz_config: vnode.attrs.ptz_config
        }),
        CameraCellPTZModel.isShowPtzSettingsCommandEdit && m(CameraPTZSettingCommandEdit, {
          setting: vnode.state.setting,
          obj: vnode.attrs.obj,
          ptz_config: vnode.attrs.ptz_config
        })
      ]);
    }
  };

  type CameraPTZSettingCommandAddAttributes = {
    obj: UUID,
    ptz_config: PtzConfig,
    setting_name: string
  }
  type CameraPTZSettingCommandAddState = {}
  const CameraPTZSettingCommandAdd: m.Component<CameraPTZSettingCommandAddAttributes, CameraPTZSettingCommandAddState> = {
    view: function (vnode) {
      return [
        m("table",
          m("tbody", [
            m("tr", [
              m("td", __("Name")),
              m("td", m("input[name='setting_name'][type='text']"))
            ]),
            m("tr", [
              m("td", __("Command")),
              m("td", m("input[name='com'][type='text']"))
            ]),
            m("tr", [
              m("td", __("Value")),
              m("td", m("input[name='val'][type='text']"))
            ]),
            m("tr", [
              m("td", __("Option")),
              m("td", m("input[name='opt'][type='text']"))
            ]),
          ])
        ),
        m(".menu-btn", [
          m("input", {
            value: __("Try"),
            type: "button",
            onclick: function () {
            }
          }),
          m("input", {
            value: __("Apply"),
            type: "button",
            onclick: function () {
              const settingNameNode: HTMLInputElement | null = vnode.dom.querySelector("input[name='setting_name']");
              const commandNode: HTMLInputElement | null = vnode.dom.querySelector("input[name='com']");
              const valueNode: HTMLInputElement | null = vnode.dom.querySelector("input[name='val']");
              const optionNode: HTMLInputElement | null = vnode.dom.querySelector("input[name='opt']");

              if (!settingNameNode || !commandNode || !valueNode || !optionNode) {
                return;
              }

              const setting_name = settingNameNode.value;
              // @ts-ignore
              const command = commandNode.value;
              const value = valueNode.value;
              // @ts-ignore
              const option = optionNode.value;

              var setting_command: PtzConfigSettingsCmds = {
                name: setting_name,
                cmd: {
                  mode: "settings",
                  cmd: "thermal-hot",
                  param: (value as PtzConfigSettingsCmds["cmd"]["param"])
                }
              };

              if (setting_name) {
                vnode.attrs.ptz_config.settings.map(function (item, index, arr) {
                  if (item.param.name == vnode.attrs.setting_name) {
                    item.param.cmds.push(setting_command);
                  }
                });

                api.setAttributes({
                  obj: vnode.attrs.obj,
                  attributes: {PTZ_CONFIG: JSON.stringify(vnode.attrs.ptz_config)}
                }).then((response) => {
                  Log.info(__("Command create"));
                  CameraCellPTZModel.isShowPtzSettingsCommandEdit = false;
                  CameraCellPTZModel.isShowPtzSettingsCommandAdd = false;
                });
              } else {
                Log.error(__(""));
              }
            }
          }),
          m("input", {
            value: __("Cancel"),
            type: "button",
            onclick: function () {
              CameraCellPTZModel.isShowPtzSettingsCommandEdit = false;
              CameraCellPTZModel.isShowPtzSettingsCommandAdd = false;
            }
          })
        ])
      ];
    }
  };

  type CameraPTZSettingCommandEditAttributes = {
    obj: UUID,
    setting: PtzConfigSettingsCmds,
    ptz_config: PtzConfig
  }
  type CameraPTZSettingCommandEditState = {}
  const CameraPTZSettingCommandEdit: m.Component<CameraPTZSettingCommandEditAttributes, CameraPTZSettingCommandEditState> = {
    view: function (vnode) {
      return [
        m("table",
          m("tbody", [
            m("tr", [
              m("td", __("Name")),
              m("td",
                m("input", {
                  value: vnode.attrs.setting.name,
                  name: "setting_name",
                  type: "text"
                })
              )
            ]),
            m("tr", [
              m("td", __("Command")),
              m("td",
                m("select[name='com']",
                  m("option", {
                      value: vnode.attrs.setting.cmd.mode + "/" + vnode.attrs.setting.cmd.cmd
                    },
                    vnode.attrs.setting.cmd.mode + "/" + vnode.attrs.setting.cmd.cmd
                  )
                )
              )
            ]),
            m("tr", [
              m("td", __("Value")),
              m("td",
                m("select[name='val']",
                  m("option", {
                      value: vnode.attrs.setting.cmd.param
                    },
                    vnode.attrs.setting.cmd.param
                  )
                )
              )
            ]),
            m("tr", [
              m("td", __("Option")),
              m("td",
                m("input", {
                  name: "opt",
                  type: "text",
                  value: ""
                })
              )
            ]),
          ])
        ),
        m(".menu-btn", [
          m("input", {
            value: __("Try"),
            type: "button",
            onclick: function () {
            }
          }),
          m("input", {
            value: __("Apply"),
            type: "button",
            onclick: function () {
              /*
              var setting_name = vnode.dom.querySelector("input[name='setting_name']");
              var command = vnode.dom.querySelector("input[name='com']");
              var value = vnode.dom.querySelector("input[name='val']");
              var option = vnode.dom.querySelector("input[name='opt']");
              */
            }
          }),
          m("input", {
            value: __("Cancel"),
            type: "button",
            onclick: function () {
              CameraCellPTZModel.isShowPtzSettingsCommandEdit = false;
              CameraCellPTZModel.isShowPtzSettingsCommandAdd = false;
            }
          })
        ])
      ];
    }
  };

  const CameraCellPTZPresetsCount = {
    view: function () {
      return m('.ptz-menu.ptz-preset-popup', [
        m("h4"),
        m('.count-preset',
          m("input[type='checkbox'][name='allow_preset']"),
          m("label[for='allow_preset']", "Allow PTZ preset control")
        ),
        m('.count-preset',
          m("label[for='number_preset']", "Number of presets"),
          m("input[name='number_preset'][type='text'][value='30']")
        ),
        m(".menu-btn", [
          m("input[value='Apply'][type='button']", {
            onclick: function () {
            }
          }),
          m("input[value='Cancel'][type='button']", {
            onclick: function () {
              CameraCellPTZModel.isShowPtzPresetsCountEdit = false;
            }
          })
        ])
      ]);
    }
  };

  type CameraCellPTZPresetsAttributes = {
    obj: UUID,
    ptz_config: PtzConfig
  }
  type CameraCellPTZPresetsState = {
    preset_id: number,
    preset_name: string
  }
  const CameraCellPTZPresets: m.Component<CameraCellPTZPresetsAttributes, CameraCellPTZPresetsState> = {
    oninit: ({state}) => {
      state.preset_id = 0;
      state.preset_name = '';
    },
    view: function (vnode) {
      return [
        m(".ptz-menu.ptz-preset-menu", [
          m("h4", [
            "PTZ Presets",
            m("i.far.fa-plus-square", {
              onclick: function () {
                CameraCellPTZModel.isShowPtzPresetAdd = true;
              }
            }),
            m("i.far.fa-eye-slash", {
              onclick: function () {
                CameraCellPTZModel.isShowPtzMenuPresets = false;
              }
            })
          ]),
          CameraCellPTZModel.isShowPtzPresetAdd && m(CameraPTZPresetAdd, {
            obj: vnode.attrs.obj,
            ptz_config: vnode.attrs.ptz_config
          }),
          m("ul", vnode.attrs.ptz_config.presets.length > 0
            ? vnode.attrs.ptz_config.presets.map((conf) => {
              return m("li", [
                m("i.fas.fa-edit", {
                  onclick: function () {
                    CameraCellPTZModel.isShowPtzPresetEdit = true;
                    vnode.state.preset_id = conf.id;
                    vnode.state.preset_name = conf.name;
                  }
                }),
                m("span",
                  {
                    onclick: function () {
                      joystick.ptz(Ptz_Mode.PRESET, {
                        name: "goto",
                        value: Number(conf.id)
                      })

                    }
                  },
                  conf.id + ". " + conf.name
                )
              ]);
            })
            : m("li", 'no preset')
          ),
          CameraCellPTZModel.isShowPtzPresetEdit && m(CameraPTZPresetEdit, {
            id: vnode.state.preset_id,
            name: vnode.state.preset_name,
            obj: vnode.attrs.obj,
            ptz_config: vnode.attrs.ptz_config
          })
        ]),
      ];
    }
  };

  type CameraPTZPresetAddAttribtues = {
    obj: UUID,
    ptz_config: PtzConfig
  }
  type CameraPTZPresetAddState = {
    preset_count: number[]
  }
  const CameraPTZPresetAdd: m.Component<CameraPTZPresetAddAttribtues, CameraPTZPresetAddState> = {
    oninit: function (vnode) {
      vnode.state.preset_count = [1, 2, 3, 4, 5, 6, 7, 8, 9, 10]; //test
      vnode.attrs.ptz_config.presets.length > 0
        ? vnode.attrs.ptz_config.presets.map((conf) => {
          if (vnode.state.preset_count.indexOf(conf.id) != -1) {
            vnode.state.preset_count.splice(vnode.state.preset_count.indexOf(conf.id), 1);
          }
        })
        : '';
    },
    view: function (vnode) {
      return m('.ptz-menu',
        m(".ptz-preset-popup.add-preset", [
          m("h4",
            __("Add PTZ preset"),
            m("i.far.fa-eye-slash", {
              onclick: function () {
                CameraCellPTZModel.isShowPtzPresetAdd = false;
              }
            })
          ),
          m("label[for='preset']",
            __("Preset name")
          ),
          m("select[name='preset_id']",
            vnode.state.preset_count.map((val) => {
              return m("option[value='" + val + "']", val);
            })
          ),
          m("input[name='name_preset'][type='text']"),
          m("br"),
          m(".menu-btn", [
            m("input", {
              value: __("Apply"),
              type: "button",
              onclick: function () {
                const namePresetNode: HTMLInputElement | null = vnode.dom.querySelector("input[name='name_preset']");
                const presetIdNode: HTMLInputElement | null = vnode.dom.querySelector("[name='preset_id']");
                if (!namePresetNode || !presetIdNode) {
                  return;
                }

                const name_preset = namePresetNode.value;
                const preset_id = presetIdNode.value;
                var preset = {
                  "id": parseInt(preset_id),
                  "name": "" + name_preset + ""
                };

                if (name_preset && preset_id) {
                  vnode.attrs.ptz_config.presets.push(preset);
                  CameraCellPTZModel.isShowPtzPresetAdd = false;
                  api.setAttributes({
                    obj: vnode.attrs.obj,
                    attributes: {PTZ_CONFIG: JSON.stringify(vnode.attrs.ptz_config)}
                  }).then((response) => {
                    joystick.ptz(Ptz_Mode.PRESET, {
                      name: "save",
                      value: Number(preset_id)
                    })
                    Log.info(__("Preset create"));
                  });
                } else {
                  Log.error(__("Enter preset name or id!"));
                }
              }
            }),
            m("input", {
              value: __("Cancel"),
              type: "button",
              onclick: function () {
                CameraCellPTZModel.isShowPtzPresetAdd = false;
              }
            })
          ])
        ])
      );
    }
  };

  type CameraPTZPresetEditAttributes = {
    obj: UUID,
    id: number,
    name: string,
    ptz_config: PtzConfig
  }
  type CameraPTZPresetEditState = {
    name: string
  }
  const CameraPTZPresetEdit: m.Component<CameraPTZPresetEditAttributes, CameraPTZPresetEditState> = {
    oninit: function (vnode) {
      vnode.state.name = vnode.attrs.name;
    },
    view: function (vnode) {
      return m('.ptz-menu',
        m(".ptz-preset-popup", [
          m("h4", "Edit PTZ preset " + vnode.attrs.id,
            m("i.far.fa-eye-slash", {
              onclick: function () {
                CameraCellPTZModel.isShowPtzPresetEdit = false;
              }
            })
          ),
          m("label[for='preset']",
            "Preset name"
          ),
          m("input[data-id='" + vnode.attrs.id + "']", {
            oninput: function (e: KeyboardEvent) {
              vnode.state.name = (e.target as HTMLInputElement).value;
            },
            value: vnode.state.name,
            name: 'name_preset',
            placeholder: 'preset name',
            type: "text"
          }),
          m("br"),
          m(".menu-btn", [
            m("input[value='Apply'][type='button']", {
              onclick: function () {
                const namePresetNode: HTMLInputElement | null = vnode.dom.querySelector("input[name='name_preset']");
                if (!namePresetNode) {
                  return;
                }

                const id = namePresetNode.getAttribute("data-id");
                if (!id) {
                  console.warn("namePresetNode data-id attribute is undefined")
                  return;
                }

                const searchPreset = searchInJson(vnode.attrs.ptz_config.presets, 'id', id);

                searchPreset[0].name = namePresetNode.value;

                api.setAttributes({
                  obj: vnode.attrs.obj,
                  attributes: {PTZ_CONFIG: JSON.stringify(vnode.attrs.ptz_config)}
                }).then((response) => {
                  joystick.ptz(Ptz_Mode.PRESET, {
                    name: "save",
                    value: Number(id)
                  })
                  Log.info(__("Preset save"));
                });
                CameraCellPTZModel.isShowPtzPresetEdit = false;
              }
            }),
            m("input[value='Cancel'][type='button']", {
              onclick: function () {
                CameraCellPTZModel.isShowPtzPresetEdit = false;
              }
            }),
            m("input[value='Delete'][type='button']", {
              onclick: function () {
                const namePresetNode: HTMLInputElement | null = vnode.dom.querySelector("input[name='name_preset']");
                if (!namePresetNode) {
                  return;
                }

                const id = namePresetNode.getAttribute("data-id");

                vnode.attrs.ptz_config.presets.map(function (item, index, arr) {
                  if (item.id == id) {
                    arr.splice(index, 1);
                  }
                });

                api.setAttributes({
                  obj: vnode.attrs.obj,
                  attributes: {PTZ_CONFIG: JSON.stringify(vnode.attrs.ptz_config)}
                }).then((response) => {
                  joystick.ptz(Ptz_Mode.PRESET, {
                    name: "clear",
                    value: Number(id)
                  })
                  Log.info(__("Preset delete"));
                });
                CameraCellPTZModel.isShowPtzPresetEdit = false;
              }
            })
          ])
        ])
      );
    }
  };

  type CameraCellPTZControlAttributes = {
    obj: UUID,
    ptz_config: PtzConfig,
    onHide?: () => void,
    setClickOnScreen?: (enabled: boolean) => void 
  }
  type CameraCellPTZControlState = {}
  const CameraCellPTZControl: m.Component<CameraCellPTZControlAttributes, CameraCellPTZControlState> = {
    oncreate: function ({attrs}) {
      joystick.init();
    },
    view: function (vnode) {
      return m(`${CameraCellPTZModel.isClickOnScreen ? ".ptz click-on-screen" : ".ptz"}`, [
        m(".connected-device",
          m(".ptz-ctl-box.ptz", [
            m(".ptz-menu.ptz-menu-general",
              m(".device-info-box",
                CameraCellPTZModel.isShowCameraCellPTZ
                  ? m("i.far.fa-eye-slash", {
                    onclick: function () {
                      CameraCellPTZModel.isShowCameraCellPTZ = false;
                    }
                  })
                  : m("i.fas.fa-eye", {
                    onclick: function () {
                      CameraCellPTZModel.isShowCameraCellPTZ = true;
                    }
                  })
              ),
              m(".device-info-box.close-ptz", {
                  onclick: function () {
                    vnode.attrs.onHide && vnode.attrs.onHide();
                    CameraCellPTZModel.isShowPtzPresetEdit = false;
                  }
                },
                m("span.name", m("i.fas.fa-times"))
              ),
              CameraCellPTZModel.isShowCameraCellPTZ && m("ul", [
                m(CameraCellPTZModel.isShowPtzMenuOptical ? "li.active" : "li",
                  m("i.fas.fa-edit", {
                    onclick: function () {
                      CameraCellPTZModel.isShowPtzMenuOpticalEdit = true;
                    }
                  }),
                  m("a[href='']", {
                      onclick: function (event: MouseEvent) {
                        CameraCellPTZModel.isShowPtzMenuOptical = true;
                        CameraCellPTZModel.isShowPtzMenuSettings = false;
                        CameraCellPTZModel.isShowPtzMenuPresets = false;
                        CameraCellPTZModel.isShowCameraCellPTZ = false;
                        CameraCellPTZModel.isClickOnScreen = false;

                        // turnClickOnScreenPtzStyles(false);
                        vnode.attrs.setClickOnScreen?.(false);
                        event.preventDefault();
                      }
                    },
                    "Optical PTZ"
                  )
                ),
                /*vnode.attrs.ptz_config.optical["click-to-reposition"] && m(CameraCellPTZModel.isClickOnScreen ? "li.active" : "li",
                  m("i.fas.fa-edit", {
                    onclick: function () {
                      CameraCellPTZModel.isShowPtzMenuOpticalEdit = true;;
                    }
                  }),
                  m("a[href='']", {
                      onclick: function (event: MouseEvent) {
                        const isClickOnScreen = !CameraCellPTZModel.isClickOnScreen

                        CameraCellPTZModel.isShowPtzMenuOptical = false;
                        CameraCellPTZModel.isClickOnScreen = isClickOnScreen;
                        CameraCellPTZModel.isShowPtzMenuSettings = false;
                        CameraCellPTZModel.isShowPtzMenuPresets = false;
                        CameraCellPTZModel.isShowCameraCellPTZ = false;

                        turnClickOnScreenPtzStyles(isClickOnScreen);
                        vnode.attrs.setClickOnScreen?.(isClickOnScreen);
                        event.preventDefault();
                      }
                    },
                    "Click on screen"
                  )
                ),
                
                m(CameraCellPTZ.isShowPtzMenuSettings ? "li.active" : "li",
                m("i.fas.fa-edit"),
                m("a[href='']",{
                onclick: function (event) {
                CameraCellPTZ.isShowPtzMenuOptical  = false;
                CameraCellPTZ.isShowPtzMenuSettings = true;
                CameraCellPTZ.isShowPtzMenuPresets = false;
                CameraCellPTZ.isShowCameraCellPTZ = false;
                event.preventDefault();
                }},
                "PTZ Settings"
                )
                ),
                */
                m(CameraCellPTZModel.isShowPtzMenuPresets ? "li.active" : "li",
                  m("i.fas.fa-edit", {
                    onclick: function () {
                      //CameraCellPTZ.isShowPtzPresetsCountEdit = true;
                    }
                  }),
                  m("a[href='']", {
                      onclick: function (event: MouseEvent) {
                        CameraCellPTZModel.isShowPtzMenuOptical = false;
                        CameraCellPTZModel.isShowPtzMenuSettings = false;
                        CameraCellPTZModel.isShowPtzMenuPresets = true;
                        CameraCellPTZModel.isShowCameraCellPTZ = false;
                        CameraCellPTZModel.isClickOnScreen = false;

                        // turnClickOnScreenPtzStyles(false);
                        vnode.attrs.setClickOnScreen?.(false);
                        event.preventDefault();
                      }
                    },
                    "PTZ Presets"
                  )
                )
              ]),
              CameraCellPTZModel.isShowPtzMenuOpticalEdit && m(CameraCellPTZOpticalEdit, {
                obj: vnode.attrs.obj,
                ptz_config: vnode.attrs.ptz_config
              }),
              CameraCellPTZModel.isShowPtzPresetsCountEdit && m(CameraCellPTZPresetsCount)
            ),
            (CameraCellPTZModel.isShowPtzMenuOptical || CameraCellPTZModel.isClickOnScreen) && m(CameraCellPTZOptical, {
              obj: vnode.attrs.obj,
              ptz_config: vnode.attrs.ptz_config
            }),
            CameraCellPTZModel.isShowPtzMenuSettings && m(CameraCellPTZSettings, {
              obj: vnode.attrs.obj,
              ptz_config: vnode.attrs.ptz_config
            }),
            CameraCellPTZModel.isShowPtzMenuPresets && m(CameraCellPTZPresets, {
              obj: vnode.attrs.obj,
              ptz_config: vnode.attrs.ptz_config
            })
          ])
        )
      ]);

      // function turnClickOnScreenPtzStyles(enabled: boolean) {
      //   if (!vnode.attrs.ptz_config.optical["click-to-reposition"]) return;
      //   vnode.dom.classList.toggle("click-on-screen", enabled);
      // }
    }
  };

  return {
    joystick,
    CameraCellPTZControl
  };
}
