import { useState, useEffect, useRef, useCallback, useMemo } from "react";
import { IconProp } from "@fortawesome/fontawesome-svg-core";
import produce from "immer";
import {
  useSolidConfigLazyQuery,
  useUpdateSolidConfigMutation,
  CreatingViewInput,
  useCurrentRealmLazyQuery,
  useUpdateCurrentRealmMutation,
  useSharedViewsUpdateSubscription,
  useUpdateSolidConfigCurrentViewMutation,
  SolidConfigCurrentViewInput,
  useSolidConfigUpdateSubscription,
  WidgetId,
  ActivityId,
  ViewId
} from "@generated/graphql";
import { ModuleInfo, isModuleInfo, ViewLayoutItemType, ViewLayoutItemOrientation, ModuleWindowProps } from "@core/types";
import { useStore } from "@core/store";
import { parseJSON, EventPubSub } from "utils";
import useEffectWithPrevious from "@core/useEffectWithPrevious";
import {isElectron} from "@solid/libs/utils";
import { LayoutType } from "components/Layout";
import { Widgets } from "components/Widgets";
import {__} from "@solid/libs/i18n";
import { EventView } from "components/EventList";
import { useAccessability } from "./accessability";
import useIsMounted from "@core/useIsMounted";

export const currentVersion = "v2";

export type SolidConfigRoot = {
  [currentVersion]: SolidConfig;
};

export type SolidConfig = {
  views?: ModuleInfo[];
  allViews?: ModuleInfo[];
  recentViewIds?: string[];
  layout?: string;
  windowProps?: Record<string, ModuleWindowProps>;
  settings?: GlobalSettings;
  timestamp?: number;
};

type SharedViewsRoot = {
  [currentVersion]: SharedViews;
};

type SharedViews = {
  sharedViews: ModuleInfo[];
  layout?: string;
};

export type GlobalSettings = {
  openViewListOnStart?: boolean;
  saveCurrentSession?: boolean;
};

function isConfig(value: any): value is SolidConfigRoot {
  if (!value) {
    return false;
  }
  const config = (value as SolidConfigRoot)[currentVersion];
  if (!config) {
    return false;
  }
  const views = config.views;
  if (views && !isModuleInfoArray(views)) {
    return false;
  }
  const allViews = config.allViews;
  if (allViews && !isModuleInfoArray(allViews)) {
    return false;
  }
  return true;
}

function isSharedViews(value: any): value is SharedViewsRoot {
  if (!value) {
    return false;
  }
  const views = (value as SharedViewsRoot)[currentVersion];
  return views && views.sharedViews && isModuleInfoArray(views.sharedViews);
}

function isModuleInfoArray(value: any): value is ModuleInfo[] {
  if (!Array.isArray(value)) {
    return false;
  }
  for (const view of value) {
    if (!isModuleInfo(view)) {
      return false;
    }
  }
  return true;
}

const systemViews: ModuleInfo[] = [
  { id: "start", name: __("Start Page"), module: "Views/ViewStart", isSystem: true, hideCloseButton: true, hideHeaderControls: true },

  { id: "archiveCameraGrid", name: __("Archive Camera Grid"), module: "ViewLayouts/ViewCustom", hasLayout: true, isSystem: true, isShared: true,
    aspectRatio: 16 / 9,
    layout: [
      { type: ViewLayoutItemType.Split, orientation: ViewLayoutItemOrientation.Horizontal, size: 1.0, items: [
        { type: ViewLayoutItemType.Cell, size: 0.2 },
        { type: ViewLayoutItemType.Split, orientation: ViewLayoutItemOrientation.Vertical, size: 0.8, items: [
          { type: ViewLayoutItemType.Split, orientation: ViewLayoutItemOrientation.Horizontal, size: 1.0, items: [
            { type: ViewLayoutItemType.Cell, size: 0.5 },
            { type: ViewLayoutItemType.Cell, size: 0.5 },
          ] },
          { type: ViewLayoutItemType.Split, orientation: ViewLayoutItemOrientation.Horizontal, size: 1.0, items: [
            { type: ViewLayoutItemType.Cell, size: 0.5 },
            { type: ViewLayoutItemType.Cell, size: 0.5 },
          ] },
          { type: ViewLayoutItemType.Split, orientation: ViewLayoutItemOrientation.Horizontal, size: 1.0, items: [
            { type: ViewLayoutItemType.Cell, size: 1.0 },
          ]},
        ]},
      ] }
    ],
    widgets: [
      { id: "", name: "", module: "CameraList", widgetId: WidgetId.DeviceList },
      { id: "ArchiveViewerCameraGrid1", name: "", module: "CellContainer/CameraCell", widgetId: WidgetId.ArchiveViewer },
      { id: "ArchiveViewerCameraGrid2", name: "", module: "CellContainer/CameraCell", widgetId: WidgetId.ArchiveViewer },
      { id: "ArchiveViewerCameraGrid3", name: "", module: "CellContainer/CameraCell", widgetId: WidgetId.ArchiveViewer },
      { id: "ArchiveViewerCameraGrid4", name: "", module: "CellContainer/CameraCell", widgetId: WidgetId.ArchiveViewer },
      { id: "", name: "", module: "Timeline", widgetId: WidgetId.Timeline },
    ] },

  { id: "cameraGrid", name: __("Camera Grid"), module: "ViewLayouts/ViewCustom", hasLayout: true, isSystem: true, isShared: true,
    aspectRatio: 16 / 9,
    layout: [
      { type: ViewLayoutItemType.Split, orientation: ViewLayoutItemOrientation.Horizontal, size: 1.0, items: [
        { type: ViewLayoutItemType.Cell, size: 0.2 },
        { type: ViewLayoutItemType.Split, orientation: ViewLayoutItemOrientation.Vertical, size: 0.4, items: [
          { type: ViewLayoutItemType.Cell, size: 0.5 },
          { type: ViewLayoutItemType.Cell, size: 0.5 },
        ] },
        { type: ViewLayoutItemType.Split, orientation: ViewLayoutItemOrientation.Vertical, size: 0.4, items: [
          { type: ViewLayoutItemType.Cell, size: 0.5 },
          { type: ViewLayoutItemType.Cell, size: 0.5 },
        ] },
      ] }
    ],
    widgets: [
      { id: "", name: "", module: "CameraList", widgetId: WidgetId.DeviceList },
      { id: "", name: "", module: "CellContainer/CameraCell", widgetId: WidgetId.CameraLive },
      { id: "", name: "", module: "CellContainer/CameraCell", widgetId: WidgetId.CameraLive },
      { id: "", name: "", module: "CellContainer/CameraCell", widgetId: WidgetId.CameraLive },
      { id: "", name: "", module: "CellContainer/CameraCell", widgetId: WidgetId.CameraLive },
    ] },

  { id: "vmx", name: __("VMX"), module: "ViewLayouts/ViewCustom", hasLayout: true, isSystem: true, isShared: true,
    aspectRatio: 16 / 9,
    layout: [
      { type: ViewLayoutItemType.Split, orientation: ViewLayoutItemOrientation.Horizontal, size: 1.0, items: [
        { type: ViewLayoutItemType.Cell, size: 0.2, },
        { type: ViewLayoutItemType.Cell, size: 0.8, },
      ] }
    ],
    widgets: [
      { id: "", name: "", module: "CameraList", widgetId: WidgetId.DeviceList },
      { id: "", name: "", module: "VMX", widgetId: WidgetId.Vmx },
    ] },
  { id: "essential", name: __("Essential"), module: "ViewLayouts/ViewCustom", hasLayout: true, isSystem: true, isShared: true,
    layout: [
      { type: ViewLayoutItemType.Split, orientation: ViewLayoutItemOrientation.Horizontal, size: 1.0, items: [
        { type: ViewLayoutItemType.Cell, size: 0.1  },
        { type: ViewLayoutItemType.Cell, size: 0.8, },
        { type: ViewLayoutItemType.Cell, size: 0.1, },
      ] }
    ],
    widgets: [
      { id: "", name: "", module: "EventList", widgetId: WidgetId.EventList, props: {
        eventView: EventView.Tiles, limitWidth: true, fixedWidth: 400
      }},
      { id: "", name: "", module: "CellContainer/CameraCell", widgetId: WidgetId.CameraLive },
      { id: "", name: "", module: "CameraList", widgetId: WidgetId.DeviceList, props: {
        limitWidth: true, fixedWidth: 250,
      }},
    ] },
];

export type SideBarItem = {
  id: ActivityId;
  name: string;
  title: string;
  icon: IconProp;
  views: ModuleInfo[];
  hidden?: boolean;
};

const eventListWidget = Widgets.find(w => w.id === WidgetId.EventList);
const linkProvisionWidget = Widgets.find(w => w.id === WidgetId.LinkProvision);
const usersWidget = Widgets.find(w => w.id === WidgetId.Users);
const groupsWidget = Widgets.find(w => w.id === WidgetId.Groups);
const zonesWidget = Widgets.find(w => w.id === WidgetId.Zones);
const policiesWidget = Widgets.find(w => w.id === WidgetId.Policies);
const devicesWidget = Widgets.find(w => w.id === WidgetId.AdminDevices);
const linksWidget = Widgets.find(w => w.id === WidgetId.AdminAvatars);
const hierarchyWidget = Widgets.find(w => w.id === WidgetId.HierarchyEdit);
const labelWidget = Widgets.find(w => w.id === WidgetId.LabelEdit);
const notificationPoliciesWidget =  Widgets.find(w => w.id === WidgetId.NotificationPolicies);
const realmSettingsWidget = Widgets.find(w => w.id === WidgetId.RealmSettings);
const manageAvatarsWidget = Widgets.find(w => w.id === WidgetId.ManageAvatars);
const WatchlistLPRWidget = Widgets.find(w => w.id === WidgetId.WatchlistLpr);
const WatchlistFacialWidget = Widgets.find(w => w.id === WidgetId.WatchlistFacial);
const setsWidget = Widgets.find(w => w.id === WidgetId.Sets);
const reportWidget = Widgets.find(w => w.id === WidgetId.Reports);
const videoWallsWidget = Widgets.find(w => w.id === WidgetId.VideoWalls);
const subscriptionWidget = Widgets.find(w => w.id === WidgetId.Subscription);
const quickStartWidget = Widgets.find(w => w.id === WidgetId.QuickStart);
const sensorsWidget = Widgets.find(w => w.id === WidgetId.AdminSensors);
const gatewayWidget = Widgets.find(w => w.id === WidgetId.AdminGateways);

const advancedAdminViews: SideBarItem[] = [
  {
    id: ActivityId.Devices,
    name: __("Devices"),
    title: __("Devices"),
    icon: "microchip",
    views: [
      { id: ViewId.Cameras, name: __("Cameras"), module: "ViewLayouts/ViewCustom", svgIcon: devicesWidget?.icon,
        hasLayout: true, isAdmin: true, hideHeaderControls: true,
        layout: [
          { type: ViewLayoutItemType.Cell, size: 1.0, },
        ],
        widgets: [
          { id: "", name: "", module: "Admin/AdminCameraList", widgetId: WidgetId.AdminDevices,  },
        ] },
      { id: ViewId.Sensors, name: __("Sensors"), module: "ViewLayouts/ViewCustom", svgIcon: sensorsWidget?.icon,
        hasLayout: true, isAdmin: true, hideHeaderControls: true,
        layout: [
          { type: ViewLayoutItemType.Cell, size: 1.0, },
        ],
        widgets: [
          { id: "", name: "", module: "Admin/Sensors", widgetId: WidgetId.AdminSensors },
        ] },
      { id: ViewId.Gateways, name: __("Gateways"), module: "ViewLayouts/ViewCustom", svgIcon: gatewayWidget?.icon,
        hasLayout: true, isAdmin: true, hideHeaderControls: true,
        layout: [
          { type: ViewLayoutItemType.Cell, size: 1.0, },
        ],
        widgets: [
          { id: "", name: "", module: "Admin/Gateways", widgetId: WidgetId.AdminGateways },
        ] },
      { id: ViewId.Avatars, name: __("Avatars"), module: "ViewLayouts/ViewCustom", svgIcon: linksWidget?.icon,
        hasLayout: true, isAdmin: true, hideHeaderControls: true,
        layout: [
          { type: ViewLayoutItemType.Cell, size: 1.0, },
        ],
        widgets: [
          { id: "", name: "", module: "Admin/AdminLinkList", widgetId: WidgetId.AdminAvatars },
        ] },
      { id: ViewId.AvatarActivation, name: __("Avatar Activation"), module: "ViewLayouts/ViewCustom", svgIcon: linkProvisionWidget?.icon,
        hasLayout: true, isAdmin: true, hideHeaderControls: true,
        layout: [
          { type: ViewLayoutItemType.Cell, size: 1.0, },
        ],
        widgets: [
          { id: "", name: "Avatar Activation", module: "Admin/AvatarActivation", widgetId: WidgetId.AvatarProvision },
        ] },
      { id: ViewId.DeviceAudit, name: __("Audit"), module: "ViewLayouts/ViewCustom", svgIcon: eventListWidget?.icon,
        hasLayout: true, isAdmin: true, hideHeaderControls: true,
        layout: [
          {type: ViewLayoutItemType.Split, orientation: ViewLayoutItemOrientation.Horizontal, size: 1.0, items: [
            { type: ViewLayoutItemType.Cell, size: 0.9, },
            { type: ViewLayoutItemType.Cell, size: 0.1, },
          ]
          }
        ],
        widgets: [
          { id: "", name: "", module: "DeviceEventList", widgetId: WidgetId.EventList },
          { id: "", name: "", module: "AuditEventInfo", widgetId: WidgetId.EventDetails },
        ] },
      { id: ViewId.Hierarchy, name: __("Hierarchy"), module: "ViewLayouts/ViewCustom", svgIcon: hierarchyWidget?.icon,
        hasLayout: true, isAdmin: true, hideHeaderControls: true,
        layout: [
          {type: ViewLayoutItemType.Cell, size: 1.0}
        ],
        widgets: [
          {id: "", name: "Hierarchy", module: "LabelsAndHierarchies/HierarchyEditor", widgetId: WidgetId.HierarchyEdit}
        ] },
      { id: ViewId.Labels, name: __("Labels"), module: "ViewLayouts/ViewCustom", svgIcon: labelWidget?.icon,
        hasLayout: true, isAdmin: true, hideHeaderControls: true,
        layout: [
          {type: ViewLayoutItemType.Cell, size: 1.0}
        ],
        widgets: [
          {id: "", name: "Labels", module: "LabelsAndHierarchies/LabelEditor", widgetId: WidgetId.LabelEdit}
        ] },
      { id: ViewId.Reports, name: __("Reports"), module: "ViewLayouts/ViewCustom", svgIcon: reportWidget?.icon,
        hasLayout: true, isAdmin: true, hideHeaderControls: true,
        layout: [
          {type: ViewLayoutItemType.Cell, size: 1.0}
        ],
        widgets: [
          {id: "", name: "Reports", module: "Admin/ConfigurationStatusReport", widgetId: WidgetId.Reports}
        ] },
      { id: ViewId.VideoWalls, name: __("Video Walls"), module: "ViewLayouts/ViewCustom", svgIcon: videoWallsWidget?.icon,
        hasLayout: true, isAdmin: true, hideHeaderControls: true,
        layout: [
          { type: ViewLayoutItemType.Cell, size: 1.0, },
        ],
        widgets: [
          { id: "", name: "", module: "VideoWalls", widgetId: WidgetId.VideoWalls },
        ] },
    ]
  },
  {
    id: ActivityId.Access,
    name: __("Access"),
    title: __("Access"),
    icon: "shield-alt",
    views: [
      { id: ViewId.Zones, name: __("Zones"), module: "ViewLayouts/ViewCustom", svgIcon: zonesWidget?.icon,
        hasLayout: true, isAdmin: true, hideHeaderControls: true,
        layout: [
          { type: ViewLayoutItemType.Cell, size: 1.0, },
        ],
        widgets: [
          { id: "", name: "", module: "Admin/Zones", widgetId: WidgetId.Zones },
        ] },
      { id: ViewId.Sets, name: __("Sets"), module: "ViewLayouts/ViewCustom", svgIcon: setsWidget?.icon,
        hasLayout: true, isAdmin: true, hideHeaderControls: true,
        layout: [
          { type: ViewLayoutItemType.Cell, size: 1.0, },
        ],
        widgets: [
          { id: "", name: "", module: "Admin/Sets", widgetId: WidgetId.Sets },
        ] },
      { id: ViewId.Policies, name: __("Authorization Policies"), module: "ViewLayouts/ViewCustom", svgIcon: policiesWidget?.icon,
        hasLayout: true, isAdmin: true, hideHeaderControls: true,
        layout: [
          { type: ViewLayoutItemType.Cell, size: 1.0, },
        ],
        widgets: [
          { id: "", name: "", module: "Admin/Policies", widgetId: WidgetId.Policies },
        ] },
      { id: ViewId.Users, name: __("Users"), module: "ViewLayouts/ViewCustom", svgIcon: usersWidget?.icon,
        hasLayout: true, isAdmin: true, hideHeaderControls: true,
        layout: [
          { type: ViewLayoutItemType.Cell, size: 1.0, },
        ],
        widgets: [
          { id: "", name: "", module: "Admin/Users", widgetId: WidgetId.Users },
        ] },
      { id: ViewId.Groups, name: __("Groups"), module: "ViewLayouts/ViewCustom", svgIcon: groupsWidget?.icon,
        hasLayout: true, isAdmin: true, hideHeaderControls: true,
        layout: [
          { type: ViewLayoutItemType.Cell, size: 1.0, },
        ],
        widgets: [
          { id: "", name: "", module: "Admin/Groups", widgetId: WidgetId.Groups },
        ] },
      { id: ViewId.UserAudit, name: __("Audit"), module: "ViewLayouts/ViewCustom", svgIcon: eventListWidget?.icon,
        hasLayout: true, isAdmin: true, hideHeaderControls: true,
        layout: [
          {type: ViewLayoutItemType.Split, orientation: ViewLayoutItemOrientation.Horizontal, size: 1.0, items: [
            { type: ViewLayoutItemType.Cell, size: 0.9, },
            { type: ViewLayoutItemType.Cell, size: 0.1, },
          ]
          }
        ],
        widgets: [
          { id: "", name: "", module: "UserEventList", widgetId: WidgetId.EventList },
          { id: "", name: "", module: "AuditEventInfo", widgetId: WidgetId.EventDetails },
        ] }
    ]
  },
  {
    id: ActivityId.Settings,
    name: __("Settings"),
    title: __("Settings"),
    icon: "sliders-h",
    views: [
      { id: ViewId.NotificationPolicies, name: __("Notification Policies"), module: "ViewLayouts/ViewCustom", svgIcon: notificationPoliciesWidget?.icon,
        hasLayout: true, isAdmin: true, hideHeaderControls: true,
        layout: [
          { type: ViewLayoutItemType.Cell, size: 1.0, },
        ],
        widgets: [
          { id: "", name: "", module: "NotificationPolicies", widgetId: WidgetId.NotificationPolicies },
        ] },
      { id: ViewId.RealmSettings, name: __("Realm Settings"), module: "ViewLayouts/ViewCustom", svgIcon: realmSettingsWidget?.icon,
        hasLayout: true, isAdmin: true, hideHeaderControls: true,
        layout: [
          { type: ViewLayoutItemType.Cell, size: 1.0, },
        ],
        widgets: [
          { id: "", name: "", module: "RealmSettings", widgetId: WidgetId.RealmSettings },
        ] },
      { id: ViewId.ManageAvatars, name: __("Manage Avatars"), module: "ViewLayouts/ViewCustom", svgIcon: manageAvatarsWidget?.icon,
        hasLayout: true, isAdmin: true, hideHeaderControls: true,
        layout: [
          { type: ViewLayoutItemType.Cell, size: 1.0, },
        ],
        widgets: [
          { id: "", name: "", module: "ManageAvatars", widgetId: WidgetId.ManageAvatars },
        ] },
      { id: ViewId.ManageAvatarsLpr, name: __("Watchlist: License Plate Recognition"), module: "ViewLayouts/ViewCustom", svgIcon: WatchlistLPRWidget?.icon,
        hasLayout: true, isAdmin: true, hideHeaderControls: true,
        layout: [
          { type: ViewLayoutItemType.Cell, size: 1.0, },
        ],
        widgets: [
          { id: "", name: "", module: "WatchlistLPR", widgetId: WidgetId.WatchlistLpr },
        ] },
      { id: ViewId.ManageAvatarsFacial, name: __("Watchlist: Facial"), module: "ViewLayouts/ViewCustom", svgIcon: WatchlistFacialWidget?.icon,
        hasLayout: true, isAdmin: true, hideHeaderControls: true,
        layout: [
          { type: ViewLayoutItemType.Cell, size: 1.0, },
        ],
        widgets: [
          { id: "", name: "", module: "WatchlistFacial", widgetId: WidgetId.WatchlistFacial },
        ] },
      { id: ViewId.EditMap, name: __("Edit Map"), module: "ViewLayouts/ViewCustom", svgIcon: WatchlistFacialWidget?.icon,
        hasLayout: true, isAdmin: true, hideHeaderControls: true, hidden: true,
        layout: [
          { type: ViewLayoutItemType.Cell, size: 1.0, },
        ],
        widgets: [
          { id: "", name: "", module: "GEOMap", widgetId: WidgetId.GeoMap },
        ] },
    ]
  },
  {
    id: ActivityId.About,
    name: "",
    title: __("About"),
    icon: "question",
    hidden: true,
    views: [
      { id: ViewId.Subscription, name: __("Subscription"), module: "ViewLayouts/ViewCustom", svgIcon: subscriptionWidget?.icon,
        hasLayout: true, isAdmin: true, hideHeaderControls: true,
        layout: [
          { type: ViewLayoutItemType.Cell, size: 1.0, },
        ],
        widgets: [
          { id: "", name: "", module: "Subscription", widgetId: WidgetId.Subscription },
        ] },
      { id: ViewId.QuickStart, name: __("Quick Start"), module: "ViewLayouts/ViewCustom", svgIcon: quickStartWidget?.icon,
        hasLayout: true, isAdmin: true, hideHeaderControls: true,
        layout: [
          { type: ViewLayoutItemType.Cell, size: 1.0, },
        ],
        widgets: [
          { id: "", name: "", module: "QuickStart", widgetId: WidgetId.QuickStart },
        ] },
      { id: ViewId.TrainingVideos, name: __("Training Videos"), module: "ViewLayouts/ViewCustom", svgIcon: quickStartWidget?.icon,
        hasLayout: true, isAdmin: true, hideHeaderControls: true,
        layout: [
          { type: ViewLayoutItemType.Cell, size: 1.0, },
        ],
        widgets: [
          { id: "", name: "", module: "TrainingVideos", widgetId: WidgetId.TrainingVideos },
        ] },
    ]
  }
];

const LinkProvisionView =  {
  id: ViewId.LinkProvision, name: __("Avatar Discovery"), module: "ViewLayouts/ViewCustom", hasLayout: true, isAdmin: true, hideHeaderControls: true,
  layout: [
    { type: ViewLayoutItemType.Cell, size: 1.0, },
  ],
  widgets: [
    { id: "", name: "", module: "Admin/LinkProvision", widgetId: WidgetId.LinkProvision },
  ]
};

export type ConfigOptions = {
  loadFromDb?: boolean;
  onLoadedFromDb?: (prevConfig: SolidConfig, nextConfig: SolidConfig, sharedViews: ModuleInfo[]) => void;
  onWriteToDb?: (config: SolidConfig) => void;
  onLoadedLayout?: (config: SolidConfig, layoutJSON: string, viewsChanged: boolean) => void;
};

export type ConfigResult = {
  views: ModuleInfo[];
  allViews: ModuleInfo[];
  startView: ModuleInfo;
  recentViews: ModuleInfo[];
  adminViews: SideBarItem[];
  layout: string;
  settings: GlobalSettings;
  setConfig: (config: SolidConfig, localOnly?: boolean, creatingViews?: CreatingViewInput[]) => void;
  clearRecentViews: () => void;
  loading: boolean;
  error?: Error;
  updateLoading: boolean;
  updateError?: Error;
  loadFromDbEvent: LoadFromDbEvent;
  routesViews: ModuleInfo[];
};

export type LoadFromDbArgs = {
  prevConfig: SolidConfig;
  nextConfig: SolidConfig;
  sharedViews: ModuleInfo[];
  views: ModuleInfo[];
  allViews: ModuleInfo[];
};

export class LoadFromDbEvent extends EventPubSub<LoadFromDbArgs> {}

export function useConfig(options: ConfigOptions = {}): ConfigResult {
  const { store: { session: { info }, workspace: { solidConfigJSON, localViewsJSON, sharedViewsJSON } }, setStore } = useStore();
  const { config: { limitedVideoAnalyticsAccess } } = useAccessability();
  const startView = systemViews[0];
  const defaultConfig: SolidConfig = useMemo(() => {
    const config = { views: [startView], allViews: systemViews };
    if (info && info.user.solidConfiguration.widgets.length > 0) {
      const availableWidgetsId = info.user.solidConfiguration.widgets.map(widgetConf => widgetConf.widgetId);
      const availableViews: ModuleInfo[] = [];
      for (const view of config.allViews) {
        const viewWidgetsId = view.widgets?.map(widget => widget.widgetId);
        if (viewWidgetsId && viewWidgetsId.some(widgetId => widgetId && availableWidgetsId.indexOf(widgetId) < 0)) {
          continue;
        }
        availableViews.push(view);
      }
      config.allViews = availableViews;
    }

    return config;
  }, [info]);

  const defaultConfigRoot: SolidConfigRoot = { [currentVersion]: defaultConfig };
  const [views, setViews] = useState<ModuleInfo[]>(defaultConfig.views!);
  const [allViews, setAllViews] = useState<ModuleInfo[]>(defaultConfig.allViews!);
  const [recentViews, setRecentViews] = useState<ModuleInfo[]>([]);
  const [layout, setLayout] = useState("");
  const [settings, setSettings] = useState<GlobalSettings>({});
  const [getSolidConfig, { data, loading, error }] = useSolidConfigLazyQuery();
  const [updateSolidConfig, result] = useUpdateSolidConfigMutation();
  const { data: updateData, loading: updateLoading, error: updateError } = result;
  const [getRealm, { data: realmData, loading: realmLoading, error: realmError }] = useCurrentRealmLazyQuery({ fetchPolicy: "no-cache" });
  const [updateRealm, realmUpdResult] = useUpdateCurrentRealmMutation();
  const { data: updateRealmData, loading: updateRealmLoading, error: updateRealmError } = realmUpdResult;
  const [updateCurrentView, currViewUpdResult] = useUpdateSolidConfigCurrentViewMutation();
  const { data: updateCurrViewData, loading: updateCurrViewLoading, error: updateCurrViewError } = currViewUpdResult;
  const loadFromDb = options.loadFromDb !== undefined ? options.loadFromDb : false;
  const [subSkip, setSubSkip] = useState(!loadFromDb);
  const { data: subConfigData, error: subConfigError } = useSolidConfigUpdateSubscription({ shouldResubscribe: !subSkip, skip: subSkip });
  const { data: subSharedData, error: subSharedError } = useSharedViewsUpdateSubscription({ shouldResubscribe: !subSkip, skip: subSkip });
  const stateUpdateTimestamp = useRef(0);
  const pendingConfigRef = useRef("");
  const pendingSharedViewsRef = useRef("");
  const pendingCurrViewRef = useRef<SolidConfigCurrentViewInput | undefined>();
  const configLoadedRef = useRef(false);
  const sharedLoadedRef = useRef(false);
  const layoutLoadedRef = useRef(false);
  const prevConfRef = useRef<SolidConfig>();
  const nextConfRef = useRef<SolidConfig>();
  const sharedViewsRef = useRef<ModuleInfo[]>();
  const loadFromDbEventRef = useRef(new LoadFromDbEvent());
  const isMounted = useIsMounted();

  const availableConfig = useMemo(() => {
    let availableViews: ViewId[] = [];
    let availableActivities: ActivityId[] = [];
    if (info && info.user.solidConfiguration.views.length > 0) {
      availableViews = info.user.solidConfiguration.views.map(viewConf => viewConf.viewId);
    }
    if (info && info.user.solidConfiguration.activities.length > 0) {
      availableActivities = info.user.solidConfiguration.activities;
    }
    if (limitedVideoAnalyticsAccess) {
      availableViews = availableViews.filter(viewId =>
        viewId !== ViewId.ManageAvatarsLpr &&
        viewId !== ViewId.ManageAvatarsFacial);
    }

    const config: Record<string, Array<ViewId | ActivityId>> = {
      views: availableViews,
      activities: availableActivities
    };

    return config;
  }, [info, limitedVideoAnalyticsAccess]);

  useEffectWithPrevious(([previousInfo]) => {
    if (info && loadFromDb) {
      if (JSON.stringify(previousInfo) !== JSON.stringify(info)) {
        getSolidConfig({ variables: { userId: info.user.id } });
        getRealm();
      }
    }
  }, [info, loadFromDb]);

  useEffect(() => {
    setSubSkip(!loadFromDb);
  }, [loadFromDb]);

  useEffect(useCallback(() => {
    error && console.error("Error executing config query:", error);

    let prevConfRoot = parseJSON<SolidConfigRoot>(solidConfigJSON);
    if (!isConfig(prevConfRoot)) {
      prevConfRoot = defaultConfigRoot;
    }

    let nextConfRoot = prevConfRoot;
    if (data && data.solidConfig) {
      const confRoot = parseJSON<SolidConfigRoot>(data.solidConfig);
      if (isConfig(confRoot)) {
        const conf = confRoot[currentVersion];
        const localViews = parseJSON<ModuleInfo[]>(localViewsJSON);
        if (localViews) {
          if (!conf.views) {
            conf.views = localViews;
          }
          else {
            conf.views = conf.views.concat(localViews);
          }
        }
        nextConfRoot = setConfigToState(conf, confRoot, undefined, true)[0];
      }
      else {
        nextConfRoot = setConfigToState(undefined, confRoot, undefined, true)[0];
        console.warn("Config object is empty or invalid.");
      }
    }

    if (data) {
      configLoadedRef.current = true;
      const prevConfig = prevConfRoot[currentVersion];
      const nextConfig = nextConfRoot[currentVersion];
      prevConfRef.current = prevConfig;
      nextConfRef.current = nextConfig;
      if (sharedLoadedRef.current) {
        const sharedViews = sharedViewsRef.current ?? [];
        onLoadedFromDb(prevConfig, nextConfig, sharedViews);
      }
    }
  }, [solidConfigJSON, localViewsJSON, sharedViewsJSON, loading, error, data, info]), [loading, error, data]);

  useEffect(() => {
    realmError && console.error("Error executing realm query:", realmError);

    if (realmData?.currentRealm) {
      const sharedViewsRoot = parseJSON(realmData.currentRealm.sharedViews);
      let sharedViews: ModuleInfo[] = [];
      if (isSharedViews(sharedViewsRoot)) {
        sharedViews = sharedViewsRoot[currentVersion].sharedViews;
        setStore({ workspace: { sharedViewsJSON: realmData.currentRealm.sharedViews }});
      }
      sharedLoadedRef.current = true;
      sharedViewsRef.current = sharedViews;
      const prevConfig = prevConfRef.current;
      const nextConfig = nextConfRef.current;
      if (configLoadedRef.current && prevConfig && nextConfig) {
        onLoadedFromDb(prevConfig, nextConfig, sharedViews);
      }
    }
  }, [realmData, realmLoading, realmError]);

  function onLoadedFromDb(prevConfig: SolidConfig, nextConfig: SolidConfig, sharedViews: ModuleInfo[]): void {
    options.onLoadedFromDb && options.onLoadedFromDb(prevConfig, nextConfig, sharedViews);
    loadFromDbEventRef.current.publish({ prevConfig, nextConfig, sharedViews,
      views: getViews(nextConfig, sharedViews), allViews: getAllViews(nextConfig, sharedViews) });
  }

  useEffect(useCallback(() => {
    subConfigError && console.error("Config update subscription error:", subConfigError);

    if (subConfigData) {
      const { solidConfigUpdate: { solidConfig } } = subConfigData;
      const confRoot = parseJSON(solidConfig);
      if (!isConfig(confRoot)) {
        return;
      }
      const { allViews, layout, settings } = confRoot[currentVersion];
      const conf: SolidConfig = { allViews, layout, settings };
      const currConfRoot = parseJSON<SolidConfigRoot>(solidConfigJSON);
      if (isConfig(currConfRoot)) {
        const { views, allViews, layout, settings } = currConfRoot[currentVersion];
        if (JSON.stringify(conf) === JSON.stringify({ allViews, layout, settings })) {
          return;
        }
        const newViews = views?.map(v => conf.allViews?.find(av => av.id === v.id) ?? v);
        if (newViews && JSON.stringify(newViews) !== JSON.stringify(views)) {
          conf.views = newViews;
        }
      }
      setConfigToState(conf, confRoot, undefined, true);
    }
  }, [solidConfigJSON, localViewsJSON, sharedViewsJSON, data, info, subConfigData, subConfigError]), [subConfigData, subConfigError]);

  useEffect(useCallback(() => {
    subSharedError && console.error("Shared views subscription error:", subSharedError);

    if (subSharedData) {
      const { sharedViewsUpdate: { sharedViews } } = subSharedData;
      if (isSharedViews(parseJSON(sharedViews)) && sharedViewsJSON !== sharedViews) {
        setStore({ workspace: { sharedViewsJSON: sharedViews }});
      }
    }
  }, [subSharedData, subSharedError, info, sharedViewsJSON]), [subSharedData, subSharedError]);

  useEffect(() => {
    updateError && console.error("Could not update config:", updateError);

    const userId = info?.user?.id;
    if (updateError && pendingConfigRef.current && userId) {
      console.log("Retrying config update in 1 second...");
      const solidConfig = pendingConfigRef.current;
      setTimeout(() => {
        if (!isMounted()) {
          return;
        }

        updateSolidConfig({ variables: { userId, solidConfig } });
      }, 1000);
    }

    if (!updateLoading && !updateError && updateData) {
      console.log("Config updated.");
      pendingConfigRef.current = "";
    }
  }, [updateLoading, updateError, updateData]);

  useEffect(() => {
    updateRealmError && console.error("Could not update shared views:", updateRealmError);

    const id = info?.realm?.id;
    if (updateRealmError && pendingSharedViewsRef.current && id) {
      console.log("Retrying shared views update in 1 second...");
      const sharedViews = pendingSharedViewsRef.current;
      setTimeout(() => {
        if (!isMounted) {
          return;
        }

        updateRealm({ variables: { realm: { id, sharedViews } } });
      }, 1000);
    }

    if (!updateRealmLoading && !updateRealmError && updateRealmData) {
      console.log("Shared views updated.");
      pendingSharedViewsRef.current = "";
    }
  }, [updateRealmLoading, updateRealmError, updateRealmData]);

  useEffect(() => {
    updateCurrViewError && console.error("Could not update current view:", updateCurrViewError);

    if (updateCurrViewError && pendingCurrViewRef.current) {
      console.log("Retrying current view update in 1 second...");
      const input = pendingCurrViewRef.current;
      setTimeout(() => {
        if (!isMounted) {
          return;
        }

        updateCurrentView({ variables: { input } });
      }, 1000);
    }

    if (!updateCurrViewLoading && !updateCurrViewError && updateCurrViewData) {
      console.log("Current view updated.");
      pendingCurrViewRef.current = undefined;
    }
  }, [updateCurrViewLoading, updateCurrViewError, updateCurrViewData]);

  useEffect(useCallback(() => {
    const newConfRoot = parseJSON<SolidConfigRoot>(solidConfigJSON);

    if (!isConfig(newConfRoot)) {
      return;
    }

    const newConf = newConfRoot[currentVersion];
    if (newConf.timestamp && newConf.timestamp <= stateUpdateTimestamp.current) {
      return;
    }

    const sharedViewsRoot = parseJSON(sharedViewsJSON);
    const sharedViews = isSharedViews(sharedViewsRoot) ? sharedViewsRoot[currentVersion].sharedViews : [];

    const newLocalViews = parseJSON<ModuleInfo[]>(localViewsJSON);

    const newViews = getViews(newConf, sharedViews, newLocalViews);
    const viewsChanged = JSON.stringify(views) !== JSON.stringify(newViews);
    setViews(newViews);

    const newAllViews = getAllViews(newConf, sharedViews);
    setAllViews(newAllViews);

    const newRecentViews = getRecentViews(newConf, newAllViews);
    setRecentViews(newRecentViews);

    const layoutJSON = mergeLayout(newConf.layout, sharedViewsJSON);
    onLoadedLayout(newConf, layoutJSON, viewsChanged);

    if (newConf.settings) {
      setSettings(newConf.settings);
    }

    stateUpdateTimestamp.current = Date.now();
  }, [solidConfigJSON, localViewsJSON, sharedViewsJSON, views]), [solidConfigJSON, localViewsJSON]);

  useEffect(useCallback(() => {
    const confRoot = parseJSON(solidConfigJSON);
    if (!isConfig(confRoot)) {
      return;
    }

    const conf = confRoot[currentVersion];
    const sharedViewsRoot = parseJSON(sharedViewsJSON);
    const sharedViews = isSharedViews(sharedViewsRoot) ? sharedViewsRoot[currentVersion].sharedViews : [];

    const newLocalViews = parseJSON<ModuleInfo[]>(localViewsJSON);
    const newViews = getViews(conf, sharedViews, newLocalViews);
    const viewsChanged = JSON.stringify(views) !== JSON.stringify(newViews);
    setViews(newViews);

    const newAllViews = getAllViews(conf, sharedViews);
    setAllViews(newAllViews);

    const newRecentViews = getRecentViews(conf, newAllViews);
    setRecentViews(newRecentViews);

    const layoutJSON = mergeLayout(conf.layout, sharedViewsJSON);
    onLoadedLayout(conf, layoutJSON, viewsChanged);
  }, [solidConfigJSON, localViewsJSON, sharedViewsJSON, views]), [sharedViewsJSON]);

  function onLoadedLayout(conf: SolidConfig, layoutJSON: string, viewsChanged: boolean): void {
    if (!layoutLoadedRef.current && layoutJSON && configLoadedRef.current && sharedLoadedRef.current) {
      layoutLoadedRef.current = true;
      options.onLoadedLayout && options.onLoadedLayout(conf, layoutJSON, viewsChanged);
    }
  }

  function mergeLayout(confLayout: string | undefined, sharedViewsJson: string | null | undefined, setToState: boolean = true): string {
    const sharedViewsRoot = parseJSON(sharedViewsJson);
    if (!isSharedViews(sharedViewsRoot) || !sharedViewsRoot[currentVersion].layout) {
      if (confLayout && setToState) {
        setLayout(confLayout);
      }
      return confLayout ?? "";
    }

    const layout = parseJSON<LayoutType>(confLayout) ?? {};
    if (!layout.allViews) {
      layout.allViews = {};
    }

    const sharedLayout = sharedViewsRoot[currentVersion].layout;
    const shared = parseJSON<LayoutType>(sharedLayout) ?? {};
    for (const id in shared) {
      if (!layout.allViews[id]) {
        layout.allViews[id] = shared[id];
      }
    }

    const layoutJSON = JSON.stringify(layout);
    if (setToState) {
      setLayout(layoutJSON);
    }
    return layoutJSON;
  }

  function addUniqueViews(views: ModuleInfo[], sourceViews?: ModuleInfo[]): ModuleInfo[] {
    const newViews: ModuleInfo[] = sourceViews || [];
    for (const view of views) {
      const index = newViews.findIndex(v => v.id === view.id);
      if (index >= 0)
      {
        newViews.splice(index, 1, view);
      }
      else {
        newViews.push(view);
      }
    }
    return newViews;
  }

  function getViews(conf: SolidConfig, sharedViews?: ModuleInfo[], localViews?: ModuleInfo[]): ModuleInfo[] {
    let newViews: ModuleInfo[] = [];
    if (conf.views) {
      newViews = addUniqueViews(conf.views, newViews);
      if (sharedViews) {
        newViews = newViews.filter(v => !v.isShared || v.isSystem || sharedViews.some(sv => sv.id === v.id));
        for (let i = 0; i < newViews.length; i++) {
          const view = newViews[i];
          const index = sharedViews.findIndex(v => v.id === view?.id);
          if (index >= 0) {
            newViews[i] = { ...sharedViews[index] };
          }
        }
      }
    }

    if (localViews) {
      newViews = addUniqueViews(localViews as ModuleInfo[], newViews);
    }

    const tabCount = isElectron() ? newViews.filter(v => !v.inWindow).length : newViews.length;
    if (tabCount > 1 && newViews.some(v => v.id === startView.id)) {
      newViews = newViews.filter(v => v.id !== startView.id);
    }
    if (tabCount === 0) {
      newViews.splice(0, 0, { ...startView, index: 0 });
    }

    newViews = newViews.map(view => {
      const wProps = conf.windowProps ? conf.windowProps[view.id] : undefined;
      return wProps ? { ...view, ...wProps } : view;
    });

    newViews.sort((a, b) => (a.index ?? -1) - (b.index ?? -1));
    return newViews;
  }

  function getAllViews(conf: SolidConfig, sharedViews: ModuleInfo[]): ModuleInfo[] {
    let newAllViews = Array.from(systemViews);
    if (conf.allViews) {
      newAllViews = addUniqueViews(conf.allViews, newAllViews);
    }

    newAllViews = addUniqueViews(sharedViews, newAllViews);

    newAllViews = newAllViews.map(view => {
      const wProps = conf.windowProps ? conf.windowProps[view.id] : undefined;
      return wProps ? { ...view, ...wProps } : view;
    });
    return newAllViews;
  }

  function getRecentViews(conf: SolidConfig, allViews: ModuleInfo[]): ModuleInfo[] {
    let newRecentViews: ModuleInfo[] = [];
    if (conf.recentViewIds) {
      newRecentViews = addUniqueViews(conf.recentViewIds
        .filter(id => id !== startView.id && allViews.some(view => view.id === id))
        .map(id => allViews.find(view => view.id === id)!));
    }
    return newRecentViews;
  }

  const setConfigToState = useCallback((
    conf?: SolidConfig,
    root?: Record<string, any>,
    creatingViews?: CreatingViewInput[],
    fromDb?: boolean
  ): [SolidConfigRoot, string, boolean, boolean, SolidConfigCurrentViewInput | undefined] => {
    let configRoot = defaultConfigRoot;
    const storeConfRoot = parseJSON<SolidConfigRoot>(solidConfigJSON);
    if (isConfig(storeConfRoot)) {
      configRoot = storeConfRoot;
    }

    if (loadFromDb && !data) {
      return [configRoot, sharedViewsJSON ?? "", false, false, undefined]; // Do not allow setting config before it is read from DB.
    }

    const timestamp = Date.now();

    let newViews: ModuleInfo[] = conf?.views ? Array.from(conf.views) : [];

    const sharedViewsRoot = parseJSON<SharedViewsRoot>(sharedViewsJSON) ?? { [currentVersion]: { sharedViews: [] } };

    let newAllViews: ModuleInfo[] = conf?.allViews ? Array.from(conf.allViews) : [];
    if (fromDb && isSharedViews(sharedViewsRoot)) {
      newAllViews = newAllViews.concat(sharedViewsRoot[currentVersion].sharedViews);
    }

    const currLocal = addUniqueViews(parseJSON<ModuleInfo[]>(localViewsJSON) ?? [], conf?.views?.filter(v => v.isLocal));
    const currShared = conf?.allViews?.filter(v => v.isShared) ?? sharedViewsRoot[currentVersion].sharedViews;

    const newConfRoot = produce(configRoot, draftConfRoot => {
      if (root) {
        for (const key in root) {
          if (key !== currentVersion) {
            draftConfRoot[key] = root[key];
          }
        }
      }

      const draftConf = draftConfRoot[currentVersion];
      if (conf) {
        const windowProps = {};

        if (conf.views) {
          newViews = newViews.map((view, index) => {
            const { inWindow, displayId, rect, ...props } = view;
            return { ...props, index };
          });

          draftConf.views = addUniqueViews(newViews.filter(v => !v.isLocal && (!v.isSystem || systemViews.some(s => s.id === v.id))));
          for (let i = 0; i < draftConf.views.length; i++) {
            const view = draftConf.views[i];
            const sysView = systemViews.find(s => s.id === view.id);
            if (sysView) {
              const { index, inWindow, displayId, rect, ...props } = view;
              if (JSON.stringify(props) !== JSON.stringify(sysView)) {
                draftConf.views[i] = { ...sysView, index: newViews.findIndex(v => v.id === view.id) };
              }
            }
          }

          const tabCount = isElectron() ? newViews.filter(v => !v.inWindow).length : newViews.length;
          if (tabCount === 0) {
            draftConf.views.splice(0, 0, { ...startView, index: 0 });
          }
        }

        if (conf.allViews) {
          conf.allViews.forEach(view => {
            const { inWindow, displayId, rect } = view;
            windowProps[view.id] = { inWindow, displayId, rect };
          });

          const allViews = addUniqueViews(conf.allViews.filter(v => !v.isLocal && !v.isSystem && !v.isShared && !systemViews.some(s => s.id === v.id)));
          draftConf.allViews = allViews.map(view => {
            const { inWindow, displayId, rect, ...props } = view;
            return props;
          });
        }

        if (conf.recentViewIds) {
          draftConf.recentViewIds = conf.recentViewIds.filter(id => id !== startView.id);
        }

        if (conf.layout) {
          draftConf.layout = conf.layout;
        }

        if (sharedViewsJSON) {
          draftConf.layout = mergeLayout(draftConf.layout, sharedViewsJSON, false);
          const layout = parseJSON<LayoutType>(draftConf.layout) ?? {};
          if (layout.allViews && draftConf.allViews) {
            for (const id in layout.allViews) {
              if (!draftConf.allViews.some(v => v.id === id) &&
                  !currShared.some(v => v.id === id) &&
                  !currLocal.some(v => v.id === id || v.id === id + "_selectWidgets")) {
                delete layout.allViews[id];
              }
            }
          }
          draftConf.layout = JSON.stringify(layout);
        }

        if (fromDb) {
          if (conf.windowProps) {
            draftConf.windowProps = conf.windowProps;
          }
        }
        else if (conf.allViews) {
          draftConf.windowProps = windowProps;
        }

        if (conf.settings) {
          draftConf.settings = conf.settings;
        }
      }

      draftConf.timestamp = timestamp;
    });

    const newConf = newConfRoot[currentVersion];
    let sharedLayoutChanged = false;

    const newSharedViewsRoot = produce(sharedViewsRoot, draftSharedViewsRoot => {
      if (conf?.allViews) {
        const sharedViews = addUniqueViews(newAllViews.filter(v => v.isShared && !v.isSystem)).map(view => {
          const { inWindow, displayId, rect, ...props } = view;
          return props;
        });
        draftSharedViewsRoot[currentVersion] = { sharedViews };
      }

      if (conf?.layout) {
        const layout = parseJSON<LayoutType>(conf.layout);
        const shared: LayoutType = parseJSON<LayoutType>(draftSharedViewsRoot[currentVersion].layout) ?? {};
        if (layout && layout.allViews) {
          const currShared = draftSharedViewsRoot[currentVersion].sharedViews;
          const allViews = getAllViews(newConf, currShared);
          for (const id in layout.allViews) {
            const view = allViews.find(v => v.id === id && v.isShared) ?? currShared.find(v => v.id === id);
            if (view?.userId && view?.userId === info?.user?.id && JSON.stringify(shared[id]) !== JSON.stringify(layout.allViews[id])) {
              shared[id] = layout.allViews[id];
              sharedLayoutChanged = true;
            }
          }
          for (const id in shared) {
            if (!allViews.some(v => v.id === id) && !currShared.some(v => v.id === id)) {
              delete shared[id];
              sharedLayoutChanged = true;
            }
          }
        }
        draftSharedViewsRoot[currentVersion].layout = JSON.stringify(shared);
      }
    });

    let newSharedViewsJSON: string | undefined;
    let sharedViewsUpdated = false;
    if (conf?.allViews || sharedLayoutChanged) {
      newSharedViewsJSON = JSON.stringify(newSharedViewsRoot);
      sharedViewsUpdated = newSharedViewsJSON !== sharedViewsJSON;
    }

    const newConfJSON = JSON.stringify(newConfRoot);
    // TODO: ignore timestamp field when comparing or they always will be different
    const configUpdated = newConfJSON !== solidConfigJSON;

    let currViewInput: SolidConfigCurrentViewInput | undefined;
    if (configUpdated && info?.user?.id) {
      const { views: oldViews, recentViewIds: oldRecent, windowProps: oldWindowProps, layout: oldLayout, timestamp: ts1, ...oldProps } = configRoot[currentVersion];
      const { views: newViews, recentViewIds: newRecent, windowProps: newWindowProps, layout: newLayout, timestamp: ts2, ...newProps } = newConfRoot[currentVersion];
      if (JSON.stringify(oldProps) === JSON.stringify(newProps) && (!conf?.layout || oldLayout === newLayout)) {
        const oldWindowPropsString = JSON.stringify(oldWindowProps);
        const newWindowPropsString = JSON.stringify(newWindowProps);
        currViewInput = {
          userId: info.user.id,
          version: currentVersion,
          currentViewId: newViews && newViews.length > 0 ? newViews[0].id : "",
          recentViewIds: newRecent ?? null,
          windowProps: oldWindowPropsString !== newWindowPropsString ? newWindowProps : null
        };
      }
    }

    let localViews: ModuleInfo[] | undefined;
    let newLocalViewsJSON: string | undefined;
    if (conf?.views) {
      localViews = addUniqueViews(newViews.filter(v => v.isLocal));
      newLocalViewsJSON = JSON.stringify(localViews);
    }

    setStore({ workspace: { solidConfigJSON: newConfJSON, localViewsJSON: newLocalViewsJSON, sharedViewsJSON: newSharedViewsJSON, creatingViews } });

    if (conf) {
      const sharedViews = newSharedViewsRoot[currentVersion].sharedViews;
      if (conf.views) {
        setViews(getViews(newConf, sharedViews, localViews));
      }
      if (conf.allViews) {
        setAllViews(getAllViews(newConf, sharedViews));
      }
      if (conf.recentViewIds && newConf.allViews) {
        setRecentViews(getRecentViews(newConf, getAllViews(newConf, sharedViews)));
      }
      mergeLayout(newConf.layout, newSharedViewsJSON || sharedViewsJSON);
      if (conf.settings) {
        setSettings(conf.settings);
      }
      stateUpdateTimestamp.current = timestamp;
    }
    return [newConfRoot, newSharedViewsJSON || sharedViewsJSON || "", configUpdated, sharedViewsUpdated, currViewInput];
  }, [solidConfigJSON, localViewsJSON, sharedViewsJSON, data, info]);

  const updateConfig = useCallback((conf: SolidConfig, localOnly?: boolean, creatingViews?: CreatingViewInput[]): void => {
    if (options.onWriteToDb && !localOnly) {
      options.onWriteToDb(conf);
    }

    const [newConfRoot, sharedViews, configUpdated, sharedViewsUpdated, currViewInput] = setConfigToState(conf, undefined, creatingViews);
    if (!localOnly) {
      if (configUpdated) {
        if (!currViewInput) {
          const userId = info?.user?.id;
          const solidConfig = JSON.stringify(newConfRoot);
          pendingConfigRef.current = solidConfig;
          if (userId && solidConfig) {
            updateSolidConfig({ variables: { userId, solidConfig } });
          }
        }
        else {
          pendingCurrViewRef.current = currViewInput;
          updateCurrentView({ variables: { input: currViewInput } });
        }
      }

      if (sharedViews && sharedViewsUpdated) {
        const id = info?.realm?.id;
        pendingSharedViewsRef.current = sharedViews;
        if (id && sharedViews) {
          updateRealm({ variables: { realm: { id, sharedViews } } });
        }
      }
    }
  }, [setConfigToState, info]);

  const clearRecentViews = useCallback((): void => {
    updateConfig({ recentViewIds: [] });
  }, [updateConfig]);

  let adminViews = [...advancedAdminViews];

  if (availableConfig.activities.length > 0) {
    adminViews = adminViews.filter(activity => availableConfig.activities.includes(activity.id));
  }

  if (availableConfig.views.length > 0) {
    adminViews = adminViews.map(activity => {
      const availableViews = activity.views.filter(view => availableConfig.views.includes(view.id as ViewId));
      activity.views = availableViews;
      return activity;
    });
  }

  const adminViewConfigs = adminViews.map(i => i.views).flat();
  const routesViews =  [...allViews, ...adminViewConfigs, LinkProvisionView];
  return {
    views,
    allViews,
    startView,
    recentViews,
    adminViews,
    layout,
    settings,
    setConfig: updateConfig,
    clearRecentViews,
    loading: loading || realmLoading,
    error: error || realmError,
    updateLoading: updateLoading || updateRealmLoading || updateCurrViewLoading,
    updateError: updateError || updateRealmError || updateCurrViewError,
    loadFromDbEvent: loadFromDbEventRef.current,
    routesViews
  };
}
