/* eslint-disable import/no-extraneous-dependencies */
import React, { useEffect, useState } from "react";
import ReactDOM from "react-dom";
import { HashRouter, Routes, Route, Navigate } from "react-router-dom";
import { HTML5Backend } from "react-dnd-html5-backend";
import { DndProvider } from "react-dnd";
import { ApolloProvider } from "@apollo/client";
import { useSessionInfoLazyQuery } from "@generated/graphql";
import { Apollo } from "@core/api";
import {isElectron} from "@solid/libs/utils";
import { origin, getBranding } from "core/api";
import Root from "./Root";
import PasswordExpiredPage from "pages/PasswordExpiredPage";
import LoginPage from "pages/LoginPage";
import ViewWindow from "components/View/ViewWindow";
import {useStore} from "@core/store";
import NotFound from "components/NotFound";
import Matrix from "components/Matrix";
import EulaPage from "pages/EulaPage";
import Example from "components/examples/Example";
import ConfirmCodePage from "pages/ConfirmCodePage";
import SelectGUIPage from "pages/SelectGUIPage";
import SelectRealmPage from "pages/SelectRealmPage/SelectRealmPage";
import AccountRecoveryPage from "pages/AccountRecoveryPage";
import ChangePasswordPage from "pages/ChangePasswordPage";

import "./style.css";

import "@fortawesome/fontawesome-free/css/all.css";
import "semantic-ui-less/semantic.less";
import "react-virtualized/styles.css";
import "styles/react_virtualized.css";

Apollo.init();

const defaultURL = process.env.START_URL || "/license";
const inWindow = isElectron() && !window.isMainWindow && window.location.hash.startsWith("#/view/");

const App = () => {
  const { setStore, store: { session: { isLoggedIn }, branding } } = useStore();
  const [sessionInfo, { error }] = useSessionInfoLazyQuery({ fetchPolicy: "network-only" });
  const [iconPath, setIconPath] = useState<string>("");

  const storeBrandingOptions = async () => {
    try {
      const options = await getBranding();
      if (options.branding) {
        setStore({ branding: options.branding });
      }
    } catch (e) {
      console.error(e);
    }
  };

  useEffect(() => {
    // remove loading indicator logic
    document.getElementById("root")?.removeAttribute("class");
    document.querySelector("link[href*='loading.css']")?.remove();
  }, []);

  useEffect(() => {
    const continueSession = setInterval(() => {
      isLoggedIn && sessionInfo();
    }, 300000);

    return () => clearInterval(continueSession);
  }, [isLoggedIn]);

  useEffect(() => {
    if (error) {
      error.graphQLErrors.forEach((err) => {
        err.extensions.code === 401 && window.location.reload();
      });
    }
  }, [error]);

  useEffect(() => {
    if (!branding) {
      storeBrandingOptions();
      return;
    }

    if (iconPath.length === 0) {
      setIconPath(`${origin}${branding.ICON_PATH}`);
    }
  }, [branding]);

  useEffect(() => {
    if (iconPath.length > 0) {
      const head = document.head.getElementsByClassName("favicon")[0] as HTMLLinkElement;
      if (head) {
        head.href = iconPath;
      }
    }
  }, [iconPath]);

  const noSlashAfterHashRedirect = () => {
    const hash = window.location.hash;
    if (hash[1] && hash[1] !== "/") {
      window.location.href = window.location.pathname;
    }
  };

  noSlashAfterHashRedirect();

  window.addEventListener("hashchange", noSlashAfterHashRedirect);

  return (
    <HashRouter>
      {!isLoggedIn
        ? (
          <Routes>
            <Route path="/passwordexpired" element={<PasswordExpiredPage/>} />
            <Route path="/confirm/code/:secret?" element={<ConfirmCodePage/>} />
            <Route path="/confirm/email/:secret" element={<ChangePasswordPage mode="email"/>} />
            <Route path="/recover/password/:secret" element={<ChangePasswordPage mode="password"/>} />
            <Route path="/recover/account" element={<AccountRecoveryPage/>} />
            <Route path="/*" element={<LoginPage/>} />
            <Route path="*" element={<NotFound message="Not Logged In"/>} />
          </Routes>
        )
        : (
          <DndProvider backend={HTML5Backend}>
            {inWindow ?
              <Routes>
                <Route path="/view/" element={<ViewWindow/>} />
                <Route path="/view/:viewId" element={<ViewWindow/>} />
                <Route path="*" element={<NotFound message="Separate Window"/>} />
              </Routes> :
              <Routes>
                <Route path="/" element={<Navigate to={defaultURL} replace/>} />
                <Route path="/matrix/:cameraId/" element={<Matrix/>} />
                <Route path="/matrix/:cameraId/:timestamp" element={<Matrix/>} />
                <Route path="/example" element={<Example/>}/>
                <Route path="/license" element={<EulaPage/>} />
                <Route path="/selectrealm" element={<SelectRealmPage/>} />
                <Route path="/view/" element={<Root/>} />
                <Route path="/view/:viewId" element={<Root/>} />
                <Route path="/view/link/edit/:linkId" element={<Root/>} />
                <Route path="/view/link/edit/:linkId/:tabId" element={<Root/>} />
                <Route path="/view/camera/edit/:cameraId" element={<Root/>} />
                <Route path="/view/camera/vae/:cameraId" element={<Root/>} />
                <Route path="/view/camera/add" element={<Root/>} />
                <Route path="/view/camera/import" element={<Root/>} />
                <Route path="/view/camera/discovery" element={<Root/>} />
                <Route path="/view/user/add" element={<Root/>} />
                <Route path="/view/user/edit/:userId" element={<Root/>} />
                <Route path="/view/policy/edit/:policyId" element={<Root/>} />
                <Route path="/view/policy/add" element={<Root/>} />
                <Route path="/view/group/edit/:groupId" element={<Root/>} />
                <Route path="/view/group/add" element={<Root/>} />
                <Route path="/view/zone/edit/:zoneId" element={<Root/>} />
                <Route path="/view/zone/add" element={<Root/>} />
                <Route path="/view/set/edit/:setId" element={<Root/>} />
                <Route path="/view/set/add" element={<Root/>} />
                <Route path="/view/sensor/add" element={<Root/>} />
                <Route path="/view/sensor/edit/:sensorId" element={<Root/>} />
                <Route path="/view/gateway/add" element={<Root/>} />
                <Route path="/view/gateway/edit/:gatewayId" element={<Root/>} />
                <Route path="/main" element={<Root/>} />
                <Route path="/guilist" element={<SelectGUIPage/>} />
                <Route path="*" element={<NotFound/>} />
              </Routes>}
          </DndProvider>
        )
      }
    </HashRouter>
  );
};

// @ts-ignore
if (window.initialized) {
  console.log("Cleanup");
  const root = document.getElementById("root");
  // @ts-ignore
  ReactDOM.unmountComponentAtNode(root);

  document.body.innerHTML = "<div class=\"loader-wrapper\"><div class=\"loader-container\"><div class=\"loader\"></div></div></div><div id=\"root\"></div>";
}
// @ts-ignore
window.initialized = true;

// remove loader container
document.querySelector(".loader-wrapper")?.remove();

ReactDOM.render(
  <ApolloProvider client={Apollo.client!}>
    <App/>
  </ApolloProvider>,
  document.getElementById("root")
);
