import { ApolloClient, InMemoryCache, NormalizedCacheObject, ApolloLink, split, HttpLink } from "@apollo/client";
import { onError } from "@apollo/client/link/error";
import { WebSocketLink } from "@apollo/client/link/ws";
import { SubscriptionClient } from "subscriptions-transport-ws";
import { API, getOrigin } from "@solid/libs";
import introspectionQueryResultData, {
  AppStore,
  SessionInfo,
  StoreQuery,
  StoreQueryVariables,
  StoreDocument,
  SetStoreDocument,
  SetStoreMutation,
  SetStoreMutationVariables,
  SocketEventType,
  Branding,
} from "@generated/graphql";
import typeDefs from "graphql/localSchema.graphql";
import resolvers from "graphql/resolvers";
import { toMaybe, addStartingSlash } from "utils";
import {getMainDefinition} from "@apollo/client/utilities";

console.log("CREATE API");

const api = new API();

export const origin = getOrigin();

export type ApolloType = {
  client: ApolloClient<NormalizedCacheObject> | null;
  subscriptionClient: SubscriptionClient | null;
  init: () => void;
  defaultStore: () => AppStore;
};

export const Apollo: ApolloType = {
  client: null,
  subscriptionClient: null,
  defaultStore: () => {
    return {
      __typename: "AppStore",
      session: {
        __typename: "Session",
        isLoggedIn: false,
        isAdmin: false,
        info: null
      },
      select: null,
      workspace: {
        __typename: "Workspace",
        event: null,
        updatedEvent: null,
        creatingViews: null,
        solidConfigJSON: null,
        localViewsJSON: null,
        sharedViewsJSON: null,
        cameraWidgetProps: null
      },
      socketEvent: null,
      branding: null
    };
  },
  init: () => {
    if (Apollo.client) return;

    console.log("init Apollo");

    let httpURI = process.env.API_GRAPHQL || "/api/graphql";
    httpURI = httpURI.startsWith("http") ? httpURI : origin + addStartingSlash(httpURI);

    console.log("process.env.API_GRAPHQL", process.env.API_GRAPHQL);
    const httpLink = new HttpLink({
      uri: httpURI,
      credentials: process.env.CREDENTIALS || "same-origin"
    });

    const wsURI = httpURI.replace("http", "ws");
    const subscriptionClient = new SubscriptionClient(wsURI, { reconnect: true });

    subscriptionClient.onError((e: Event) => {
      if (e && e.type === "error") {
        console.error("[Socket error]");
        setSocketEvent(SocketEventType.Error, e.timeStamp);
      }
    });

    subscriptionClient.onDisconnected(() => console.log("Subscription client disconnected"));
    subscriptionClient.onConnecting(() => console.log("Subscription client connecting..."));
    subscriptionClient.onConnected(() => console.log("Subscription client connected"));
    subscriptionClient.onReconnecting(() => console.log("Subscription client reconnecting..."));
    subscriptionClient.onReconnected(() => {
      setSocketEvent(SocketEventType.Reconnected, Date.now());
      console.log("Subscription client reconnected");
    });

    Apollo.subscriptionClient = subscriptionClient;

    const wsLink = new WebSocketLink(subscriptionClient);
    const link = split(
      ({ query }) => {
        const definition = getMainDefinition(query);
        return (definition.kind === "OperationDefinition" && definition.operation === "subscription");
      },
      wsLink,
      httpLink,
    );

    const errorLink = onError(({ operation, response, graphQLErrors, networkError }) => {
      if (operation.operationName === "userInfo") {
        return;
      }
      graphQLErrors && graphQLErrors.map(e => console.error(`[GraphQL error]: Message: ${e.message}, Path: ${e.path}, Error:`, e));
      networkError && console.error(`[Network error]: Message: ${networkError.message}, Error:`, networkError);
      networkError && networkError.message.includes("401") && window.location.reload();
    });

    const cache = new InMemoryCache({
      possibleTypes: introspectionQueryResultData.possibleTypes,
      typePolicies: {
        Query: {
          fields: {
            store: {
              merge(existing, incoming) {
                return { ...existing, ...incoming };
              }
            },
            policies: {
              merge: false
            },
            groups: {
              merge: false
            },
            users: {
              merge: false
            }
          }
        },
        Policy: {
          fields: {
            statements: {
              merge: false
            }
          }
        },
        Group: {
          fields: {
            users: {
              merge: false
            }
          }
        },
        User: {
          fields: {
            groups: {
              merge: false
            }
          }
        },
        Zone: {
          fields: {
            devices: {
              merge: false
            }
          }
        },
        Label: {
          fields: {
            objects: {
              merge: false
            }
          }
        }
      }
    });

    Apollo.client = new ApolloClient({
      link: ApolloLink.from([errorLink, link]),
      connectToDevTools: true,
      cache,
      typeDefs,
      resolvers
    });

    const data = { store: toMaybe(Apollo.defaultStore()) };

    cache.writeQuery<StoreQuery, StoreQueryVariables>({ query: StoreDocument, data });

    Apollo.client.onResetStore(() => new Promise<void>((resolve, reject) => {
      try {
        cache.writeQuery<StoreQuery, StoreQueryVariables>({ query: StoreDocument, data });
        resolve();
      }
      catch (e) {
        reject(e);
      }
    }));

    function setSocketEvent(type: SocketEventType, timeStamp: number): void {
      if (Apollo.client) {
        Apollo.client.mutate<SetStoreMutation, SetStoreMutationVariables>({
          mutation: SetStoreDocument,
          variables: { store: { socketEvent: { type, timeStamp } } } });
      }
    }
  }
};

export async function verify(code: string) {
  return api.verify<{info: SessionInfo}>({code});
}

export async function getBranding() {
  return api.getBranding<{branding: Branding}>();
}

