import { useState, RefObject, useEffect, useRef } from "react";
import { UUID } from "@solid/types";
import { getSnapshotURL } from "@solid/libs";
import { openDB, DBSchema, IDBPDatabase } from "idb";

interface SnapshotDB extends DBSchema {
  snapshots: {
    key: string,
    value: SnapshotValue
  }
}

type SnapshotValue = {
  deviceId: UUID,
  time: number,
  imageBase64: string
};

type SnapshotParams = {
  load: boolean;
  deviceId: UUID;
  imageRef?: RefObject<HTMLImageElement>;
};

type SnapshotResult = {
  loading: boolean;
  imageSrc: string;
  updating: boolean;
};

const snapshotTTL = 60 * 1000;

export function useSnapshot({ load, deviceId, imageRef }: SnapshotParams): SnapshotResult {
  const [loading, setLoading] = useState(false);
  const [imageSrc, setImageSrc] = useState("");
  const [updating, setUpdating] = useState<boolean>(false);
  const [snapshotDB, setSnapshotDB] = useState<IDBPDatabase<SnapshotDB>>();
  const deviceIdRef = useRef(deviceId);
  const mountedRef = useRef<boolean>();

  useEffect(() => {
    if (snapshotDB) {
      return;
    }

    mountedRef.current = true;
    initSnapshotDB().then((db) => {
      mountedRef.current && setSnapshotDB(db);
    });

    // eslint-disable-next-line consistent-return
    return () => {
      mountedRef.current = false;
    };
  }, []);

  useEffect(() => {
    deviceIdRef.current = deviceId;
  }, [deviceId]);

  useEffect(() => {
    if (!snapshotDB) {
      return;
    }

    if (load && deviceId) {
      getSnapshotById(deviceId)
        .then((snapshot) => {
          if (!snapshot && mountedRef.current) {
            setUpdating(false);
            setLoading(true);
            const isExternalImage = !!imageRef?.current;
            const loadingDeviceId = deviceId;
            const img = imageRef?.current ?? document.createElement("img");
            img.onload = () => {
              if (!mountedRef.current) {
                return;
              }
              setLoading(false);
              const src = storeImage(img, loadingDeviceId);
              if (!isExternalImage && loadingDeviceId === deviceIdRef.current) {
                setImageSrc(src);
              }
            };
            img.onerror = () => {
              if (!mountedRef.current) {
                return;
              }
              setLoading(false);
            };
            if (isExternalImage) {
              const imageUrl = getSnapshotURL(deviceId);
              setImageSrc(imageUrl);
            }
            else {
              const imageUrl = getSnapshotURL(deviceId);
              img.src = imageUrl;
            }
          }
          if (snapshot && mountedRef.current) {
            setImageSrc(snapshot.imageBase64);
            const time = snapshot.time;

            if (isNaN(time) || Date.now() - time > snapshotTTL) {
              setUpdating(true);
              const updatedImg = document.createElement("img");
              const loadingDeviceId = deviceId;
              const imageUrl = getSnapshotURL(loadingDeviceId);
              updatedImg.src = imageUrl;
              updatedImg.onload = () => {
                if (!mountedRef.current) {
                  return;
                }
                const updatedSrc = storeImage(updatedImg, loadingDeviceId);
                setImageSrc(updatedSrc);
                setUpdating(false);
              };
              updatedImg.onerror = () => {
                if (!mountedRef.current) {
                  return;
                }
                setUpdating(false);
              };
            }
          }
        })
        .catch((e) => {
          console.error(e);
          const imageUrl = getSnapshotURL(deviceId);
          setImageSrc(imageUrl);
        });
    }
  }, [load, deviceId, snapshotDB]);

  async function initSnapshotDB() {
    const db = await openDB<SnapshotDB>("SnapshotDB", 1, {
      upgrade(db) {
        db.createObjectStore("snapshots", {
          keyPath: "deviceId"
        });
      },
    });
    return db;
  }

  async function getSnapshotById(id: UUID): Promise<SnapshotValue | undefined> {
    try {
      const result = await snapshotDB?.get("snapshots", id);
      if (result) {
        return result;
      }
    } catch (e) {
      console.error(e);
    }
    return undefined;
  }

  function storeImage(img: HTMLImageElement, deviceId: UUID): string {
    const canvas = document.createElement("canvas");
    const context = canvas.getContext("2d");
    canvas.width = img.naturalWidth;
    canvas.height = img.naturalHeight;
    context?.drawImage(img, 0, 0);
    const src = canvas.toDataURL();
    saveSnapshot(src, deviceId);
    return src;
  }

  async function saveSnapshot(base64: string, deviceId: UUID): Promise<boolean> {
    const newSnapshot = {
      deviceId,
      time: Date.now(),
      imageBase64: base64
    };
    try {
      const result = await snapshotDB?.put("snapshots", newSnapshot);
      return !!result;
    } catch (e) {
      console.error(e);
      return false;
    }
  }

  return { loading, imageSrc, updating };
}

// Stub for clearing db
export const clearSnapshotsDB = async () => {
  const db = await openDB<SnapshotDB>("SnapshotDB", 1, {
    upgrade(db) {
      db.createObjectStore("snapshots", {
        keyPath: "deviceId"
      });
    },
  });

  return db.clear("snapshots");
};
