import { useEffect } from "react";
import { defaultDataIdFromObject, useApolloClient } from "@apollo/client";
import {
  AssignLabelsDocument,
  AssignLabelsMutation,
  AssignLabelsMutationVariables,
  DeleteLabelDocument,
  DeleteLabelMutation,
  DeleteLabelMutationVariables,
  InsertLabelsDocument,
  InsertLabelsMutation,
  InsertLabelsMutationVariables,
  Label,
  LabelDocument,
  LabelFieldsFragment,
  LabelFieldsFragmentDoc,
  LabelInput,
  LabelOrHierarchyUpdateType,
  LabelQuery,
  LabelQueryVariables,
  LabelsDocument,
  LabelsQuery,
  LabelsQueryVariables,
  RemoveLabelsDocument,
  RemoveLabelsMutation,
  RemoveLabelsMutationVariables,
  UpdateLabelDocument,
  UpdateLabelMutation,
  UpdateLabelMutationVariables,
  useLabelUpdateSubscription
} from "@generated/graphql";
import { useStore } from "@core/store";
import { readQuery } from "@core/utils";

type LabelActions = {
  insertLabels: (labels: LabelInput[]) => Promise<boolean>;
  updateLabel: (label: LabelInput) => Promise<boolean>;
  deleteLabel: (id: string) => Promise<boolean>;
  assignLabels: (objectId: string, labelIds: string[]) => Promise<boolean>;
  removeLabels: (objectId: string, labelIds: string[]) => Promise<boolean>;
};

export function useLabelActions(): LabelActions {
  const client = useApolloClient();

  async function insertLabels(labels: LabelInput[]): Promise<boolean> {
    const { data, errors } = await client.mutate<InsertLabelsMutation, InsertLabelsMutationVariables>({
      mutation: InsertLabelsDocument,
      variables: { labels },
      refetchQueries: [{ query: LabelsDocument }],
      awaitRefetchQueries: true,
      errorPolicy: "all"
    });
    if (data) {
      return data.insertLabels.length > 0;
    }
    if (errors) {
      throw errors[0];
    }
    return false;
  }

  async function updateLabel(label: LabelInput): Promise<boolean> {
    const { data, errors } = await client.mutate<UpdateLabelMutation, UpdateLabelMutationVariables>({
      mutation: UpdateLabelDocument,
      variables: { label },
      refetchQueries: [{ query: LabelsDocument }],
      awaitRefetchQueries: true,
      errorPolicy: "all"
    });
    if (data) {
      return data.updateLabel;
    }
    if (errors) {
      throw errors[0];
    }
    return false;
  }

  async function deleteLabel(id: string): Promise<boolean> {
    const { data, errors } = await client.mutate<DeleteLabelMutation, DeleteLabelMutationVariables>({
      mutation: DeleteLabelDocument,
      variables: { id },
      refetchQueries: [{ query: LabelsDocument }],
      awaitRefetchQueries: true,
      errorPolicy: "all"
    });
    if (data) {
      return data.deleteLabel;
    }
    if (errors) {
      throw errors[0];
    }
    return false;
  }

  async function assignLabels(objectId: string, ids: string[]): Promise<boolean> {
    const { data, errors } = await client.mutate<AssignLabelsMutation, AssignLabelsMutationVariables>({
      mutation: AssignLabelsDocument,
      variables: { objectId, ids },
      refetchQueries: [{ query: LabelsDocument }],
      awaitRefetchQueries: true,
      errorPolicy: "all"
    });
    if (data) {
      return data.assignLabels;
    }
    if (errors) {
      throw errors[0];
    }
    return false;
  }

  async function removeLabels(objectId: string, ids: string[]): Promise<boolean> {
    const { data, errors } = await client.mutate<RemoveLabelsMutation, RemoveLabelsMutationVariables>({
      mutation: RemoveLabelsDocument,
      variables: { objectId, ids },
      refetchQueries: [{ query: LabelsDocument }],
      awaitRefetchQueries: true,
      errorPolicy: "all"
    });
    if (data) {
      return data.removeLabels;
    }
    if (errors) {
      throw errors[0];
    }
    return false;
  }

  return {
    insertLabels,
    updateLabel,
    deleteLabel,
    assignLabels,
    removeLabels
  };
}

export function useLabelsSubscription(): void {
  const client = useApolloClient();
  const { store: { session: { isLoggedIn } } } = useStore();
  const { data: subData, error: subError } = useLabelUpdateSubscription({ skip: !isLoggedIn });

  useEffect(() => {
    subError && console.error("Labels update subscription error:", subError);

    if (!subData) {
      return;
    }

    const { type, itemIds } = subData.labelUpdate;
    switch (type) {
      case LabelOrHierarchyUpdateType.Insert:
        for (const id of itemIds) {
          if (!isLabelInCache(id)) {
            client.query<LabelQuery, LabelQueryVariables>({
              query: LabelDocument,
              variables: { id },
              fetchPolicy: "no-cache"
            })
              .then(({ data }) => {
                if (data.label && !isLabelInCache(id)) {
                  const cache = readQuery<LabelsQuery, LabelsQueryVariables>(client, { query: LabelsDocument });
                  if (cache) {
                    client.writeQuery<LabelsQuery>({
                      query: LabelsDocument,
                      data: { labels: [...cache.labels, data.label] }
                    });
                  }
                }
              })
              .catch(e => console.error("Label query error:", e));
          }
        }
        break;

      case LabelOrHierarchyUpdateType.Update:
        for (const id of itemIds) {
          if (isLabelInCache(id)) {
            client.query<LabelQuery, LabelQueryVariables>({
              query: LabelDocument,
              variables: { id },
              fetchPolicy: "no-cache"
            })
              .then(({ data }) => {
                if (data.label) {
                  const label = readLabelFromCache(id);
                  if (label && JSON.stringify(label) !== JSON.stringify(data.label)) {
                    client.writeFragment({
                      id: defaultDataIdFromObject({ __typename: "Label", id }) ?? id,
                      fragment: LabelFieldsFragmentDoc,
                      fragmentName: "LabelFields",
                      data: data.label
                    });
                  }
                }
              })
              .catch(e => console.error("Label query error:", e));
          }
        }
        break;

      case LabelOrHierarchyUpdateType.Delete:
        const data = readQuery<LabelsQuery, LabelsQueryVariables>(client, { query: LabelsDocument });
        if (data && data.labels.some(label => itemIds.includes(label.id))) {
          client.writeQuery<LabelsQuery>({
            query: LabelsDocument,
            data: { labels: [...data.labels.filter(label => !itemIds.includes(label.id))] }
          });
        }
        break;
    }
  }, [subData, subError]);

  function isLabelInCache(id: string): boolean {
    return !!readLabelFromCache(id);
  }

  function readLabelFromCache(id: string): Label | null {
    try {
      return client.readFragment<LabelFieldsFragment, LabelQueryVariables>({
        id: defaultDataIdFromObject({ __typename: "Label", id }) ?? id,
        fragment: LabelFieldsFragmentDoc,
        fragmentName: "LabelFields"
      });
    }
    catch {
      return null;
    }
  }
}
