import { useEffect } from "react";
import { ApolloClient, defaultDataIdFromObject, useApolloClient } from "@apollo/client";
import {
  DeleteHierarchyDocument,
  DeleteHierarchyMutation,
  DeleteHierarchyMutationVariables,
  HierarchiesDocument,
  HierarchiesQuery,
  HierarchiesQueryVariables,
  HierarchyDocument,
  HierarchyFieldsFragment,
  HierarchyFieldsFragmentDoc,
  HierarchyInput,
  HierarchyQuery,
  HierarchyQueryVariables,
  InsertHierarchyDocument,
  InsertHierarchyMutation,
  InsertHierarchyMutationVariables,
  LabelOrHierarchyUpdateType,
  UpdateHierarchyDocument,
  UpdateHierarchyMutation,
  UpdateHierarchyMutationVariables,
  useHierarchyUpdateSubscription
} from "@generated/graphql";
import { useStore } from "@core/store";
import { readQuery } from "@core/utils";

type HierarchyActions = {
  insertHierarchy: (hierarchy: HierarchyInput) => Promise<string>;
  updateHierarchy: (hierarchy: HierarchyInput) => Promise<boolean>;
  deleteHierarchy: (id: string) => Promise<boolean>;
};

export function useHierarchyActions(): HierarchyActions {
  const client = useApolloClient();

  async function insertHierarchy(hierarchy: HierarchyInput): Promise<string> {
    const { data, errors } = await client.mutate<InsertHierarchyMutation, InsertHierarchyMutationVariables>({
      mutation: InsertHierarchyDocument,
      variables: { hierarchy },
      refetchQueries: [{ query: HierarchiesDocument }],
      awaitRefetchQueries: true,
      errorPolicy: "all"
    });
    if (data) {
      return data.insertHierarchy.id;
    }
    if (errors) {
      throw errors[0];
    }
    return "";
  }

  async function updateHierarchy(hierarchy: HierarchyInput): Promise<boolean> {
    const { data, errors } = await client.mutate<UpdateHierarchyMutation, UpdateHierarchyMutationVariables>({
      mutation: UpdateHierarchyDocument,
      variables: { hierarchy },
      refetchQueries: [{ query: HierarchiesDocument }],
      awaitRefetchQueries: true,
      errorPolicy: "all"
    });
    if (data) {
      return data.updateHierarchy;
    }
    if (errors) {
      throw errors[0];
    }
    return false;
  }

  async function deleteHierarchy(id: string): Promise<boolean> {
    const { data, errors } = await client.mutate<DeleteHierarchyMutation, DeleteHierarchyMutationVariables>({
      mutation: DeleteHierarchyDocument,
      variables: { id },
      refetchQueries: [{ query: HierarchiesDocument }],
      awaitRefetchQueries: true,
      errorPolicy: "all"
    });
    if (data) {
      return data.deleteHierarchy;
    }
    if (errors) {
      throw errors[0];
    }
    return false;
  }

  return {
    insertHierarchy,
    updateHierarchy,
    deleteHierarchy
  };
}

export async function getHierarchies(client: ApolloClient<any>): Promise<HierarchyFieldsFragment[]> {
  const { data } = await client.query<HierarchiesQuery, HierarchiesQueryVariables>({ query: HierarchiesDocument });
  const hierarchies = [...data.hierarchies];
  hierarchies.sort((a, b) => a.name.localeCompare(b.name, undefined, { sensitivity: "base" }));
  return hierarchies;
}

export function useHierarchiesSubscription(): void {
  const client = useApolloClient();
  const { store: { session: { isLoggedIn } } } = useStore();
  const { data: subData, error: subError } = useHierarchyUpdateSubscription({ skip: !isLoggedIn });

  useEffect(() => {
    subError && console.error("Hierarchies update subscription error:", subError);

    if (!subData) {
      return;
    }

    const { type, itemIds } = subData.hierarchyUpdate;
    switch (type) {
      case LabelOrHierarchyUpdateType.Insert:
        for (const id of itemIds) {
          if (!isHierarchyInCache(id)) {
            client.query<HierarchyQuery, HierarchyQueryVariables>({
              query: HierarchyDocument,
              variables: { id },
              fetchPolicy: "no-cache"
            })
              .then(({ data }) => {
                if (data.hierarchy && !isHierarchyInCache(id)) {
                  const cache = readQuery<HierarchiesQuery, HierarchiesQueryVariables>(client, { query: HierarchiesDocument });
                  if (cache) {
                    client.writeQuery<HierarchiesQuery>({
                      query: HierarchiesDocument,
                      data: { hierarchies: [...cache.hierarchies, data.hierarchy] }
                    });
                  }
                }
              })
              .catch(e => console.error("Hierarchy query error:", e));
          }
        }
        break;

      case LabelOrHierarchyUpdateType.Update:
        for (const id of itemIds) {
          if (isHierarchyInCache(id)) {
            client.query<HierarchyQuery, HierarchyQueryVariables>({
              query: HierarchyDocument,
              variables: { id },
              fetchPolicy: "no-cache"
            })
              .then(({ data }) => {
                if (data.hierarchy) {
                  const hierarchy = readHierarchyFromCache(id);
                  if (hierarchy && JSON.stringify(hierarchy) !== JSON.stringify(data.hierarchy)) {
                    client.writeFragment({
                      id: defaultDataIdFromObject({ __typename: "Hierarchy", id }) ?? id,
                      fragment: HierarchyFieldsFragmentDoc,
                      fragmentName: "HierarchyFields",
                      data: data.hierarchy
                    });
                  }
                }
              })
              .catch(e => console.error("Hierarchy query error:", e));
          }
        }
        break;

      case LabelOrHierarchyUpdateType.Delete:
        const data = readQuery<HierarchiesQuery, HierarchiesQueryVariables>(client, { query: HierarchiesDocument });
        if (data && data.hierarchies.some(hier => itemIds.includes(hier.id))) {
          client.writeQuery<HierarchiesQuery>({
            query: HierarchiesDocument,
            data: { hierarchies: [...data.hierarchies.filter(hier => !itemIds.includes(hier.id))] }
          });
        }
        break;
    }
  }, [subData, subError]);

  function isHierarchyInCache(id: string): boolean {
    return !!readHierarchyFromCache(id);
  }

  function readHierarchyFromCache(id: string): HierarchyFieldsFragment | null {
    try {
      return client.readFragment<HierarchyFieldsFragment, HierarchyQueryVariables>({
        id: defaultDataIdFromObject({ __typename: "Hierarchy", id }) ?? id,
        fragment: HierarchyFieldsFragmentDoc,
        fragmentName: "HierarchyFields"
      });
    }
    catch {
      return null;
    }
  }
}
