import { useState, useEffect, useRef, useCallback } from "react";
import { useStore } from "@core/store";
import { Widget, Widgets, WidgetPropGroup } from "components/Widgets";
import { CreatingView, CreatingViewInput } from "@generated/graphql";
import { queryToInput, FilterEvent } from "utils";

export type WidgetPropGroupsParams = {
  currentViewId: string;
  onAddRemoveGroups: (addGroups?: WidgetPropGroup[], removeGroups?: WidgetPropGroup[]) => void;
  filterEvent?: FilterEvent;
};

export function useWidgetPropGroups({ currentViewId, onAddRemoveGroups, filterEvent }: WidgetPropGroupsParams): void {
  const { store: { workspace: { creatingViews } }, setStore } = useStore();
  const [widget, setWidget] = useState<Widget | undefined>();
  const widgetParamsRef = useRef<object | undefined>();
  const groupsRef = useRef<WidgetPropGroup[]>([]);

  useEffect(() => {
    return function cleanup() {
      if (onAddRemoveGroups) {
        onAddRemoveGroups(undefined, groupsRef.current);
      }
    };
  }, []);

  useEffect(() => {
    selectWidget();
  }, [currentViewId, creatingViews]);

  useEffect(() => {
    const removeGroups = groupsRef.current.length > 0 ? groupsRef.current : undefined;
    const widgetGroups = widget ? widget.getPropGroups() : [];
    groupsRef.current = widgetGroups;
    let addGroups: WidgetPropGroup[] | undefined;
    if (widget && widgetGroups.length > 0) {
      addGroups = widgetGroups.map(group => {
        const widgetParams = group.acceptsFilter ? { ...widgetParamsRef.current, filterEvent } : widgetParamsRef.current;
        return { ...group, content: widget.renderPropGroupEditor(group, onParamsSelected, widgetParams) };
      });
    }
    if (onAddRemoveGroups) {
      onAddRemoveGroups(addGroups, removeGroups);
    }
  }, [widget]);

  const selectWidget = useCallback(() => {
    if (!creatingViews) {
      setWidget(undefined);
      widgetParamsRef.current = undefined;
      return;
    }

    const view = creatingViews.find(v => v.viewId + "_selectWidgets" === currentViewId);
    if (!view || !view.selectedWidget || !view.widgets) {
      setWidget(undefined);
      widgetParamsRef.current = undefined;
      return;
    }

    const { widgets, selectedWidget: { widgetId, index, propsJSON } } = view;
    let w = Widgets.find(w => w.id === widgetId);
    if (!w) {
      setWidget(undefined);
      widgetParamsRef.current = undefined;
      return;
    }

    w = w.createInstance(currentViewId, index, widgets);
    setWidget(w);
    widgetParamsRef.current = propsJSON ? JSON.parse(propsJSON) : undefined;
  }, [creatingViews, currentViewId]);

  const onParamsSelected = useCallback((props: Record<string, any>) => {
    if (!creatingViews) {
      return;
    }

    const newViews = queryToInput<CreatingView[], CreatingViewInput[]>(creatingViews);
    const viewIndex = newViews.findIndex(v => v.viewId + "_selectWidgets" === currentViewId);
    if (viewIndex < 0) {
      return;
    }

    const view = newViews[viewIndex];
    if (!view || !view.selectedWidget || !view.widgets) {
      return;
    }

    const { widgets, selectedWidget: { index } } = view;
    const newWidgets = Array.from(widgets);
    let newWidgetProps = newWidgets[index].propsJSON;
    if (newWidgetProps) {
      if (props.objects) {
        delete props.objects;
      }
      newWidgetProps = { ...JSON.parse(newWidgetProps), ...props};
    }

    newWidgets[index] = { ...newWidgets[index], index, propsJSON: JSON.stringify(newWidgetProps || props) };
    newViews[viewIndex] = { ...newViews[viewIndex], widgets: newWidgets, selectedWidget: newWidgets[index] };
    setStore({ workspace: { creatingViews: newViews } });
  }, [creatingViews, currentViewId]);
}
