import React from "react";
import { Popup, Icon } from "semantic-ui-react";
import { Widget, WidgetParamsCallback, WidgetPropGroup } from "./Widget";
import { WidgetProps } from "components/Widgets";
import CameraList, { ShowHideFilterEvent } from "components/CameraList";
import { DragObject, DragObjectType, isDragObject, isObject } from "@core/types";
import { isCameraCellProps } from "components/CellContainer/CameraCell";
import CameraCellStub from "components/CellContainer/CameraCellStub";
import {__} from "@solid/libs/i18n";

export class CameraWidget extends Widget {
  private showHideFilterEvent = new ShowHideFilterEvent();

  protected override createNew(): Widget {
    return new CameraWidget(this.id, this.name, this.title, this.module, this.icon, this.options);
  }

  override renderStub(): React.ComponentType<WidgetProps> {
    return Stub;
  }

  override getPropGroups(): WidgetPropGroup[] {
    const groups = super.getPropGroups();
    groups.push({
      id: "Devices",
      name: __("Device List"),
      key: this.id + "_Devices",
      scrollable: true,
      extraTitleContent: (
        <Popup
          trigger={
            <Icon className="CameraList-TitleIcon" name="search" onClick={(e: React.MouseEvent) => {
              e.stopPropagation();
              this.showHideFilterEvent.publish({});
            }}/>
          }
          content={__("Show/Hide Filter")}
          position="top right"
        />
      )
    });
    return groups;
  }

  override renderPropGroupEditor(group: WidgetPropGroup, callback: WidgetParamsCallback, params?: object): JSX.Element | undefined {
    function onSelected(id: string, name: string): void {
      const camera: DragObject = { type: DragObjectType.Camera, obj: { "data-id": id, name } };
      callback({ object: camera });
    }

    if (group.id === "Devices") {
      const selectedId = isCameraCellProps(params) && isDragObject(params.object) && isObject(params.object.obj) ?
        params.object.obj["data-id"] : undefined;
      return (
        <CameraList
          key={`${this.viewId}_${this.index}`}
          selectable
          onSelected={onSelected}
          selectedId={selectedId}
          allFilters
          allowLabelHierarchyEdit
          openMultipleBranches
          showHideFilterEvent={this.showHideFilterEvent}
        />
      );
    }

    return super.renderPropGroupEditor(group, callback, params);
  }
}

const Stub = (props: WidgetProps) => {
  return <CameraCellStub {...props}/>;
};
