import {useState, useEffect, RefObject, useRef} from "react";
import { RTabPanel, CurrentChangedArgs, RWidgetLoader, ModulesUpdateArgs } from "components/Layout";
import { ModuleInfo } from "@core/types";
import {isElectron} from "@solid/libs/utils";

export type CurrentViewOptions = {
  views?: ModuleInfo[];
  tabPanelRef?: RefObject<RTabPanel>;
  viewLoaderRef?: RefObject<RWidgetLoader>;
};

export type CurrentViewResult = {
  currentViewId: string;
  currentView?: ModuleInfo;
};

export function useCurrentView({ views, tabPanelRef, viewLoaderRef }: CurrentViewOptions): CurrentViewResult {
  const [currentViewId, setCurrentViewId] = useState("");
  const [currentView, setCurrentView] = useState<ModuleInfo | undefined>();
  const isMountedRef = useRef<boolean>(false);

  useEffect(() => {
    isMountedRef.current = true;

    return () => {
      isMountedRef.current = false;
    };
  }, []);

  // TODO: useEffect MUST be used with second argument only
  useEffect(() => {
    const id = `${Date.now()}.${Math.random()}`;
    const tabPanel = tabPanelRef?.current;
    const loader = viewLoaderRef?.current;

    if (tabPanel) {
      tabPanel.currentChanged.subscribe(id, onCurrentViewChanged);
    }
    if (loader) {
      loader.modulesUpdated.subscribe(id, onModulesUpdated);
    }

    return function cleanup() {
      if (tabPanel) {
        tabPanel.currentChanged.unsubscribe(id);
      }
      if (loader) {
        loader.modulesUpdated.unsubscribe(id);
      }
    };
  });

  function onCurrentViewChanged({ id }: CurrentChangedArgs): void {
    setCurrentViewId(id);
    if (views) {
      setCurrentView(views.find(v => v.id === id));
    }
  }

  function onModulesUpdated({ modules }: ModulesUpdateArgs): void {
    if (!isMountedRef.current) {
      return;
    }
    if (tabPanelRef?.current) {
      return;
    }

    let views = modules;
    if (isElectron()) {
      views = views.filter(v => !v.inWindow);
    }
    if (views.length > 0) {
      setCurrentViewId(views[0].id);
      setCurrentView(views[0]);
    } else {
      setCurrentViewId("");
      setCurrentView(undefined);
    }
  }

  return { currentViewId, currentView };
}
