import React, { useState, useEffect, useRef } from "react";
import { Accordion, AccordionPanelProps, AccordionTitleProps, Icon, Input, Popup } from "semantic-ui-react";
import { ViewManagerResult } from "./viewManager";
import { FilterEvent } from "utils";
import { Utils } from "@solid/libs/utils";
import {__} from "@solid/libs/i18n";

export type PanelProps = AccordionPanelProps & {
  key: string;
  label: string;
  panelContent: JSX.Element;
  contentClass?: string;
  extraTitleContent?: JSX.Element;
  showFilter?: boolean;
};

export type AccordionParams = {
  panels: PanelProps[];
  activeIndices: Set<number>;
  recreatePanels?: (vm: ViewManagerResult, filter: Map<string, string>, panels: PanelProps[]) => PanelProps[];
};

export type AccordionResult = {
  panels: PanelProps[];
  activeIndices: Set<number>;
  filter: Map<string, string>;
  filterEvent: FilterEvent;
  setPanels: (callback: (panels: PanelProps[]) => PanelProps[]) => void;
  setActiveIndices: (callback: (indices: Set<number>) => Set<number>) => void;
  toggleActive: (index: number) => void;
  onTitleClick: (e: React.MouseEvent<HTMLDivElement>, props: AccordionTitleProps) => void;
  recreatePanels: (vm: ViewManagerResult, filter: Map<string, string>) => void;
};

export function useAccordion(params: AccordionParams): AccordionResult {
  const [panels, setPanels] = useState<PanelProps[]>([]);
  const [activeIndices, setActiveIndices] = useState(params.activeIndices);
  const [filter, setFilter] = useState(new Map<string, string>());
  const filterEventRef = useRef(new FilterEvent());
  const isMountedRef = useRef<boolean>(false);

  const recreatePanelsRef = useRef(Utils.throttle((vm: ViewManagerResult, filter: Map<string, string>) => {
    if (!isMountedRef.current) {
      return;
    }

    const func = params.recreatePanels;
    if (func) {
      updatePanels(panels => func(vm, filter, panels));
    }
  }, 250));

  useEffect(() => {
    isMountedRef.current = true;

    updatePanels(panels => params.panels);

    return () => {
      isMountedRef.current = false;
    };
  }, []);

  function getTitle(component: React.ElementType<AccordionTitleProps>, panels: PanelProps[], props: AccordionTitleProps) {
    const { index, active } = props;
    const i = typeof index === "string" ? parseInt(index) : (typeof index === "number" ? index : -1);
    const panelProps = i >= 0 ? panels[i] : undefined;
    return (
      <Accordion.Title {...props}>
        <Icon name={active ? "caret down" : "caret right"}/><span>{panelProps?.label ?? ""}</span>

        {!!panelProps?.extraTitleContent && panelProps.extraTitleContent}

        {!!panelProps?.showFilter &&
          <TitleFilter value={filter.get(panelProps.key)} onChange={e => onFilterChange(panelProps.key, e)}/>}
      </Accordion.Title>
    );
  }

  function onFilterChange(key: string, e: React.ChangeEvent<HTMLInputElement>): void {
    const value = e.currentTarget.value;
    filterEventRef.current.publish({ filter: value, key });
    setFilter(filter => {
      const newFilter = new Map<string, string>(filter);
      newFilter.set(key, value);
      return newFilter;
    });
  }

  function updatePanels(callback: (panels: PanelProps[]) => PanelProps[]): void {
    setPanels(panels => {
      const newPanels = callback(panels);
      return newPanels.map(p => getPanelProps(newPanels, p));
    });
  }

  function getPanelProps(panels: PanelProps[], props: PanelProps): PanelProps {
    return {
      ...props,
      title: {
        children: (component: React.ElementType<AccordionTitleProps>, props: AccordionTitleProps) => getTitle(component, panels, props),
      },
      content: {
        content: props.panelContent,
        className: props.contentClass
      }
    };
  }

  function toggleActive(index: number): void {
    setActiveIndices(indices => {
      const newIndices = new Set<number>(indices);
      if (newIndices.has(index)) {
        newIndices.delete(index);
      }
      else {
        newIndices.add(index);
      }
      return newIndices;
    });
  }

  function onTitleClick(e: React.MouseEvent<HTMLDivElement>, { index }: AccordionTitleProps) {
    if (index !== undefined) {
      const i = typeof index === "string" ? parseInt(index) : index;
      toggleActive(i);
    }
  }

  return {
    panels,
    activeIndices,
    filter,
    filterEvent: filterEventRef.current,
    setPanels: updatePanels,
    setActiveIndices,
    toggleActive,
    onTitleClick,
    recreatePanels: recreatePanelsRef.current
  };
}

type TitleFilterProps = {
  value?: string;
  onChange?: (e: React.ChangeEvent<HTMLInputElement>) => void;
};

const TitleFilter = ({ onChange, ...props }: TitleFilterProps) => {
  const [value, setValue] = useState("");
  const [visible, setVisible] = useState(false);

  useEffect(() => {
    setValue(props.value ?? "");
  }, [props.value]);

  return (
    <>
      <Popup position="bottom right" content={visible ? __("Hide Filter") : __("Show Filter")} trigger={
        <Icon
          name="search"
          className="ViewActivity-TitleButton"
          onClick={(e: React.MouseEvent) => {
            e.stopPropagation();
            setVisible(value => !value);
          }}
        />
      }
      />

      {visible &&
      <Input
        icon="search"
        placeholder={__("Filter...")}
        value={value}
        onChange={e => {
          setValue(e.currentTarget.value);
          onChange && onChange(e);
        }}
        onClick={(e: React.MouseEvent) => e.stopPropagation()}
      />}
    </>
  );
};
