import React, { useEffect } from "react";
import { Accordion, Popup, Icon } from "semantic-ui-react";
import { useNavigate } from "react-router-dom";
import { FontAwesomeIcon } from "@fortawesome/react-fontawesome";
import classNames from "classnames";
import { ViewManagerResult } from "components/View";
import { useAccordion, PanelProps } from "../accordion";
import UserViewList from "components/View/Manager/UserViewList";
import RecentViewList from "components/View/Manager/RecentViewList";
import SharedViewList from "components/View/Manager/SharedViewList";
import {__} from "@solid/libs/i18n";
import { useStore } from "@core/store";

import "../style.css";

type ViewActivityProps = {
  vm: ViewManagerResult;
};

const ViewActivity = ({ vm }: ViewActivityProps) => {
  const { store: { session: { isAdmin } } } = useStore();
  const basePanels = getPanels(vm, new Map<string, string>());
  const { panels, activeIndices, filter, onTitleClick, recreatePanels } = useAccordion({
    panels: basePanels,
    activeIndices: new Set<number>(basePanels.map((p, index) => index)),
    recreatePanels: (vm, filter, panels) => getPanels(vm, filter)
  });
  const navigate = useNavigate();

  useEffect(() => {
    recreatePanels(vm, filter);
  }, [vm, filter]);

  function getPanels(vm: ViewManagerResult, filter: Map<string, string>): PanelProps[] {
    return [
      { key: "UserViews", label: __("My Views"), panelContent: <UserViewList vm={vm} filter={filter.get("UserViews")}/>, showFilter: true,
        extraTitleContent: getUserViewsExtraTitleContent(vm) },
      { key: "RecentViews", label: __("Recent Views"), panelContent: <RecentViewList vm={vm}/>,
        extraTitleContent: getRecentViewsExtraTitleContent(vm) },
      { key: "SharedViews", label: __("Shared Views"), panelContent: <SharedViewList vm={vm} filter={filter.get("SharedViews")}/>, showFilter: true },
    ];
  }

  function getUserViewsExtraTitleContent(vm: ViewManagerResult) {
    if (isAdmin) {
      return (
        <Popup position="bottom right" content={__("Add New View")} disabled={vm.editMode} trigger={
          <Icon
            name="plus"
            className={classNames("ViewActivity-TitleButton", { "disabled": vm.editMode })}
            onClick={(e: React.MouseEvent) => {
              if (!vm.editMode) {
                e.stopPropagation();
                vm.createView();
              }
              navigate("/view/new");
            }}/>
        }/>
      );
    }
    return undefined;
  }

  function getRecentViewsExtraTitleContent(vm: ViewManagerResult) {
    return (
      <Popup position="bottom right" content={__("Clear Recent Views List")} trigger={
        <FontAwesomeIcon icon="broom" className="ViewActivity-TitleButton" onClick={(e) => {
          e.stopPropagation();
          vm.config.clearRecentViews();
        }}/>
      }/>
    );
  }

  return (
    <div className="ViewActivity">
      <Accordion panels={panels} exclusive={false} activeIndex={Array.from(activeIndices.keys())}
        onTitleClick={(e, props) => onTitleClick(e as React.MouseEvent<HTMLDivElement>, props)}
        styled fluid className="ViewActivity-Accordion"/>
    </div>
  );
};

export default ViewActivity;
