import React, { useEffect, useMemo, useRef, useState } from "react";
import { FontAwesomeIcon } from "@fortawesome/react-fontawesome";
import { Button, Header, Icon, Modal, Progress } from "semantic-ui-react";
import { useIdentityAttributeQuery } from "@generated/graphql";
import { makeVar } from "@apollo/client";
import { withStore, WithStoreProps } from "@core/store";
import { RWidgetLoader } from "components/Layout";
import { useCurrentView } from "components/View";
import { CommonWidgetEvent, ProgressAction, ProgressEventArgs, WidgetEvent, WidgetEventArgs } from "components/Widgets";
import { __ } from "@solid/libs";
import VDMTrigger from "components/VideoDeliveryManager/VDMTrigger";
// import EATrigger from "components/EventAcknowledgment/EATrigger";

import "./style.css";

type StatusBarProps = WithStoreProps & {
  loaderRef?: React.RefObject<RWidgetLoader>;
};

export const StatusBarOpenComponentId = makeVar<string | undefined>(undefined);

const StatusBar = ({ store: { session: { info } }, loaderRef }: StatusBarProps) => {
  const { currentViewId } = useCurrentView({ viewLoaderRef: loaderRef });
  const { data } = useIdentityAttributeQuery();
  const subRef = useRef("");
  const eventRef = useRef<WidgetEvent | undefined>();
  const [isProgressVisible, setIsProgressVisible] = useState(false);
  const [progressPercent, setProgressPercent] = useState(0);
  const [progressValue, setProgressValue] = useState<number | string | undefined>();
  const [progressLabel, setProgressLabel] = useState<string | undefined>();
  const [progressCancelable, setIsProgressCancelable] = useState(false);
  const [isModal, setIsModal] = useState(false);
  const identityAttribute: string | undefined = useMemo(() => data?.identityAttribute?.contactUs, [data]);

  useEffect(() => {
    return function cleanup() {
      eventRef.current?.unsubscribe(subRef.current);
    };
  }, []);

  useEffect(() => {
    eventRef.current?.unsubscribe(subRef.current);
    eventRef.current = undefined;
    if (currentViewId) {
      eventRef.current = loaderRef?.current?.getWidgetEvent(currentViewId);
      subRef.current = `${Date.now()}.${Math.random()}`;
      eventRef.current?.subscribe(subRef.current, onWidgetEvent);
    }
  }, [currentViewId]);

  const onWidgetEvent = (args: WidgetEventArgs): void => {
    if (args.event !== CommonWidgetEvent.Progress) {
      return;
    }
    const { action, percent, value, label, cancelable } = args.args as ProgressEventArgs;
    setIsProgressVisible(action === ProgressAction.Start || action === ProgressAction.Progress);
    setProgressPercent(percent);
    setProgressValue(value);
    setProgressLabel(label);
    setIsProgressCancelable(!!cancelable);
  };

  const cancelProgress = () => {
    eventRef.current?.publish({ event: CommonWidgetEvent.CancelProgress, args: {} });
  };

  return (
    <>
      {isModal && identityAttribute &&
      <Modal
        closeIcon
        open={isModal}
        onClose={() => setIsModal(false)}
        onOpen={() => setIsModal(true)}
        className="Contacts-Modal"
    >
        <Header icon="info" content={__("Contact us")} />
        <Modal.Content className="Contacts">
          {identityAttribute}
        </Modal.Content>
      </Modal>
    }
      <div className="StatusBar">
        <div className="StatusBar-left">
          <div className="StatusBar-UserRoleName">
            <FontAwesomeIcon icon="user"/>
            <span>{info?.user?.name}</span>
          </div>
          { isProgressVisible &&
          <div className="StatusBar-ProgressContainer">
            { !!progressLabel && <div className="StatusBar-ProgressLabel">{progressLabel}</div> }
            <div className="StatusBar-ProgressBar">
              <Progress color="green" active percent={progressPercent} className="StatusBar-Progress"/>
              { progressValue !== undefined && <div className="StatusBar-ProgressValue">{progressValue}</div> }
            </div>
            { progressCancelable && <Button negative onClick={cancelProgress}><Icon name="stop"/>{__("Stop")}</Button> }
          </div> }
        </div>

        <div className="StatusBar-right">
          <div className="StatusBar-EventAcknowledgment">
            {/* <EATrigger /> */}
          </div>
          <div className="StatusBar-VideoDeliveryManager">
            <VDMTrigger/>
          </div>
          <div className="StatusBar-Space">
            <FontAwesomeIcon icon="info" onClick={() => setIsModal(true)}/>
            <span onClick={() => setIsModal(true)}>{__("Contact Us")}</span>
          </div>
        </div>
      </div>
    </>
  );
};

export default withStore(StatusBar);
