import React, { RefObject, useCallback, useEffect, useState } from "react";
import { FontAwesomeIcon } from "@fortawesome/react-fontawesome";
import { Activity, ActivityItem, ActivityItemProps } from "components/Activity";
import { isEditView, useViewManager, ViewManagerParams } from "components/View";
import ViewActivity from "components/View/ViewActivity";
import EditActivity from "components/View/EditActivity";
import AdminActivity from "components/Admin/AdminActivity";
import SettingsActivity from "components/SettingsActivity";
import { RWidget } from "components/Layout";
import { ModuleInfo } from "@core/types";
import { useStore } from "@core/store";
import CloseConfirm from "components/View/Manager/CloseConfirm";
import { __, isElectron, Log } from "@solid/libs";
import useLogout from "@core/useLogout";
import { MessageId } from "electron/types";
import { useConfig } from "@core/store/actions";

type SidePanelProps = ViewManagerParams & {
  sidePanelWidgetRef?: RefObject<RWidget>;
};

enum ActivityIndex {
  View,
}

const SidePanel = (props: SidePanelProps) => {
  const vm = useViewManager({ ...props, onViewSelected });
  const { currentView, editMode, cancelEdit, config: { updateError, startView } } = vm;
  const [currentActivity, setCurrentActivity] = useState(0);
  const [nextActivity, setNextActivity] = useState(0);
  const [isSideBarVisible, setIsSideBarVisible] = useState(true);
  const [closeView, setCloseView] = useState<ModuleInfo | undefined>();
  const { store: {session: {isAdmin}} } = useStore();
  const { logout } = useLogout();
  const config = useConfig();
  const [isSave, allowSave] = useState(config.settings.saveCurrentSession);

  useEffect(() => {
    allowSave(config.settings.saveCurrentSession);
  }, [config.settings.saveCurrentSession]);

  useEffect(() => {
    if (isElectron()) {
      const isSave = typeof config.settings.saveCurrentSession === "undefined" ? true : config.settings.saveCurrentSession;
      window.ipcRenderer?.send(MessageId.SaveCurrentSessionCookie, isSave);
    }
  }, [isSave]);

  useEffect(() => {
    const id = `${Date.now()}.${Math.random()}`;
    const event = props.loadFromDbEvent;
    if (event) {
      event.subscribe(id, ({ views, allViews, nextConfig: { settings } }) => {
        if (settings === undefined) {
          config.setConfig({ settings: { openViewListOnStart: true } });
          setIsSideBarVisible(true);
        } else
        if (isElectron() && !settings?.openViewListOnStart) {
          const vw = views.filter(v => !isElectron() || (!v.inWindow && !allViews?.some(av => av.id === v.id && av.inWindow)));
          if (vw && vw.length > 0 && vw[0].id !== startView.id) {
            setIsSideBarVisible(false);
          }
        } else
        if (!settings?.openViewListOnStart) {
          setIsSideBarVisible(false);
        }
      });
    }

    return function cleanup() {
      event?.unsubscribe(id);
    };
  }, []);

  useEffect(() => {
    setCurrentActivity(index => editMode ? ActivityIndex.View : index);
    if (editMode) {
      setIsSideBarVisible(true);
    }
  }, [editMode]);

  useEffect(() => {
    updateError && Log.error(`${__("Configuration update error")}: ${updateError.message}`, 3000);
  }, [updateError]);

  const onChange = useCallback((index: number) => {
    if (index !== ActivityIndex.View && editMode) {
      if (currentView?.hasUnsavedChanges) {
        setCloseView(currentView);
        setNextActivity(index);
        return;
      }
      cancelEdit();
    }
    setCurrentActivity(index);
  }, [vm]);

  function onViewSelected(id: string, index: number): void {
    if (!isEditView(id)) {
      //setIsSideBarVisible(false);
    }
  }

  const onCloseConfirm = useCallback((id: string) => {
    cancelEdit();
    setCurrentActivity(nextActivity);
    setCloseView(undefined);
  }, [vm, nextActivity]);

  let items: React.ReactElement<ActivityItemProps, typeof ActivityItem>[] = [
    <ActivityItem key="View" name={__("View")} icon={<FontAwesomeIcon icon="th"/>}>
      {!editMode ? <ViewActivity vm={vm}/> : <EditActivity vm={vm}/>}
    </ActivityItem>,
  ];

  items = isAdmin ?
    items.concat(vm.config.adminViews
      .filter(item => !item.hidden)
      .map(item =>
        <ActivityItem key={item.id} name={item.name} icon={<FontAwesomeIcon icon={item.icon}/>}>
          <AdminActivity itemId={item.id} vm={vm}/>
        </ActivityItem>)) :
    items;

  items.push(
    <ActivityItem key="Separator" name="" icon={<></>} />
  );

  items.push(
    <ActivityItem key="Solid-Settings" name="" icon={<FontAwesomeIcon icon="cog"/>}>
      <SettingsActivity/>
    </ActivityItem>
  );

  const aboutItem = isAdmin ? vm.config.adminViews.filter(item => item.id === "About")[0] : undefined;

  if (aboutItem) {
    items.push(
      <ActivityItem key={aboutItem.id} name="" icon={<FontAwesomeIcon icon={aboutItem.icon}/>}>
        <AdminActivity itemId={aboutItem.id} vm={vm}/>
      </ActivityItem>);
  }

  items.push(
    <ActivityItem key="Logout" name="" holdSideBar icon={<FontAwesomeIcon icon="sign-out-alt" onClick={() => logout()}/>} />
  );

  return (
    <>
      <Activity currentIndex={currentActivity} onChange={onChange}
        isSideBarVisible={isSideBarVisible} onSideBarShow={visible => setIsSideBarVisible(visible)}
        onResize={({width}) => {
          // correct size of neighbour widgets
          props?.sidePanelWidgetRef?.current?.widget?.hide();
          props?.sidePanelWidgetRef?.current?.widget?.show();
        }}
      >
        {items}
      </Activity>

      {!!closeView && <CloseConfirm view={closeView} onConfirm={onCloseConfirm} onCancel={() => setCloseView(undefined)}/>}
    </>
  );
};

export default SidePanel;
