import React, { useCallback, useEffect, useState } from "react";
import { Button, Icon, Input } from "semantic-ui-react";

type PasswordFieldProps = {
  value?: string;
  disabled?: boolean;
  placeholder?: string;
  error?: boolean;
  required?: boolean;
  onChange?: (value?: string) => void;
  onBlur?: (value?: string) => void;
  onFocus?: (value?: string) => void;
};

const PasswordField = ({
  value,
  disabled = false,
  placeholder,
  error,
  required,
  onBlur,
  onChange,
  onFocus
}: PasswordFieldProps) => {
  const [showPassword, setShowPassword] = useState<boolean>(false);
  const [password, setPassword] = useState<string>(value ? value.toString() : "");

  useEffect(() => {
    value && setPassword(value);
  }, [value]);

  const updatePassword = useCallback((value: string) => {
    let newPassword = password;
    if (value.length === 0) {
      newPassword = "";
    }

    if (value.length !== 0 && password.length > value.length) {
      newPassword = password.substring(0, password.length - 1);
    }

    if (value.length !== 0 && password.length < value.length) {
      newPassword = password + value[value.length - 1];
    }

    setPassword(newPassword);
    onChange && onChange(newPassword);
  }, [password]);

  return (
    <Input
      disabled={disabled}
      required={required}
      error={error}
      autoComplete="off"
      labelPosition="right"
      placeholder={placeholder}
      value={showPassword ? password : getSecretValue(password)}
      label={
        <Button icon onClick={() => setShowPassword(!showPassword)}>
          <Icon name={showPassword ? "eye slash" : "eye"}/>
        </Button>}
      onBlur={onBlur}
      onFocus={onFocus}
      onInput={(e: React.FormEvent<HTMLInputElement>) => updatePassword(e.currentTarget.value)}
    />
  );
};

export default PasswordField;

function getSecretValue(value: string) {
  let secretValue = "";
  for (let i = 0; i < value.length; i++) {
    secretValue = secretValue.concat("\u{2022}");
  }
  return secretValue;
}
