import React, { useEffect, useRef, useState } from "react";
import { Checkbox, Dropdown } from "semantic-ui-react";
import classNames from "classnames";

import { __ } from "@solid/libs";

import "./style.css";

export type MixedValue = string | number;

type CheckboxOption = {
  label: string;
  value: MixedValue;
  checked?: boolean;
};

export type MultipleOption = {
  group?: CheckboxOption;
  options: CheckboxOption[];
};

type MultipleDropdownProps = {
  label?: string;
  placeholder?: string;
  disabled?: boolean;
  withAllOption?: boolean;
  grouped?: boolean;
  groups: MultipleOption[];
  onChange: (values?: MixedValue[]) => void;
};

const MultipleDropdown = ({ label, placeholder, disabled, groups, grouped = false, withAllOption = false, onChange }: MultipleDropdownProps) => {
  const [groupMap, setGroupMap] = useState<Map<string, CheckboxOption[]>>(new Map());
  const [groupOptionMap, setGroupOptionMap] = useState<Map<string, CheckboxOption>>(new Map());
  const [checkedValues, setCheckedValues] = useState<Set<MixedValue>>(new Set());

  const allValuesRef = useRef<MixedValue[]>([]);

  useEffect(() => {
    if (groups.length === 0) {
      setGroupMap(new Map());
      setGroupOptionMap(new Map());
      setCheckedValues(new Set());
      return;
    }

    const allValues: Set<MixedValue> = new Set();
    const checkedValues: Set<MixedValue> = new Set();
    const groupMap: Map<string, CheckboxOption[]> = new Map();
    const groupOptionMap: Map<string, CheckboxOption> = new Map();

    for (const [index, multipleOption] of groups.entries()) {
      const optionStub = { value: index, checked: false, label: "optionStub" };
      const multipleOptionKey = multipleOption.group?.value.toString() || String(index);
      const groupOptions = groupMap.get(multipleOptionKey ?? String(index));

      groupOptionMap.set(multipleOptionKey, multipleOption.group ?? optionStub);

      if (groupOptions) {
        groupOptions.push(...multipleOption.options);
        groupMap.set(multipleOptionKey, groupOptions);
      }
      else {
        groupMap.set(multipleOptionKey, multipleOption.options);
      }

      for (const option of multipleOption.options) {
        allValues.add(option.value);
        option.checked && checkedValues.add(option.value);
      }
    }

    allValuesRef.current = [...allValues.values()];
    setCheckedValues(checkedValues);
    setGroupMap(groupMap);
    setGroupOptionMap(groupOptionMap);
  }, [groups, grouped, withAllOption]);

  const selectAll = (select: boolean) => {
    if (select) {
      onChange(allValuesRef.current);
      setCheckedValues(new Set([...allValuesRef.current]));
    }
    else {
      onChange([]);
      setCheckedValues(new Set());
    }

    setGroupMap(prevMap => {
      const newMap = new Map(prevMap);
      for (const [group, options] of newMap) {
        newMap.set(group, getChangedOptionsTo(options, select));
      }
      return newMap;
    });

    setGroupOptionMap(prevMap => {
      const newMap = new Map(prevMap);
      for (const [group, options] of newMap) {
        newMap.set(group, getChangedOptionsTo([options], select)[0]);
      }
      return newMap;
    });
  };

  const selectGroup = (groupKey: string, select: boolean) => {
    const options = groupMap.get(groupKey);
    if (options) {
      const newCheckedValues = new Set(checkedValues);
      if (select) {
        const uncheckedValues = getUncheckedValuesFromOptions(options);
        for (const value of uncheckedValues) {
          newCheckedValues.add(value);
        }
      }
      else {
        const checkedValues = getCheckedValuesFromOptions(options);
        for (const value of checkedValues) {
          newCheckedValues.delete(value);
        }
      }
      setCheckedValues(newCheckedValues);
      onChange([...newCheckedValues]);
    }

    setGroupMap(prevMap => {
      const groupMap = new Map(prevMap);
      const groupOptions = groupMap.get(groupKey);
      if (groupOptions) {
        const newOptions = getChangedOptionsTo(groupOptions, select);
        groupMap.set(groupKey, newOptions);
      }
      return groupMap;
    });

    setGroupOptionMap(prevMap => {
      const groupOptionMap = new Map(prevMap);
      const groupOption = groupOptionMap.get(groupKey);
      if (groupOption) {
        const newGroupOption = {...groupOption, checked: select};
        groupOptionMap.set(groupKey, newGroupOption);
      }
      return groupOptionMap;
    });
  };

  const selectGroupItem = (groupKey: string, optionIndex: number, select: boolean) => {
    const itemValue = groupMap.get(groupKey)?.[optionIndex].value;
    if (itemValue) {
      const newCheckedValues = new Set(checkedValues);
      if (select) {
        newCheckedValues.add(itemValue);
      }
      else {
        newCheckedValues.delete(itemValue);
      }
      setCheckedValues(newCheckedValues);
      onChange([...newCheckedValues]);
    }

    setGroupMap(prevMap => {
      const groupMap = new Map(prevMap);
      const groupOptions = groupMap.get(groupKey);
      groupOptions?.splice(optionIndex, 1, {
        label: groupOptions?.[optionIndex].label,
        value: groupOptions?.[optionIndex].value,
        checked: select
      });
      groupOptions && groupMap.set(groupKey, groupOptions);
      return groupMap;
    });

    const groupOptions = groupMap.get(groupKey);
    if (groupOptions) {
      const prevCheckedCount = getCheckedValuesFromOptions(groupOptions).length;
      const newCheckedCount = select ? prevCheckedCount + 1 : prevCheckedCount - 1;
      setGroupOptionMap(prevMap => {
        const groupOptionMap = new Map(prevMap);
        const groupOption = groupOptionMap.get(groupKey);
        if (groupOption) {
          const newGroupOption = {...groupOption, checked: groupOptions.length === newCheckedCount};
          groupOptionMap.set(groupKey, newGroupOption);
        }
        return groupOptionMap;
      });
    }
  };

  return (
    <div className="MultipleDropdown-Wrapper">
      <label>{label}</label>
      <Dropdown item simple text={placeholder} className="MultipleDropdown" key={`MultipleDropdown_${label}`}>
        <Dropdown.Menu className="MultipleDropdown-Menu" key={`MultipleDropdown-Menu_${label}`}>
          { withAllOption &&
            <Checkbox
              key={`All_${label}`}
              label={__("All")}
              className="MultipleDropdown-All"
              checked={checkedValues.size === allValuesRef.current.length}
              onChange={(e, { checked }) => selectAll(!!checked)}
          /> }

          {[...groupMap.keys()].map((groupKey, index) =>
            <div key={`group_${groupKey}_${index}`} className="MultipleDropdown-Group">
              {grouped &&
              <Dropdown.Item>
                <Checkbox
                  label={groupOptionMap.get(groupKey)?.label}
                  className="MultipleDropdown-GroupName"
                  checked={groupOptionMap.get(groupKey)?.checked}
                  onChange={(e, { checked }) => selectGroup(groupKey, !!checked)}
                  />
              </Dropdown.Item>}
              {groupMap.get(groupKey)?.map((option, optionIndex) => (
                <Dropdown.Item key={option.value}>
                  <Checkbox
                    label={option.label}
                    className={classNames("MultipleDropdown-GroupItem", { grouped })}
                    checked={!!option.checked}
                    value={option.value}
                    onChange={(e, { checked }) => selectGroupItem(groupKey, optionIndex, !!checked)}
                  />
                </Dropdown.Item>
              ))}
            </div>
          )}
        </Dropdown.Menu>
      </Dropdown>
    </div>
  );
};

export default MultipleDropdown;

function getChangedOptionsTo(options: CheckboxOption[], checked: boolean): CheckboxOption[] {
  const newOptions = options.map(option => {
    return {...option, checked};
  });

  return newOptions;
}

function getCheckedValuesFromOptions(options: CheckboxOption[]): MixedValue[] {
  return options.filter(opt => opt.checked).map(opt => opt.value);
}

function getUncheckedValuesFromOptions(options: CheckboxOption[]): MixedValue[] {
  return options.filter(opt => !opt.checked).map(opt => opt.value);
}
