import React, { useEffect, useState } from "react";
import { Button, Form, Header, Icon, List, Message, Segment } from "semantic-ui-react";
import classNames from "classnames";
import { useNavigate } from "react-router-dom";
import { useLoginDone } from "@core/store/actions";
import WithQueryStatus from "components/WithQueryStatus";
import { __, API, Auth, isElectron } from "@solid/libs";
import { useStore } from "@core/store";
import { Apollo } from "@core/api";
import { Gui, useGetGuisLazyQuery, useUserInfoLazyQuery } from "@generated/graphql";
import {ApiError} from "@solid/types";
import useIsMounted from "@core/useIsMounted";

import "./style.css";

const Login = () => {
  const [getUserInfo, { data: info, error: infoError }] = useUserInfoLazyQuery({ fetchPolicy: "network-only" });
  const [getGuis, { loading }] = useGetGuisLazyQuery();
  const { setStore, store: { session: { isLoggedIn } } } = useStore();
  const { loginDone } = useLoginDone();
  const navigate = useNavigate();
  const [username, setUsername] = useState(process.env.API_USER || "");
  const [password, setPassword] = useState(process.env.API_PASS || "");
  const [errorMessage, setErrorMessage] = useState("");
  const [isLoading, setLoading] = useState(false);
  const [isEmailCanBeSend, setEmailCanBeSend] = useState(false);
  const [isVisibleForm, setIsVisibleForm] = useState(false);
  const isMounted = useIsMounted();

  useEffect(() => {
    if (info) {
      setStore({ session: { isLoggedIn: true, isAdmin: info.userInfo.user.isAdmin, info: info.userInfo } });
    }
    else if (!isLoggedIn && isElectron() && username && password) {
      login();
    }

    if (infoError) {
      setIsVisibleForm(true);
    }
  }, [info, infoError]);

  useEffect(() => {
    const api = new API();
    Auth.setAPI(api);

    const getParams = async () => {
      const response = await api.getParameters();

      if (!isMounted()) {
        return;
      }

      setEmailCanBeSend(response.list.EMAIL_CAN_BE_SEND);
    };
    getParams();

    return () => {
      setEmailCanBeSend(false);
    };
  }, []);

  useEffect(() => {
    if (!isLoggedIn) {
      checkUserInfo();
    }
  }, [isLoggedIn]);

  const checkUserInfo = async () => {
    try {
      await getUserInfo();
    }
    catch {
      setIsVisibleForm(true);
    }
  };

  const accountRecovery = () => {
    navigate("/recover/account");
  };

  const checkStatus = (status: string) => {
    if (status === "PasswordExpired") {
      navigate("/passwordexpired");
      return;
    }
    if (status === "LoginNotVerified") {
      navigate("/confirm/code");
    }
  };

  const loggedIn = async () => {
    setLoading(true);
    try {
      setErrorMessage("");
      await loginDone();
    }
    catch (e) {
      const {code, message} = e as ApiError;
      setErrorMessage(code ? `[${code}] ${message}` : message);
      setLoading(false);
    }
  };

  const login = async () => {
    setLoading(true);
    try {
      setErrorMessage("");
      Apollo.client?.resetStore();
      const response = await Auth.login({username, password});
      if (response.status !== "LoggedIn") {
        checkStatus(response.status);
        return;
      }

      if (isElectron()) {
        try {
          const { data, error, loading } = await getGuis();
          if (!data && !loading && error) {
            setErrorMessage(error.message);
            setLoading(false);
          }
          if (!loading && data) {
            const isSolid = data.guis.some((gui: Gui) => gui.id === "SOLID");
            if (!isSolid) {
              setLoading(false);
              setErrorMessage(__("Your account does not support the use of the Enterprise UI"));
            } else {
              loggedIn();
            }
          }
        }
        catch (error) {
          setLoading(false);
          console.error(error);
        }
      } else {
        loggedIn();
      }
    }
    catch (e) {
      const {code, message} = e as ApiError;
      Auth.logout();
      setErrorMessage(code ? `[${code}] ${message}` : message);
      setLoading(false);
    }
  };

  return (
    isVisibleForm ?
      <Segment className={classNames("login", { "loading": isLoading || loading })}>
        <WithQueryStatus loading={isLoading}>
          <Form onSubmit={(e) => e.preventDefault()}>
            <Form.Field>
              <Header as="h1" className="login-header">{__("Log in")}</Header>
              <Message visible error content={errorMessage} hidden={!errorMessage} />
            </Form.Field>
            <Form.Field>
              <div className="ui left icon input">
                <input type="text" id="username" tabIndex={1} defaultValue={username} placeholder={__("Username")} autoFocus autoComplete="username"
                  onInput={(e) => { setUsername(e.currentTarget.value); }} />
                <Icon name="user outline"/>
              </div>
            </Form.Field>
            <Form.Field>
              <div className="ui left icon input">
                <input type="password" id="password" tabIndex={2} defaultValue={password} placeholder={__("Password")} autoComplete="current-password"
                  onInput={(e) => { setPassword(e.currentTarget.value); }} />
                <Icon name="unlock alternate"/>
              </div>
            </Form.Field>
            {isEmailCanBeSend &&
            <div className="form-text">
              <List>
                <List.Item>
                  <List.Content>
                    <List.Description>
                      {__("Forgot username/password? ")}
                      <a className="recover" onClick={() => accountRecovery()}>{__("Recover account")}</a>
                    </List.Description>
                  </List.Content>
                </List.Item>
              </List>
            </div>}
            <Button id="login" primary fluid onClick={() => login()}>{__("Login")}</Button>
          </Form>
        </WithQueryStatus>
      </Segment> :
      <></>
  );
};

export default Login;
