import React from "react";
import ReactDOM from "react-dom";
import { Widget } from "@lumino/widgets";
import { Message } from "@lumino/messaging";
import { RBaseWidget, RBaseWidgetProps, WrapperHelper, IWrappedWidget } from "./RBaseWidget";
import { WrapperSplitPanel } from "./RSplitPanel";
import WidgetErrorBoundary from "./WidgetErrorBoundary";

type WidgetInfo = {
  component: JSX.Element;
  node: HTMLElement;
  widget: WrapperWidget;
};

type RWidgetProps = RBaseWidgetProps & {
  top?: boolean;
};

export class WrapperWidget extends Widget implements IWrappedWidget {
  readonly wrapper: WrapperHelper;

  constructor(node: HTMLElement, id: string, name: string, props?: Omit<RWidgetProps, "id" | "name">) {
    super({ node });
    this.wrapper = new WrapperHelper(this, id, name);
    this.setFlag(Widget.Flag.DisallowLayout);
    this.addClass("Widget-Content");
    if (props && props.top) {
      this.addClass("Widget-Content_top");
    }
  }

  override processMessage(msg: Message): void {
    super.processMessage(msg);
    //console.log("processMessage", this.id, msg.type);
  }

  override onCloseRequest(msg: Message) {
    super.onCloseRequest(msg);
    this.wrapper.publishClose();
  }

  protected override onResize(msg: Widget.ResizeMessage): void {
    super.onResize(msg);
    this.wrapper.widgetResize.publish(msg);
    if (this.wrapper.layoutChangedDisabledAlways) {
      return;
    }
    let parent = this.parent;
    while (parent) {
      if (parent instanceof WrapperSplitPanel) {
        parent.childResized(this.id);
        break;
      }
      parent = parent.parent;
    }
  }
}

export class RWidget extends RBaseWidget<RWidgetProps, WrapperWidget> {
  widgetInfo?: WidgetInfo;
  portal?: React.ReactPortal;

  protected createLayoutContainer(): undefined {
    return undefined;
  }

  protected override getWidgetToAdd(): WrapperWidget {
    const children = this.props.children;

    if (!this.widgetInfo) {
      const node = document.createElement("div");
      const widget = new WrapperWidget(node, this.props.id, this.props.name || "", this.props);
      this.widgetInfo = { node, component: <>{children}</>, widget };
    }

    return this.widgetInfo.widget;
  }

  override renderComponent(): React.ReactNode | undefined {
    if (this.widgetInfo && !this.portal) {
      this.portal = ReactDOM.createPortal(
        <WidgetErrorBoundary>{this.widgetInfo.component}</WidgetErrorBoundary>, this.widgetInfo.widget.node);
    }
    return this.portal;
  }

  override render() {
    return this.renderComponent();
  }
}
