import React, { useState, useEffect, useRef } from "react";
import { Modal, Header, Button, Icon } from "semantic-ui-react";
import { Hierarchy, HierarchyLevel } from "@generated/graphql";
import TreeView, { TreeNode, TreeViewEvent, TreeViewEventType } from "components/TreeView";
import {__} from "@solid/libs/i18n";
import {Utils} from "@solid/libs";

type HierarchyTreeProps = {
  hierarchy: Hierarchy;
  treeViewEvent?: TreeViewEvent;
  onSelectedLevelChange?: (hierarchy: Hierarchy, level: HierarchyLevel | undefined) => void;
  onHierarchyChange?: (hierarchy: Hierarchy) => Promise<boolean>;
  onBeforeNewChild?: (node?: TreeNode) => Promise<boolean>;
};

const HierarchyTree = ({ hierarchy, treeViewEvent, onSelectedLevelChange, onHierarchyChange, onBeforeNewChild }: HierarchyTreeProps) => {
  const [nodes, setNodes] = useState<TreeNode[]>([]);
  const [newChildConfirmOpen, setNewChildConfirmOpen] = useState(false);
  const [newChildLevel, setNewChildLevel] = useState<HierarchyLevel | undefined>();
  const [selectedId, setSelectedId] = useState("");
  const nodesUpdateTimeRef = useRef(Date.now());

  useEffect(() => {
    setNodes(getNodes(hierarchy.level));
  }, [hierarchy]);

  function getNodes(level: HierarchyLevel): TreeNode[] {
    const levelLevels = level.levels ?? [];
    const levels = [...levelLevels];
    return levels
      .sort((a, b) => a.name.localeCompare(b.name, undefined, { sensitivity: "base" }))
      .map(level => ({
        id: level.id,
        content: level.name,
        nodes: getNodes(level),
        icon: "folder",
        expandedIcon: "folder open",
        object: level
      }));
  }

  function getHierarchy(nodes: TreeNode[]): Hierarchy {
    return { ...hierarchy, level: { ...hierarchy.level, levels: getLevels(nodes) } };
  }

  function getLevels(nodes: TreeNode[]): HierarchyLevel[] {
    return nodes.map(({ id, content, nodes, object }) => {
      if (!object) {
        return { id, name: content?.toString() ?? "", levels: getLevels(nodes), labelIds: [], __typename: "HierarchyLevel" };
      }
      const level = object;
      return { ...level, name: content?.toString() ?? "", levels: getLevels(nodes) };
    });
  }

  async function onTreeChange(nodes: TreeNode[]): Promise<boolean> {
    if (!onHierarchyChange) {
      return true;
    }
    const startTime = Date.now();
    const id = selectedId;
    const hierarchy = getHierarchy(nodes);
    const result = await onHierarchyChange(hierarchy);
    // Wait until underlying tree nodes are updated as a result of query refetch
    // and restore previous tree item selection.
    for (let i = 0; i < 5 && startTime >= nodesUpdateTimeRef.current; i++) {
      await Utils.wait(150);
    }
    setSelectedId(id);
    return result;
  }

  function findLevel(id: string, levels: HierarchyLevel[]): HierarchyLevel | undefined {
    if (!id) {
      return undefined;
    }
    const level = levels.find(level => level.id === id);
    if (level) {
      return level;
    }
    for (const child of levels) {
      const childLevels = child.levels ?? [];
      const level = findLevel(id, childLevels);
      if (level) {
        return level;
      }
    }
    return undefined;
  }

  function onSelectedChange(id: string): void {
    setSelectedId(id);
    if (onSelectedLevelChange) {
      const levelLevels = hierarchy.level.levels ?? [];
      onSelectedLevelChange(hierarchy, findLevel(id, levelLevels));
    }
  }

  async function onNewChild(node?: TreeNode): Promise<void> {
    const startTime = Date.now();
    if (onBeforeNewChild) {
      const updated = await onBeforeNewChild(node);
      if (updated) {
        // Wait until underlying tree nodes are updated as a result of query refetch
        // before issuing a command to insert new tree node.
        for (let i = 0; i < 10 && startTime >= nodesUpdateTimeRef.current; i++) {
          await Utils.wait(150);
        }
      }
    }
    if (!node) {
      createNewChild();
      return;
    }
    const hierarchyLevelLevels = hierarchy.level.levels ?? [];
    const level = findLevel(node.id, hierarchyLevelLevels);
    if (!level) {
      return;
    }
    const levelLevels = level.levels ?? [];
    if (level.labelIds.length === 0 || levelLevels.length > 0) {
      createNewChild(node.id);
      return;
    }
    setNewChildLevel(level);
    setNewChildConfirmOpen(true);
  }

  function createNewChild(nodeId?: string): void {
    treeViewEvent?.publish({ type: TreeViewEventType.CreateNewChildCmd, nodeId });
  }

  return (
    <>
      <TreeView
        nodes={nodes}
        event={treeViewEvent}
        selectedId={selectedId}
        onSelectedChange={onSelectedChange}
        onTreeChange={onTreeChange}
        onNewChild={onNewChild}
        onNodesUpdated={() => { nodesUpdateTimeRef.current = Date.now(); }}
      />

      {newChildConfirmOpen &&
        <Modal open={newChildConfirmOpen} onClose={() => setNewChildConfirmOpen(false)}>
          <Header>{__("Add Branch")}</Header>
          <Modal.Content>
            {__("Adding branch under '{{name}}' will remove '{{name}}' labels assignments.", {name: newChildLevel?.name})}
            <br/>
            {__("Would you like to continue?")}
          </Modal.Content>
          <Modal.Actions>
            <Button
              positive
              onClick={() => {
                createNewChild(newChildLevel?.id);
                setNewChildConfirmOpen(false);
              }}>
              <Icon name="check"/>{__("Continue")}
            </Button>
            <Button onClick={() => setNewChildConfirmOpen(false)}>
              <Icon name="cancel"/>{__("Cancel")}
            </Button>
          </Modal.Actions>
        </Modal>}
    </>
  );
};

export default HierarchyTree;
