import React, { useState, useEffect } from "react";
import { Modal, Dropdown, DropdownItemProps, DropdownProps, Header, Button, Icon, Divider } from "semantic-ui-react";
import { v1 as uuid } from "uuid";
import { useLabelsQuery, LabelInput, LabelType } from "@generated/graphql";
import { useLabelActions } from "@core/actions";
import WithQueryStatus from "components/WithQueryStatus";
import Loading from "components/Loading";
import { isArray } from "utils";
import { getLabelDropdownItem } from "components/LabelsAndHierarchies/utils";
import {Log} from "@solid/libs/log";
import {__} from "@solid/libs/i18n";

import "./style.css";

type AssignLabelsDialogProps = {
  deviceId: string;
  name: string;
  open: boolean;
  onClose: () => void;
};

enum CustomSearchOptions {
  CreateLabel = "CreateLabel",
  ShowAll = "ShowAll",
  Divider1 = "Divider1",
  Divider2 = "Divider2"
}

const customSearchOptions: string[] = [CustomSearchOptions.CreateLabel, CustomSearchOptions.ShowAll, CustomSearchOptions.Divider1, CustomSearchOptions.Divider2];

const initialLoadLimit = 5;

const AssignLabelsDialog = ({ deviceId, name, open, onClose }: AssignLabelsDialogProps) => {
  const { data, loading, error } = useLabelsQuery();
  const [searchOptions, setSearchOptions] = useState<DropdownItemProps[]>([]);
  const [searchValues, setSearchValues] = useState<string[]>([]);
  const [searchQuery, setSearchQuery] = useState("");
  const [loadLimit, setLoadLimit] = useState(initialLoadLimit);
  const [createdLabels, setCreatedLabels] = useState<LabelInput[]>([]);
  const { insertLabels, assignLabels, removeLabels } = useLabelActions();
  const [updating, setUpdating] = useState(false);

  useEffect(() => {
    error && console.error("Labels query error:", error);
  }, [error]);

  useEffect(() => {
    setLoadLimit(initialLoadLimit);
  }, [searchQuery]);

  useEffect(() => {
    if (!data) {
      return;
    }

    let options: DropdownItemProps[];
    let showAllVisible = false;

    const labels = data.labels.filter(label => label.type === LabelType.Label);

    let unselectedLabels = labels.filter(label => !searchValues.includes(label.id));
    if (searchQuery) {
      unselectedLabels = unselectedLabels.filter(({ name }) => name.toLocaleUpperCase().includes(searchQuery.toLocaleUpperCase()));
    }

    if (loadLimit > 0) {
      const selectedLabels = labels.filter(label => searchValues.includes(label.id));

      showAllVisible = unselectedLabels.length > loadLimit;
      unselectedLabels = unselectedLabels.sort((a, b) => a.name.localeCompare(b.name, undefined, { sensitivity: "base" })).slice(0, loadLimit);

      options = selectedLabels.concat(unselectedLabels).map(label => getLabelDropdownItem(label));
    }
    else {
      options = labels.map(label => getLabelDropdownItem(label));
    }

    options = options.concat(createdLabels.map(label => getLabelDropdownItem({ ...label, objects: [], type: LabelType.Label })));

    options = options.sort((a, b) => a.text?.toString().localeCompare(b.text?.toString() ?? "", undefined, { sensitivity: "base" }) ?? 0);

    const createLabelVisible = searchQuery && !data.labels.some(label => label.name === searchQuery) && !createdLabels.some(label => label.name === searchQuery);
    if (createLabelVisible) {
      if (unselectedLabels.length > 0) {
        options.unshift({ key: CustomSearchOptions.Divider1, value: CustomSearchOptions.Divider1, text: "", content: <Divider/>, disabled: true });
      }
      options.unshift({ key: CustomSearchOptions.CreateLabel, value: CustomSearchOptions.CreateLabel, text: `${__("Create label")} "${searchQuery}"` });
    }

    if (showAllVisible) {
      if (options.length > 0 && options[options.length - 1].value !== CustomSearchOptions.Divider1) {
        options.push({ key: CustomSearchOptions.Divider2, value: CustomSearchOptions.Divider2, text: "", content: <Divider/>, disabled: true });
      }
      options.push({ key: CustomSearchOptions.ShowAll, value: CustomSearchOptions.ShowAll, text: __("Show all labels") });
    }

    setSearchOptions(options);
  }, [data, searchQuery, createdLabels, loadLimit, searchValues]);

  useEffect(() => {
    if (!data) {
      return;
    }

    const values = data.labels
      .filter(label => label.type === LabelType.Label && label.objects.some(object => object.objectId === deviceId))
      .sort((a, b) => a.name.localeCompare(b.name, undefined, { sensitivity: "base" }))
      .map(label => label.id);

    setSearchValues(values);
  }, [data, deviceId]);

  function onMount(): void {
    setSearchQuery("");
    setCreatedLabels([]);
    setLoadLimit(initialLoadLimit);
  }

  function filterSearchOptions(options: DropdownItemProps[], searchValue: string): DropdownItemProps[] {
    return options.filter(({ value, text }) =>
      value &&
      (customSearchOptions.includes(value.toString()) ||
       text?.toString().toLocaleUpperCase().includes(searchValue.toLocaleUpperCase())));
  }

  function onSearchValueChange(e: React.SyntheticEvent, { value }: DropdownProps): void {
    if (!isArray<string>(value, "string") || !data) {
      return;
    }

    setSearchValues(value.filter(option => !customSearchOptions.includes(option)));

    setCreatedLabels(labels => {
      if (!labels.some(label => !value.includes(label.id))) {
        return labels;
      }
      return labels.filter(label => value.includes(label.id));
    });

    if (value.includes(CustomSearchOptions.CreateLabel)) {
      createNewLabel();
    }

    if (value.includes(CustomSearchOptions.ShowAll)) {
      setLoadLimit(0);
    }
  }

  function createNewLabel(): void {
    const id = uuid();
    const label: LabelInput = { id, name: searchQuery, objectIds: [deviceId] };

    setCreatedLabels(labels => {
      const newLabels = Array.from(labels);
      newLabels.push(label);
      return newLabels;
    });

    setSearchValues(labelIds => {
      const newIds = Array.from(labelIds);
      newIds.push(id);
      return newIds;
    });

    setSearchQuery("");
  }

  function onOkClick(): void {
    if (!data) {
      onClose();
      return;
    }

    const assignLabelIds = searchValues
      .filter(id =>
        !createdLabels.some(label => label.id === id) &&
        !data.labels.some(label => label.id === id && label.objects.some(object => object.objectId === deviceId)));

    const removeLabelIds = data.labels
      .filter(label => label.type === LabelType.Label && label.objects.some(object => object.objectId === deviceId) && !searchValues.includes(label.id))
      .map(label => label.id);

    if (createdLabels.length === 0 && assignLabelIds.length === 0 && removeLabelIds.length === 0) {
      onClose();
      return;
    }

    setUpdating(true);

    const promises: Promise<boolean>[] = [];
    if (createdLabels.length > 0) {
      promises.push(insertLabels(createdLabels));
    }
    if (assignLabelIds.length > 0) {
      promises.push(assignLabels(deviceId, assignLabelIds));
    }
    if (removeLabelIds.length > 0) {
      promises.push(removeLabels(deviceId, removeLabelIds));
    }

    Promise.all(promises)
      .then(onClose)
      .catch(e => Log.error(`${__("Labels update error")}: ${e.message}`))
      .finally(() => setUpdating(false));
  }

  return (
    <Modal className="AssignLabelsDialog" open={open} onClose={onClose} onMount={onMount}>
      <Header content={__("Assign labels to '{{name}}'", {name})}/>

      <Modal.Content>
        <WithQueryStatus loading={loading} error={error}>
          <Dropdown
            placeholder={__("Labels")}
            className="AssignLabelsDialog-Dropdown"
            fluid
            selection
            multiple
            open
            search={filterSearchOptions}
            options={searchOptions}
            value={searchValues}
            searchQuery={searchQuery}
            onChange={onSearchValueChange}
            onSearchChange={(e, { searchQuery }) => setSearchQuery(searchQuery)}
          />
        </WithQueryStatus>
        {updating && <Loading text={__("Updating...")}/>}
      </Modal.Content>

      <Modal.Actions>
        <Button positive disabled={loading || !!error || updating} onClick={onOkClick}><Icon name="checkmark"/>{__("OK")}</Button>
        <Button negative onClick={onClose}><Icon name="cancel"/>{__("Cancel")}</Button>
      </Modal.Actions>
    </Modal>
  );
};

export default AssignLabelsDialog;
