import React, { useEffect, useRef, useState } from "react";
import { Button, Form, Input } from "semantic-ui-react";
import { FontAwesomeIcon } from "@fortawesome/react-fontawesome";
import moment from "moment";
import rome, { RomeDatePicker } from "@bevacqua/rome";

import { formatDateTime } from "@core/utils";

import "./style.css";

type InputDateProps = {
  label?: string;
  placeholder?: string;
  value?: Date;
  minTime?: Date;
  maxTime?: Date;
  disabled?: boolean;
  withUpdate?: boolean;
  onChange: (timestamp: number) => void;
};

const InputDate = ({ label, placeholder, value, minTime, maxTime, disabled = false, withUpdate = false, onChange }: InputDateProps) => {
  const [calendarOpen, setCalendarOpen] = useState<boolean>(false);
  const [currentValue, setCurrentValue] = useState<number>(value?.getTime() || Date.now());

  const calendarPlaceholderRef = useRef<HTMLDivElement>(null);
  const calendarElemRef = useRef<HTMLDivElement>();
  const calendarRef = useRef<RomeDatePicker>();

  useEffect(() => {
    createCalendar();

    return () => {
      destroyCalendar();
    };
  }, []);

  useEffect(() => {
    if (value instanceof Date) setCurrentValue(value.getTime());
  }, [value]);

  const createCalendar = () => {
    const calendarPlaceholder = calendarPlaceholderRef.current;
    if (!calendarPlaceholder) {
      return;
    }

    const elem = document.createElement("div");
    elem.className = withUpdate ? "Calendar withUpdate" : "Calendar";

    calendarPlaceholder.appendChild(elem);
    calendarElemRef.current = elem;

    calendarRef.current = rome(elem, {
      min: moment(minTime || Date.now() - 24 * 60 * 60 * 1000),
      max: moment(maxTime || Date.now()),
    });

    if (calendarRef.current) {
      calendarRef.current.hide();

      calendarRef.current.setValue(value);
      calendarRef.current.on("data", (timeStr: string) => {
        const value = moment(timeStr);
        const time = moment(value).valueOf();

        hideCalendar();
        setCurrentValue(time);
        onChange(time);
      });
    }
  };

  const destroyCalendar = () => {
    hideCalendar();
    if (calendarRef.current) {
      calendarRef.current.destroy();
      calendarRef.current = undefined;
    }
    if (calendarElemRef.current) {
      calendarElemRef.current.remove();
      calendarElemRef.current = undefined;
    }
  };

  const showCalendar = () => {
    calendarRef.current?.setValue(currentValue);
    calendarRef.current?.show();
    setCalendarOpen(true);
  };

  const hideCalendar = () => {
    calendarRef.current?.hide();
    setCalendarOpen(false);
  };

  const syncDate = () => {
    calendarRef.current?.hide();
    setCalendarOpen(false);
    const time = Date.now();
    setCurrentValue(time);
    onChange(time);
  };

  return (
    <Form.Field inline className="InputDate-Wrapper" >
      <label>{label}</label>
      <div className="InputDate">
        <Input
          readOnly
          autoFocus
          disabled={disabled}
          labelPosition="right"
          placeholder={placeholder}
          value={formatDateTime(new Date(currentValue || Date.now()))}
          label={
            <>
              { !calendarOpen ?
                <Button icon onClick={() => showCalendar()}>
                  <FontAwesomeIcon icon="calendar-alt"/>
                </Button> :
                <Button icon onClick={() => hideCalendar()}>
                  <FontAwesomeIcon icon="times"/>
                </Button> }
              { withUpdate &&
                <Button icon onClick={() => syncDate()}>
                  <FontAwesomeIcon icon="sync-alt"/>
                </Button> }
            </>
          }
        />
        <div ref={calendarPlaceholderRef} className="CalendarPlaceholder"/>
      </div>
    </Form.Field>
  );
};

export default InputDate;

