import React, { useState } from "react";
import { Checkbox, Dropdown, Form, Input } from "semantic-ui-react";
import { FieldDescription, FieldValue } from "components/FieldEditor";
import PasswordField from "components/PasswordField";

type FieldResolverProps = {
  field: FieldDescription;
};

const FieldResolver = ({ field }: FieldResolverProps) => {
  const { type, value, disabled = false, options = [], min = 0, max, error, hidden, required, onChange, onBlur, onFocus } = field;
  const [checkboxState, setCheckboxState] = useState<boolean>(!!value);
  const [dropdownState, setDropdownState] = useState<FieldValue>(value);
  const [stringValue, setStringValue] = useState<FieldValue>(value);
  const [numberValue, setNumberValue] = useState<FieldValue>(value);

  if (hidden) {
    return <></>;
  }

  switch (type) {
    case "label":
      return <>{value}</>;
    case "boolean":
      return (
        <Form.Field
          error={error ? { content: error, pointing: "below" } : undefined}
          disabled={disabled}
          required={required}
        >
          <Checkbox
            toggle
            checked={checkboxState}
            onChange={(e, { checked }) => {
              setCheckboxState(!!checked);
              onChange && onChange(!!checked);
            }}
        />
        </Form.Field>
      );
    case "dropdown":
      return (
        <Form.Field
          error={error ? { content: error, pointing: "below" } : undefined}
          disabled={disabled}
          required={required}
        >
          <Dropdown
            selection
            options={options}
            value={dropdownState as string}
            onChange={(e, { value }) => {
              setDropdownState(value);
              onChange && onChange(value);
            }}
        />
        </Form.Field>
      );
    case "string":
      return (
        <Form.Field
          error={error ? { content: error, pointing: "below" } : undefined}
          required={required}
          control={Form.Input}
          disabled={disabled}
          value={stringValue}
          onBlur={onBlur}
          onFocus={onFocus}
          onInput={(e: React.FormEvent<HTMLInputElement>) => {
            const value = e.currentTarget.value;
            setStringValue(value);
            onChange && onChange(value);
          }}
        />
      );
    case "number":
      return (
        <Form.Field
          error={error ? { content: error, pointing: "below" } : undefined}
          disabled={disabled}
          required={required}>
          <Input
            type="number"
            min={min}
            max={max}
            value={numberValue}
            onBlur={onBlur}
            onFocus={onFocus}
            onInput={(e: React.FormEvent<HTMLInputElement>) => {
              const value = validateNumber(e.currentTarget.value, min, max);
              setNumberValue(value);
              onChange && onChange(value);
            }}
          />
        </Form.Field>
      );
    case "password":
      return (
        <Form.Field
          error={error ? { content: error, pointing: "below" } : undefined}
          disabled={disabled}
          required={required}
        >
          <PasswordField
            value={value ? value.toString() : ""}
            onChange={onChange}
            onBlur={onBlur}
            onFocus={onFocus}
        />
        </Form.Field>
      );
    default:
      return <></>;
  }
};

export default FieldResolver;


function validateNumber(value: string, min?: number, max?: number): number {
  const num: number = isNaN(Number(value)) ? 0 : Number(value);
  if (typeof min === "number" && num < min) {
    return min;
  }
  if (typeof max === "number" && num > max) {
    return max;
  }
  return num;
}
