import React, { useState, useEffect } from "react";
import { Form } from "semantic-ui-react";
import { useAcknowledgeEventMutation, AuditEntryInput } from "@generated/graphql";
import { withStore, WithStoreProps } from "@core/store";
import { AuditEntryWithSnapshots, getAuditEntryInput, getStoreEvent } from "@core/actions";
import { queryToInput } from "utils";
import Loading from "components/Loading";
import FontAwesomeButton from "components/FontAwesomeButton";
import {Log} from "@solid/libs/log";
import {__} from "@solid/libs/i18n";

import "./style.css";

type EventAcknowledgeProps = WithStoreProps & {
  event: AuditEntryWithSnapshots;
};

const EventAcknowledge = ({ event, setStore }: EventAcknowledgeProps) => {
  const [value, setValue] = useState("");
  const [isAcknowledged, setIsAcknowledged] = useState(false);
  const [acknowledgeEvent, { data: updData, loading: updating, error: updError }] = useAcknowledgeEventMutation();

  useEffect(() => {
    setIsAcknowledged(event.isAcknowledged);
    setValue(event.note ?? "");
  }, [event]);

  useEffect(() => {
    if (updData?.acknowledgeAuditEntry) {
      setIsAcknowledged(true);
      const storeEvent = getStoreEvent<AuditEntryWithSnapshots>({...event, note: value});
      const updatedEvent = queryToInput<AuditEntryWithSnapshots, AuditEntryInput>(storeEvent);
      setStore({ workspace: { updatedEvent } });
    }
  }, [updData]);

  useEffect(() => {
    if (updError) {
      Log.error(`${__("Event acknowledge error")}: ${updError.message}`);
    }
  }, [updError]);

  function acknowledge(): void {
    if (!isAcknowledged) {
      const entry = getAuditEntryInput({ ...event, note: value });
      acknowledgeEvent({ variables: { entry }});
    }
  }

  return (
    <div className="EventAcknowledge">
      <Form>
        <Form.TextArea
          placeholder={__("User notes")}
          rows={3}
          value={value}
          readOnly={isAcknowledged}
          onInput={e => setValue(e.currentTarget.value)}
        />
      </Form>
      <div className="EventAcknowledge-Buttons">
        <FontAwesomeButton icon="check" disabled={isAcknowledged} hint={__("Acknowledge")} onClick={acknowledge}/>
        <FontAwesomeButton icon="times" disabled={isAcknowledged || !value} hint={__("Clear Notes")} onClick={() => setValue("")}/>
      </div>
      {updating && <Loading text={__("Updating...")}/>}
    </div>
  );
};

export default withStore(EventAcknowledge);
