import React, { useEffect, useMemo, useRef, useState } from "react";
import { Icon, Popup, Ref, Segment } from "semantic-ui-react";
import AutoSizer from "react-virtualized-auto-sizer";
import {FixedSizeList as List, ListChildComponentProps} from "react-window";
import classNames from "classnames";
import {
  DeviceFunctionalAspectFilter,
  DeviceFunctionalAspectType,
  HealthStatus,
  Hierarchy,
  HierarchyLevel,
  Label,
  useDeviceListByAspectTypesQuery,
  useHierarchiesLazyQuery,
  useLabelsQuery
} from "@generated/graphql";
import CameraListItem from "./CameraListItem";
import WithQueryStatus from "components/WithQueryStatus";
import ListFilter from "../ListFilter";
import TreeView, { TreeNode } from "components/TreeView";
import { EventPubSub, getLocalStorage, parseJSON, setLocalStorage } from "utils";
import { WidgetProps } from "components/Widgets";
import {__} from "@solid/libs/i18n";
import { Utils } from "@solid/libs/utils";
import { DeviceList } from "@core/actions";
import { UUID } from "@solid/types";

import "./style.css";

export class ShowHideFilterEvent extends EventPubSub<{}> {}

type CameraListProps = WidgetProps & {
  selectable?: boolean;
  selectedId?: string;
  selectionStyle?: SelectionStyle;
  onSelected?: (id: string, name: string) => void;
  multiselect?: boolean;
  selectedIds?: string[];
  onSelectedMultiple?: (ids: string[]) => void;
  hierarchyId?: string;
  openMultipleBranches?: boolean;
  hierarchySelection?: boolean;
  nameFilter?: boolean;
  labelsFilter?: boolean;
  allFilters?: boolean;
  allowLabelHierarchyEdit?: boolean;
  showHideFilterEvent?: ShowHideFilterEvent;
  hiddenIds?: string[];
  deviceTypes?: DeviceFunctionalAspectFilter[];
};

export enum SelectionStyle {
  Border = "Border",
  Background = "Background",
  Checkbox = "Checkbox"
}

type DeviceNode = {
  id: UUID,
  parentId?: string;
};

const CameraList = ({
  deviceTypes = [{ type: DeviceFunctionalAspectType.Media }],
  selectable = false,
  selectionStyle = SelectionStyle.Border,
  onSelected,
  multiselect = false,
  selectedIds,
  onSelectedMultiple,
  hierarchyId = "",
  openMultipleBranches = false,
  hierarchySelection = false,
  nameFilter = true,
  labelsFilter = true,
  allFilters = false,
  allowLabelHierarchyEdit = false,
  showHideFilterEvent,
  hiddenIds,
  viewId,
  widgetId,
  index: widgetIndex,
  cellProps,
  setCellProps,
  ...props
}: CameraListProps) => {
  const isHierarchyAllowed = !!hierarchyId || allFilters;
  const { loading, error, data } = useDeviceListByAspectTypesQuery({ variables: { types: deviceTypes } });
  const { data: labelData, error: labelError, loading: labelLoading } = useLabelsQuery();
  const [execHierarchiesQuery, { data: hierarchyData, error: hierarchyError, loading: hierarchyLoading }] = useHierarchiesLazyQuery();
  const [filterVisible, setFilterVisible] = useState(false);
  const [cameraList, setCameraList] = useState<DeviceList[]>([]);
  const [hierarchy, setHierarchy] = useState<Hierarchy | undefined>();
  const isHierarchyMode = !!hierarchyId || (allFilters && !!hierarchy);
  const [nodes, setNodes] = useState<TreeNode[]>([]);
  const [selectedId, setSelectedId] = useState(props.selectedId ?? "");
  const [searchText, setSearchText] = useState("");
  const [labels, setLabels] = useState<Label[]>([]);
  const [expandedIds, setExpandedIds] = useState(getStoredExpandedIds());
  const rootRef = useRef<HTMLDivElement>(null);
  const labelDeviceMap = useMemo(() => getLabelDeviceMap(), [labelData]);
  const selectedIdSet = useMemo(() => new Set<string>(selectedIds), [selectedIds]);
  const lastSelectIndexRef = useRef(0);

  useEffect(() => {
    const id = `${Date.now()}.${Math.random()}`;
    showHideFilterEvent?.subscribe(id, () => setFilterVisible(value => !value));

    return function cleanup() {
      showHideFilterEvent?.unsubscribe(id);
    };
  });

  useEffect(() => {
    if (isHierarchyAllowed) {
      execHierarchiesQuery();
    }
  }, [isHierarchyAllowed]);

  useEffect(() => {
    if (hierarchyData && hierarchyId && !hierarchy) {
      setHierarchy(hierarchyData.hierarchies.find(hierarchy => hierarchy.id === hierarchyId));
    }
  }, [hierarchyData]);

  useEffect(() => {
    if (!isHierarchyMode) {
      return;
    }
    if (!hierarchy || !data || !labelData) {
      setNodes([]);
      return;
    }
    setNodes(getNodes(hierarchy.level));
  }, [hierarchy, data, labelData, searchText, isHierarchyMode, selectedId, selectedIdSet, hiddenIds]);

  useEffect(() => {
    if (hierarchy) {
      setExpandedIds(getStoredExpandedIds());
    }
  }, [hierarchy]);

  useEffect(() => {
    if (!isHierarchyMode || !hierarchy || !data || !labelData || !selectable || !multiselect || !onSelectedMultiple) {
      return;
    }
    // Update list of selected devices to limit selection to visible devices only.
    const devices: DeviceNode[] = [];
    getVisibleCameraNodes(nodes, devices);
    const deviceIdSet = new Set<string>(devices.map(dev => dev.id));
    const ids = Array.from(selectedIdSet.keys()).filter(id => deviceIdSet.has(id));
    if (JSON.stringify(Array.from(selectedIds ?? []).sort()) !== JSON.stringify(ids.sort())) {
      onSelectedMultiple(ids);
    }
  }, [nodes, expandedIds]);

  useEffect(() => {
    if (isHierarchyMode) {
      return;
    }
    if (data && data.devicesByAspectTypes) {
      const cameraList = getCameraList(data.devicesByAspectTypes, searchText, labels, hiddenIds);
      setCameraList(cameraList);
      // Update list of selected devices to limit selection to visible devices only.
      if (selectable && multiselect && onSelectedMultiple) {
        const deviceIdSet = new Set<string>(cameraList.map(cam => cam.id));
        const ids = Array.from(selectedIdSet.keys()).filter(id => deviceIdSet.has(id));
        if (JSON.stringify(Array.from(selectedIds ?? []).sort()) !== JSON.stringify(ids.sort())) {
          onSelectedMultiple(ids);
        }
      }
    }
  }, [data, searchText, labels, isHierarchyMode, hiddenIds]);

  useEffect(() => {
    if (selectable) {
      setSelectedId(props.selectedId ?? "");
    }
  }, [props.selectedId]);

  useEffect(() => {
    if (!setCellProps) {
      return;
    }

    if (!allFilters && !(isHierarchyAllowed && hierarchySelection) && !(!isHierarchyMode && labelsFilter) && !nameFilter) {
      setCellProps({ title: cellProps?.title ?? "" });
      return;
    }

    const title =
      <div className="CameraList-Title">
        <div className="CameraList-TitleName">{cellProps?.title}</div>

        <Popup
          trigger={
            <Icon
              name="search"
              className="CameraList-TitleIcon"
              onClick={() => setFilterVisible(visible => !visible)}
            />}
          content={filterVisible ? __("Hide Filter") : __("Show Filter")}
          position="bottom right"
        />
      </div>;

    setCellProps({ title });
  }, [filterVisible, allFilters, hierarchySelection, labelsFilter, nameFilter, isHierarchyAllowed, isHierarchyMode]);

  function getCameraList(cameraList: DeviceList[], searchText: string = "", labels: Label[] = [], hiddenIds: string[] = []): DeviceList[] {
    const idText = searchText?.toLowerCase();
    const text = searchText?.toLocaleUpperCase();
    const list = cameraList
      .filter(cam =>
        ((!text
          || cam.name.toLocaleUpperCase().includes(text)
          || cam.healthStatus?.toLocaleUpperCase().includes(text))
        || (!idText || Utils.shortUuid(cam.id, false).toLowerCase().includes(idText)))
        && (
          labels.length === 0
          || labels.some(label => label.objects.some(object => object.objectId === cam.id))
        )
        && (
          !hiddenIds
          || !hiddenIds.includes(cam.id)
        ))
      .sort((a, b) => a.name.localeCompare(b.name, undefined, { sensitivity: "base" }))
      ?? [];
    return list;
  }

  function getNodes(level: HierarchyLevel): TreeNode[] {
    const levelLevels = level.levels ?? [];
    const levels = [...levelLevels];
    const nodes: TreeNode[] = levels
      .sort((a, b) => a.name.localeCompare(b.name, undefined, { sensitivity: "base" }))
      .map(level => {
        const levelsLength = level.levels ? level.levels.length : 0;
        let className: string | undefined;
        let nodes = levelsLength > 0 ? getNodes(level) : getCameraNodes(level);

        if (levelsLength === 0 && nodes.length > 0 && searchText) {
          const text = searchText.toLocaleUpperCase();
          nodes = nodes.filter(node => {
            return node.object?.name.toLocaleUpperCase().includes(text);
          });
          if (nodes.length === 0) {
            className = "CameraList-TreeItem_grayed";
          }
        }

        if (!className && nodes.length > 0 && nodes.every(node => node.className === "CameraList-TreeItem_grayed")) {
          className = "CameraList-TreeItem_grayed";
        }

        const node: TreeNode = {
          id: level.id,
          content: level.name,
          nodes,
          icon: "folder",
          expandedIcon: "folder open",
          className,
          object: level
        };
        return node;
      })
      .filter(node => node.nodes.length > 0 || node.className === "CameraList-TreeItem_grayed" || (node.object && node.object["createdAt"] !== undefined));
    return nodes;
  }

  function getCameraNodes(level: HierarchyLevel): TreeNode[] {
    if (!data || !labelData) {
      return [];
    }

    const devIdSet = new Set<string>();
    for (const id of level.labelIds) {
      const devIds = labelDeviceMap.get(id);
      if (devIds) {
        for (const devId of devIds) {
          if (!hiddenIds || !hiddenIds.includes(devId)) {
            devIdSet.add(devId);
          }
        }
      }
    }

    const nodes: TreeNode[] = data.devicesByAspectTypes
      .filter(dev => devIdSet.has(dev.id))
      .sort((a, b) => a.name.localeCompare(b.name, undefined, { sensitivity: "base" }))
      .map(dev => {
        const node: TreeNode = {
          id: dev.id,
          content:
            (<CameraListItem
              data-id={dev.id}
              name={dev.name}
              disabled={!dev.enabled || dev.healthStatus !== HealthStatus.Normal}
              key={dev.id}
              selectable={selectable}
              selected={isSelected(dev.id)}
              selectionStyle={selectionStyle}
              widgetId={widgetId}
              parentId={level.id}
              parentRef={rootRef}
              onClick={itemClick}
              type={dev.deviceType || undefined}
            />),
          nodes: [],
          object: {
            __typename: "HierarchyLevel",
            id: dev.id,
            name: dev.name,
            labelIds: [],
            levels: []
          },
          className: selectable
            ? (selectionStyle === SelectionStyle.Border ? "CameraList-CameraTreeItem_selectable_border" : "CameraList-CameraTreeItem_selectable_background")
            : undefined
        };
        return node;
      });
    return nodes;
  }

  function getVisibleCameraNodes(nodes: TreeNode[], devices: DeviceNode[], parentId?: string): void {
    for (const node of nodes) {
      if (node.object && node.object["createdAt"] !== undefined) {
        const device = node.object;
        devices.push({ ...device, parentId });
      }
      else if (expandedIds.includes(node.id)) {
        getVisibleCameraNodes(node.nodes, devices, node.id);
      }
    }
  }

  function getLabelDeviceMap(): Map<string, string[]> {
    const map = new Map<string, string[]>();
    if (!labelData) {
      return map;
    }
    for (const label of labelData.labels) {
      map.set(label.id, label.objects.map(object => object.objectId));
    }
    return map;
  }

  function itemClick(e: React.MouseEvent<HTMLDivElement> | undefined, id: string, name: string, parentId?: string): void {
    if (selectable) {
      if (!multiselect) {
        setSelectedId(id);
        onSelected && onSelected(id, name);
      }
      else {
        const lastSelectIndex = lastSelectIndexRef.current;
        let index = 0;
        const deviceNodes: DeviceNode[] = [];
        if (!isHierarchyMode) {
          index = cameraList.findIndex(cam => cam.id === id);
        }
        else {
          getVisibleCameraNodes(nodes, deviceNodes);
          index = deviceNodes.findIndex(dev => dev.id === id && dev.parentId === parentId);
        }
        if (e && !e.shiftKey && !e.ctrlKey && index >= 0) {
          lastSelectIndexRef.current = index;
        }

        if (!onSelectedMultiple) {
          return;
        }

        let selection: string[] = Array.from(selectedIdSet.keys());
        if (e) {
          if (!e.shiftKey && !e.ctrlKey) {
            selection = [id];
          }
          else if (e.ctrlKey) {
            if (selectedIdSet.has(id)) {
              const index = selection.indexOf(id);
              if (index >= 0) {
                selection.splice(index, 1);
              }
            }
            else {
              selection.push(id);
            }
          }
          else if (e.shiftKey && index >= 0) {
            if (!isHierarchyMode) {
              selection = cameraList.slice(Math.min(lastSelectIndex, index), Math.max(lastSelectIndex, index) + 1).map(cam => cam.id);
            }
            else {
              selection = Array.from(new Set<string>(
                deviceNodes.slice(Math.min(lastSelectIndex, index), Math.max(lastSelectIndex, index) + 1).map(dev => dev.id)).keys());
            }
          }
        }
        else if (selectedIdSet.has(id)) { // Checkbox selection
          const index = selection.indexOf(id);
          if (index >= 0) {
            selection.splice(index, 1);
          }
        }
        else {
          selection.push(id);
        }

        if (JSON.stringify(Array.from(selectedIdSet.keys()).sort()) !== JSON.stringify(Array.from(selection).sort())) {
          onSelectedMultiple(selection);
        }
      }
    }
  }

  function onHierarchyChange(hierarchy?: Hierarchy): void {
    setHierarchy(hierarchy);
    if (selectable && multiselect && onSelectedMultiple && selectedIds && selectedIds.length > 0) {
      onSelectedMultiple([]);
    }
    lastSelectIndexRef.current = 0;
  }

  function getStoredExpandedIds(): string[] {
    if (!viewId || widgetIndex === undefined) {
      return [];
    }

    const strIds = getLocalStorage(`${viewId}.${widgetIndex}.expandedIds`);
    const ids = parseJSON<string[]>(strIds);
    return ids ?? [];
  }

  function onExpandedChange(ids: string[]): void {
    setExpandedIds(ids);
    if (viewId && widgetIndex !== undefined) {
      setLocalStorage(`${viewId}.${widgetIndex}.expandedIds`, JSON.stringify(ids));
    }
  }

  function isSelected(id: string): boolean {
    if (selectable) {
      return !multiselect ? id === selectedId : selectedIdSet.has(id);
    }
    return false;
  }

  const getListItem = ({ index, style }: ListChildComponentProps) => {
    const { id, name, enabled, healthStatus, deviceType } = cameraList[index];
    return (
      <CameraListItem
        data-id={id}
        name={name}
        disabled={!enabled || healthStatus !== HealthStatus.Normal}
        key={id}
        style={(style)}
        selectable={selectable}
        selected={isSelected(id)}
        selectionStyle={selectionStyle}
        widgetId={widgetId}
        parentRef={rootRef}
        onClick={itemClick}
        type={deviceType || undefined}
      />
    );
  };

  return (
    <Ref innerRef={rootRef}>
      <Segment className="CameraList-Root">
        <WithQueryStatus
          loading={loading || labelLoading || hierarchyLoading}
          error={error || labelError || hierarchyError}>

          {filterVisible &&
          <ListFilter
            filterTextPlaceholder={__("Filter by name or status")}
            labelData={labelData}
            hierarchyData={hierarchyData}
            hierarchyId={hierarchyId}
            hierarchy={hierarchy}
            searchText={searchText}
            labels={labels}
            hierarchyMode={isHierarchyMode}
            hierarchyFilter={hierarchySelection && isHierarchyAllowed}
            nameFilter={nameFilter}
            labelsFilter={labelsFilter}
            allFilters={allFilters}
            allowLabelHierarchyEdit={allowLabelHierarchyEdit}
            localStoragePrefix={viewId && widgetIndex !== undefined ? `${viewId}.${widgetIndex}` : undefined}
            rootRef={rootRef}
            onHierarchyChange={onHierarchyChange}
            onSearchTextChange={text => setSearchText(text)}
            onLabelsChange={labels => setLabels(labels)}
          />}

          <Segment className={classNames("CameraList-List", { "hierarchical": isHierarchyMode })}>
            {!isHierarchyMode ?
              <AutoSizer>
                {({ width, height }: { width: number, height: number }) => (
                  <List
                    width={width ?? 100}
                    height={height ?? 100}
                    itemCount={cameraList.length}
                    itemSize={selectable && selectionStyle === SelectionStyle.Border ? 34 : 29}>
                    {getListItem}
                  </List>
                )}
              </AutoSizer> :

              <TreeView
                nodes={nodes}
                selectable={false}
                editable={false}
                singleBranchExpand={!openMultipleBranches}
                expandedIds={expandedIds}
                onExpandedChange={onExpandedChange}
              />}
          </Segment>
        </WithQueryStatus>
      </Segment>
    </Ref>
  );
};

export default CameraList;
