import React, { useState, useEffect, useMemo } from "react";
import { useNavigate, useParams, useLocation } from "react-router-dom";
import produce from "immer";
import { Segment, Table, Icon, Popup, Input, Modal, Button, Header } from "semantic-ui-react";
import {useZonesQuery, ZonesQuery, useDeleteZoneMutation, ZonesDocument, LabelsDocument, LabelsQuery, PoliciesQuery, PoliciesDocument, LabelType, useDeviceListByAspectTypesQuery, DeviceFunctionalAspectType, HealthStatus, DevicesByAspectTypesShortDocument, SensorListByAspectTypeDocument} from "generated/graphql";
import WithQueryStatus from "components/WithQueryStatus";
import IconButton from "components/IconButton";
import Loading from "components/Loading";
import CreateUpdateZone from "components/Admin/Zones/CreateUpdateZone";
import { WidgetProps } from "components/Widgets";
import ListText, { ListItem } from "components/Admin/Helpers/ListText";
import Help from "components/Help";
import {Log} from "@solid/libs/log";
import {__} from "@solid/libs/i18n";
import { useApolloClient } from "@apollo/client";
import type {ArrayElement, UUID} from "@solid/types";
import { RouteParams } from "@core/types";
import { DeviceList, getListItemIcon, updateZoneInDeviceCache } from "@core/actions";

import "./style.css";
import HelpMD from "./help.md";

type ZonesProps = WidgetProps;

type Zone = ArrayElement<ZonesQuery["zones"]>;

const deviceQueryTypes = [{ type: DeviceFunctionalAspectType.Media }, { type: DeviceFunctionalAspectType.Sensor }];

const Zones = ({ cellProps, setCellProps }: ZonesProps) => {
  const client = useApolloClient();
  const navigate = useNavigate();
  const { pathname } = useLocation();
  const {zoneId = "", viewId = ""} = useParams<RouteParams>();
  const { data, error, loading } = useZonesQuery();
  const { loading: deviceLoading, error: deviceError, data: deviceData } = useDeviceListByAspectTypesQuery({ variables: { types: deviceQueryTypes } });
  const [zones, setZones] = useState<Zone[]>([]);
  const [searchText, setSearchText] = useState("");
  const [selectedZoneId, setSelectedZoneId] = useState("");
  const selectedZone = selectedZoneId ? zones.find(zone => zone.id === selectedZoneId) : undefined;
  const [createUpdateOpen, setCreateUpdateOpen] = useState(false);
  const [deleteConfirmOpen, setDeleteConfirmOpen] = useState(false);
  const [deleteZone, { error: deleteError, loading: deleteLoading }] = useDeleteZoneMutation();

  const deviceMap: Map<string, DeviceList> = useMemo(() => {
    if (!deviceData?.devicesByAspectTypes) return new Map();

    const map = deviceData?.devicesByAspectTypes.reduce((acc, dev) => {
      acc.set(dev.id, dev);
      return acc;
    }, new Map());

    return map;
  }, [deviceData]);

  useEffect(() => {
    pathname.includes("/view/zone/add") && onCreateClick();
  }, [pathname]);

  useEffect(() => {
    if (!zoneId || !data) return;

    const isZoneExist = data.zones.some(zone => zone.id === zoneId);
    isZoneExist && onUpdateClick(zoneId);
  }, [data, zoneId]);

  useEffect(() => {
    if (viewId && createUpdateOpen && !deleteConfirmOpen) {
      setCreateUpdateOpen(false);
    }
  }, [viewId]);

  useEffect(() => {
    if (data) {
      const zones = Array.from(data.zones)
        .filter(zone => !searchText || zone.name.toLocaleUpperCase().includes(searchText.toLocaleUpperCase()))
        .sort((a, b) => a.name.localeCompare(b.name, undefined, { sensitivity: "base" }));
      setZones(zones);
    }
  }, [data, searchText]);

  useEffect(() => {
    if (deleteError) {
      Log.error(`Could not delete zone: ${deleteError.message}`);
    }
  }, [deleteError]);

  useEffect(() => {
    if (!setCellProps) {
      return;
    }
    let title = cellProps?.title ?? "";
    if (createUpdateOpen) {
      title = selectedZone ? __("Edit Zone") : __("Create Zone");
    }
    setCellProps({ title });
  }, [createUpdateOpen, selectedZone]);

  function onCreateClick(): void {
    setSelectedZoneId("");
    setCreateUpdateOpen(true);
    !pathname.includes("/view/zone/add") && navigate("/view/zone/add");
  }

  function onUpdateClick(zoneId: UUID): void {
    setSelectedZoneId(zoneId);
    setCreateUpdateOpen(true);
    !pathname.includes(`/view/zone/edit/${zoneId}`) && navigate(`/view/zone/edit/${zoneId}`);
  }

  function onDeleteClick(zone: Zone): void {
    setSelectedZoneId(zone.id);
    setDeleteConfirmOpen(true);
  }

  function getListTextItem(dev: { id: string; name: string }): ListItem {
    const device = deviceMap.get(dev.id);
    if (!device || !device.deviceType) return { ...dev };
    const disabled = !device.enabled || device.healthStatus !== HealthStatus.Normal;
    const icon = getListItemIcon(device.deviceType, disabled);
    return { ...dev, faIcon: icon };
  }

  async function onDeleteZone(): Promise<void> {
    setDeleteConfirmOpen(false);
    if (!selectedZone) {
      return;
    }

    const zoneId = selectedZone.id;
    const response = await deleteZone({
      variables: { id: selectedZone.id }
    });

    // update cache
    // update ZonesDocument, PoliciesDocument, LabelsDocument
    if (response.data?.deleteZone) {
      // update ZonesDocument
      const zonesCacheData = client.readQuery<ZonesQuery>({
        query: ZonesDocument
      });

      const zonesCache = zonesCacheData?.zones ?? [];
      const zoneIndex = zonesCache.findIndex((zone) => zone.id === zoneId);
      const zones = produce(zonesCache, (draft) => {
        draft.splice(zoneIndex, 1);
      });

      client.writeQuery<ZonesQuery>({
        query: ZonesDocument,
        data: {
          zones
        }
      });

      // update PoliciesDocument
      const policiesCacheData = client.readQuery<PoliciesQuery>({
        query: PoliciesDocument
      });

      const policiesCache = policiesCacheData?.policies ?? [];
      if (policiesCache.length > 0) {
        const policies = produce(policiesCache, (draft) => {
          for (const policy of draft) {
            for (const statement of policy.statements) {
              for (let i = 0; i < statement.resources.length; i++) {
                const resource = statement.resources[i];
                if (resource.id === zoneId) {
                  statement.resources.splice(i, 1);
                  break;
                }
              }
            }
          }
        });

        client.writeQuery<PoliciesQuery>({
          query: PoliciesDocument,
          data: {
            policies
          }
        });
      }


      // update LabelsDocument
      const labelsCacheData = client.readQuery<LabelsQuery>({
        query: LabelsDocument
      });

      const labelsCache = labelsCacheData?.labels ?? [];
      if (labelsCache.length > 0) {
        const labelIndex = labelsCache.findIndex((label) => label.type === LabelType.Zone && label.id === zoneId);
        if (labelIndex >= 0) {
          const labels = produce(labelsCache, (draft) => {
            draft.splice(labelIndex, 1);
          });

          client.writeQuery<LabelsQuery>({
            query: LabelsDocument,
            data: {
              labels
            }
          });
        }
      }

      // update devices
      const deviceIds: string[] = selectedZone.devices.map(dev => dev.id);
      if (deviceIds.length > 0) {
        updateZoneInDeviceCache(null, deviceIds, DeviceFunctionalAspectType.Media, DevicesByAspectTypesShortDocument, client);
        updateZoneInDeviceCache(null, deviceIds, DeviceFunctionalAspectType.Sensor, SensorListByAspectTypeDocument, client);
      }
    }
  }

  return !createUpdateOpen ?
    <Segment className="Zones">
      <WithQueryStatus loading={loading || deviceLoading} error={error || deviceError}>
        <div className="Zones-Content">
          <div className="Zones-Top">
            <Input
              placeholder={__("Filter by name")}
              icon="search"
              value={searchText}
              onChange={e => setSearchText(e.currentTarget.value)}
            />
            <IconButton
              icon="plus"
              hint={__("Create Zone")}
              onClick={onCreateClick}
            />
          </div>

          <div className="Zones-Data">
            <Table celled compact>
              <Table.Header>
                <Table.Row>
                  <Table.HeaderCell/>
                  <Table.HeaderCell>{__("Name")}</Table.HeaderCell>
                  <Table.HeaderCell>{__("Devices")}</Table.HeaderCell>
                </Table.Row>
              </Table.Header>

              <Table.Body>
                {zones.map(zone =>
                  <Table.Row key={zone.id}>
                    <Table.Cell collapsing>
                      <Popup trigger={
                        <Icon name="edit" className="Zones-IconButton" onClick={() => onUpdateClick(zone.id)}/>
                        }
                        content={__("Edit")}
                      />
                      <Popup trigger={
                        <Icon name="trash alternate" className="Zones-IconButton" onClick={() => onDeleteClick(zone)}/>
                        }
                        content={__("Delete")}
                      />
                    </Table.Cell>
                    <Table.Cell>
                      {zone.name}
                    </Table.Cell>
                    <Table.Cell>
                      <ListText items={zone.devices.map(dev => getListTextItem(dev))} maxItems={5} modalHeader={__("Devices")} icons/>
                    </Table.Cell>
                  </Table.Row>)}

                {zones.length === 0 &&
                <Table.Row>
                  <Table.Cell colSpan={3} textAlign="center">
                    {__("No zones found")}
                  </Table.Cell>
                </Table.Row>}
              </Table.Body>
            </Table>
          </div>
        </div>

        <Modal open={deleteConfirmOpen} onClose={() => setDeleteConfirmOpen(false)}>
          <Header>{__("Delete Zone")}</Header>
          <Modal.Content>{__("Are you sure you want to delete zone '{{name}}'?", {name: selectedZone?.name})}</Modal.Content>
          <Modal.Actions>
            <Button negative onClick={() => onDeleteZone()}>
              <Icon name="trash"/>{__("Delete")}
            </Button>
            <Button onClick={() => setDeleteConfirmOpen(false)}>
              <Icon name="cancel"/>{__("Cancel")}
            </Button>
          </Modal.Actions>
        </Modal>

        {deleteLoading && <Loading text={__("Updating...")}/>}

        <Help markdown={HelpMD}/>
      </WithQueryStatus>
    </Segment> :
    <CreateUpdateZone zone={selectedZone} onBack={() => navigate(-1)}/>;
};

export default Zones;
