import React, { useState, useEffect, useRef, useMemo } from "react";
import { Segment, Form, Button, Icon, Message, Table, Popup, Tab, Modal, Header } from "semantic-ui-react";
import classNames from "classnames";
import produce from "immer";
import {
  ZonesQuery,
  useZonesQuery,
  useCreateZoneMutation,
  useUpdateZoneMutation,
  ZonesDocument,
  LabelsDocument,
  DeviceFunctionalAspectType,
  LabelsQuery,
  RealmObjectType, PoliciesQuery, PoliciesDocument, LabelType, useDeviceListByAspectTypesQuery, BaseDeviceType, HealthStatus, DevicesByAspectTypesShortDocument, SensorListByAspectTypeDocument
} from "generated/graphql";
import { PropType } from "utils";
import type { ArrayElement } from "@solid/types";
import WithQueryStatus from "components/WithQueryStatus";
import Loading from "components/Loading";
import IconButton from "components/IconButton";
import CameraList, { ShowHideFilterEvent, SelectionStyle } from "components/CameraList";
import Help from "components/Help";
import ListText, { ListItem } from "components/Admin/Helpers/ListText";
import { getListItemIcon, updateZoneInDeviceCache } from "@core/actions";
import {Log} from "@solid/libs/log";
import {__} from "@solid/libs/i18n";
import { useApolloClient} from "@apollo/client";

import "./style.css";
import HelpMD from "./help.md";


type CreateUpdateZoneProps = {
  zone?: Zone;
  onBack?: () => void;
};

type Zone = PropType<ZonesQuery, "zones">[0];

type Device = {
  id: string;
  name: string;
  deviceType?: BaseDeviceType;
  status: boolean;
};

const deviceQueryTypes = [{ type: DeviceFunctionalAspectType.Media }, { type: DeviceFunctionalAspectType.Sensor }];

const CreateUpdateZone = ({ zone, onBack }: CreateUpdateZoneProps) => {
  const client = useApolloClient();
  const { data: zonesData, error: zonesError, loading: zonesLoading } = useZonesQuery();
  const { data: devicesData, error: devicesError, loading: devicesLoading } = useDeviceListByAspectTypesQuery({ variables: { types: deviceQueryTypes } });
  const [assignedDevices, setAssignedDevices] = useState<Device[]>([]);
  const assignedDeviceIds = useMemo(() => assignedDevices.map(dev => dev.id), [assignedDevices]);
  const [createZone, { error: createError, loading: createLoading }] = useCreateZoneMutation();
  const [updateZone, { error: updateError, loading: updateLoading }] = useUpdateZoneMutation();
  const [name, setName] = useState(zone?.name ?? "");
  const [nameError, setNameError] = useState("");
  const [availDevicesOpen, setAvailDevicesOpen] = useState(false);
  const [selectedAvailableDevices, setSelectedAvailableDevices] = useState<string[]>([]);
  const showHideFilterEventRef = useRef(new ShowHideFilterEvent());
  const deviceMap = useMemo(() => {
    const map = new Map<string, Device>();
    if (devicesData) {
      for (const device of devicesData.devicesByAspectTypes) {
        map.set(device.id, {
          id: device.id,
          name: device.name,
          deviceType: device.deviceType || undefined,
          status: !device.enabled || device.healthStatus !== HealthStatus.Normal
        });
      }
    }
    return map;
  }, [devicesData]);
  const hiddenDeviceIds = useMemo(() => {
    const idSet = new Set<string>(assignedDeviceIds);
    if (zonesData) {
      const thisZoneId = zone?.id;
      for (const zone of zonesData.zones) {
        if (zone.id !== thisZoneId) {
          for (const device of zone.devices) {
            idSet.add(device.id);
          }
        }
      }
    }
    return Array.from(idSet.keys());
  }, [zonesData, assignedDeviceIds]);
  const [activeTab, setActiveTab] = useState(0);
  const [changesDialogOpen, setChangesDialogOpen] = useState(false);
  const backOnUpdateRef = useRef(false);
  const nextTabRef = useRef(0);
  const prevTabRef = useRef(-1);
  const tabCount = 2;

  useEffect(() => {
    setName(zone?.name ?? "");
    const assignedDevices = getAssignedDevices(zone?.devices ?? []);
    setAssignedDevices(assignedDevices);
  }, [zone]);

  useEffect(() => {
    if (updateError && prevTabRef.current >= 0) {
      setActiveTab(prevTabRef.current);
      prevTabRef.current = -1;
    }
  }, [updateError]);

  const cancel = () => {
    onBack && onBack();
  };

  function getNameError(name: string): string {
    let error = "";
    const zoneId = zone?.id;
    if (!name.trim()) {
      error = __("Zone name should be not empty");
    } else
    if (zonesData?.zones.find(zone => zone.id !== zoneId && zone.name.toLocaleUpperCase() === name.trim().toLocaleUpperCase())) {
      error = __("Zone with the same name already exists");
    }
    return error;
  }

  function onNameInput(e: React.FormEvent<HTMLInputElement>): void {
    const name = e.currentTarget.value;
    setName(name);
    setNameError(getNameError(name));
  }

  async function onCreateUpdate(backOnUpdate = false): Promise<void> {
    if (!zone) {
      for (let i = 0; i < tabCount; i++) {
        if (!validateTab(i)) {
          setActiveTab(i);
          return;
        }
      }

      const response = await createZone({
        variables: { zone: { name: name.trim(), addDeviceIds: assignedDeviceIds } }
      });

      // update cache
      // update ZonesDocument, LabelsDocument
      const zoneId = response.data?.createZone.id;
      if (zoneId) {
        // update zones
        const zonesCacheData = client.readQuery<ZonesQuery>({
          query: ZonesDocument
        });

        const newZone: ArrayElement<ZonesQuery["zones"]> = {
          __typename: "Zone",
          id: zoneId,
          name: name.trim(),
          devices: assignedDevices.map((device) => {
            return {
              __typename: "ObjectDescriptor",
              id: device.id,
              name: device.name
            };
          }),
        };

        const zonesCache = zonesCacheData?.zones ?? [];
        const zones = produce(zonesCache, (draft) => {
          draft.push(newZone);
        });

        client.writeQuery<ZonesQuery>({
          query: ZonesDocument,
          data: {
            zones
          }
        });

        // update labels
        const labelsCacheData = client.readQuery<LabelsQuery>({
          query: LabelsDocument
        });

        const newLabel: ArrayElement<LabelsQuery["labels"]> = {
          __typename: "Label",
          id: zoneId,
          name: name.trim(),
          type: LabelType.Zone,
          objects: assignedDevices.map((device) => {
            return {
              __typename: "LabelObject",
              objectId: device.id,
              type: RealmObjectType.Device
            };
          }),
        };

        const labelsCache = labelsCacheData?.labels ?? [];
        const labels = produce(labelsCache, (draft) => {
          draft.push(newLabel);
        });

        client.writeQuery<LabelsQuery>({
          query: LabelsDocument,
          data: {
            labels
          }
        });

        // update devices
        if (assignedDevices.length > 0) {
          const deviceZone = { id: zoneId, name: name.trim() };
          const addDeviceIds = assignedDevices.map(dev => dev.id);
          updateZoneInDeviceCache(deviceZone, addDeviceIds, DeviceFunctionalAspectType.Media, DevicesByAspectTypesShortDocument, client);
          updateZoneInDeviceCache(deviceZone, addDeviceIds, DeviceFunctionalAspectType.Sensor, SensorListByAspectTypeDocument, client);
        }

        if (response.data?.createZone.warning) {
          Log.error(response.data.createZone.warning);
        }
        onBack && onBack();
      }
    } else {
      if (!validateTab(activeTab)) {
        return;
      }

      backOnUpdateRef.current = backOnUpdate;
      prevTabRef.current = activeTab;

      const zoneId = zone.id;

      let response;
      switch (activeTab) {
        case 0:
          response = await updateZone({
            variables: { id: zone.id, zone: { name: name.trim() } }
          });

          // update cache
          // update ZonesDocument, PoliciesDocument, LabelsDocument
          if (response.data?.updateZone) {
            // update zones
            const zonesCacheData = client.readQuery<ZonesQuery>({
              query: ZonesDocument
            });

            const zonesCache = zonesCacheData?.zones ?? [];
            const zoneIndex = zonesCache.findIndex((zone) => zone.id === zoneId);
            const zones = produce(zonesCache, (draft) => {
              draft[zoneIndex].name = name.trim();
            });

            client.writeQuery<ZonesQuery>({
              query: ZonesDocument,
              data: {
                zones
              }
            });

            // update policies
            const policiesCacheData = client.readQuery<PoliciesQuery>({
              query: PoliciesDocument
            });

            const policiesCache = policiesCacheData?.policies ?? [];
            if (policiesCache.length > 0) {
              const policies = produce(policiesCache, (draft) => {
                for (const policy of draft) {
                  for (const statement of policy.statements) {
                    for (let i = 0; i < statement.resources.length; i++) {
                      const resource = statement.resources[i];
                      if (resource.id === zoneId) {
                        resource.name = name.trim();
                        break;
                      }
                    }
                  }
                }
              });

              client.writeQuery<PoliciesQuery>({
                query: PoliciesDocument,
                data: {
                  policies
                }
              });
            }

            // update labels
            const labelsCacheData = client.readQuery<LabelsQuery>({
              query: LabelsDocument
            });

            const labelsCache = labelsCacheData?.labels ?? [];
            if (labelsCache.length > 0) {
              const labelIndex = labelsCache.findIndex((label) => label.type === LabelType.Zone && label.id === zoneId);
              if (labelIndex >= 0) {
                const labels = produce(labelsCache, (draft) => {
                  draft[labelIndex].name = name.trim();
                });

                client.writeQuery<LabelsQuery>({
                  query: LabelsDocument,
                  data: {
                    labels
                  }
                });
              }
            }
          }
          break;
        case 1:
          const addDeviceIds = assignedDeviceIds.filter(id => !zone.devices.some(dev => dev.id === id));
          const removeDeviceIds = zone.devices.filter(dev => !assignedDeviceIds.includes(dev.id)).map(dev => dev.id);
          response = await updateZone({
            variables: {
              id: zone.id,
              zone: {
                addDeviceIds,
                removeDeviceIds
              }
            }
          });

          // update cache
          // update ZonesDocument, LabelsDocument

          // update zones
          const zonesCacheData = client.readQuery<ZonesQuery>({
            query: ZonesDocument
          });

          const zonesCache = zonesCacheData?.zones ?? [];
          const zoneIndex = zonesCache.findIndex((zone) => zone.id === zoneId);
          const zones = produce(zonesCache, (draft) => {
            draft[zoneIndex].devices = assignedDevices.map((device) => { return { __typename: "ObjectDescriptor", id: device.id, name: device.name }; });
          });

          client.writeQuery<ZonesQuery>({
            query: ZonesDocument,
            data: {
              zones
            }
          });

          // update labels
          const labelsCacheData = client.readQuery<LabelsQuery>({
            query: LabelsDocument
          });

          const labelsCache = labelsCacheData?.labels ?? [];
          if (labelsCache.length > 0) {
            const labelIndex = labelsCache.findIndex((label) => label.type === LabelType.Zone && label.id === zoneId);
            if (labelIndex >= 0) {
              const labels = produce(labelsCache, (draft) => {
                draft[labelIndex].objects = assignedDevices.map((device) => { return { __typename: "LabelObject", objectId: device.id, type: RealmObjectType.Device }; });
              });

              client.writeQuery<LabelsQuery>({
                query: LabelsDocument,
                data: {
                  labels
                }
              });
            }
          }

          // update devices
          if (addDeviceIds.length > 0) {
            const deviceZone = { id: zone.id, name: zone.name };
            updateZoneInDeviceCache(deviceZone, addDeviceIds, DeviceFunctionalAspectType.Media, DevicesByAspectTypesShortDocument, client);
            updateZoneInDeviceCache(deviceZone, addDeviceIds, DeviceFunctionalAspectType.Sensor, SensorListByAspectTypeDocument, client);
          }
          if (removeDeviceIds.length > 0) {
            updateZoneInDeviceCache(null, removeDeviceIds, DeviceFunctionalAspectType.Media, DevicesByAspectTypesShortDocument, client);
            updateZoneInDeviceCache(null, removeDeviceIds, DeviceFunctionalAspectType.Sensor, SensorListByAspectTypeDocument, client);
          }

          break;
      }

      if (response?.data?.updateZone) {
        prevTabRef.current = -1;
        backOnUpdateRef.current && onBack && onBack();
      }
    }
  }

  function onActiveTabChange(index: number | string | undefined): void {
    if (typeof index !== "number") {
      return;
    }

    if (!zone) {
      if (index < activeTab || validateTab(activeTab)) {
        setActiveTab(index);
      }
    } else {
      if (!hasChangesOnTab(activeTab)) {
        setActiveTab(index);
        return;
      }

      if (!validateTab(activeTab)) {
        return;
      }

      nextTabRef.current = index;
      setChangesDialogOpen(true);
    }
  }

  function hasChangesOnTab(tabIndex: number): boolean {
    if (!zone) {
      switch (tabIndex) {
        case 0: return !!name.trim();
        case 1: return assignedDeviceIds.length > 0;
        default: return false;
      }
    } else {
      switch (tabIndex) {
        case 0: return name.trim() !== zone.name;
        case 1: return zone.devices.some(dev => !assignedDeviceIds.includes(dev.id)) ||
          assignedDeviceIds.some(id => !zone.devices.some(dev => dev.id === id));
        default: return false;
      }
    }
  }

  function validateTab(tabIndex: number): boolean {
    let result = true;
    switch (tabIndex) {
      case 0:
        const nameError = getNameError(name);
        if (nameError) {
          setNameError(nameError);
          result = false;
        }
        break;
    }
    return result;
  }

  function discardTabChanges(tabIndex: number): void {
    switch (tabIndex) {
      case 0:
        setName(zone?.name ?? "");
        setNameError("");
        break;
      case 1:
        const assignedDevices = getAssignedDevices(zone?.devices ?? []);
        setAssignedDevices(assignedDevices);
        break;
    }
  }

  function addDevices(): void {
    if (selectedAvailableDevices.length === 0) {
      return;
    }

    setAvailDevicesOpen(false);

    setAssignedDevices(value => value.concat(selectedAvailableDevices
      .map(id => deviceMap.get(id))
      .filter(dev => !!dev)
      .map(dev => dev!)));
    setSelectedAvailableDevices([]);
  }

  function removeDevice(device: Device): void {
    setAssignedDevices(value => value.filter(dev => dev.id !== device.id));
  }

  function getAssignedDevices(devices: { id: string, name: string }[]): Device[] {
    const assignedDevices: Device[] = [];
    for (const dev of devices) {
      const device = deviceMap.get(dev.id);
      device && assignedDevices.push(device);
    }
    return assignedDevices;
  }

  function getAssignedDevicesListText(device: Device): ListItem[] {
    if (!device.deviceType) return [{ ...device }];
    const icon = getListItemIcon(device.deviceType, device.status);
    return [{ ...device, faIcon: icon }];
  }

  const error = createError ?? updateError;

  const panes = [
    {
      menuItem: __("Zone Properties"),
      render: () => (
        <Tab.Pane>
          <Form className="CreateUpdateZone-Form" onSubmit={e => { e.preventDefault(); }}>
            <Form.Field
              control={Form.Input}
              label={__("Name")}
              placeholder={__("Name")}
              autoFocus
              value={name}
              error={nameError ? { content: nameError, pointing: "below" } : undefined}
              onInput={onNameInput}
            />
          </Form>
        </Tab.Pane>
      )
    },
    {
      menuItem: __("Devices"),
      render: () => (
        <Tab.Pane>
          <div className={classNames("CreateUpdateZone-Devices", { "AvailOpen": availDevicesOpen })}>
            <div className="CreateUpdateZone-AssignedDevices">
              <div className="CreateUpdateZone-TopTableButtons">
                <Button onClick={() => setAvailDevicesOpen(true)}>
                  {__("Assign Devices")}
                </Button>
              </div>
              <div className="CreateUpdateZone-Table">
                <Table celled compact>
                  <Table.Header>
                    <Table.Row>
                      <Table.HeaderCell/>
                      <Table.HeaderCell width={16}>{__("Name")}</Table.HeaderCell>
                    </Table.Row>
                  </Table.Header>

                  <Table.Body>
                    {assignedDevices.map(device =>
                      <Table.Row key={device.id}>
                        <Table.Cell collapsing>
                          <Popup trigger={
                            <Icon name="remove" className="CreateUpdateZone-IconButton" onClick={() => removeDevice(device)}/>
                            }
                            content={__("Remove Device from Zone")}
                          />
                        </Table.Cell>
                        <Table.Cell>
                          <ListText items={getAssignedDevicesListText(device)} icons />
                        </Table.Cell>
                      </Table.Row>)}

                    {assignedDevices.length === 0 &&
                    <Table.Row>
                      <Table.Cell colSpan={2} textAlign="center">
                        {__("No devices assigned")}
                      </Table.Cell>
                    </Table.Row>}
                  </Table.Body>
                </Table>
              </div>
            </div>

            {availDevicesOpen &&
            <div className="CreateUpdateZone-AvailableDevices">
              <div className="CreateUpdateZone-CameraListHeader">
                {__("Select devices to assign")}
                <IconButton
                  icon="search"
                  hint={__("Show/Hide Devices Filter")}
                  hintPosition="bottom right"
                  onClick={() => showHideFilterEventRef.current.publish({})}
                />
              </div>
              <CameraList
                deviceTypes={[{ type: DeviceFunctionalAspectType.Media }, { type: DeviceFunctionalAspectType.Sensor }]}
                selectable
                selectionStyle={SelectionStyle.Checkbox}
                multiselect
                selectedIds={selectedAvailableDevices}
                onSelectedMultiple={selection => setSelectedAvailableDevices(selection)}
                hiddenIds={hiddenDeviceIds}
                allFilters
                allowLabelHierarchyEdit
                openMultipleBranches
                showHideFilterEvent={showHideFilterEventRef.current}
              />
              <div className="CreateUpdateZone-BottomTableButtons">
                <Button positive disabled={selectedAvailableDevices.length === 0} onClick={addDevices}>{__("Select")}</Button>
                <Button onClick={() => setAvailDevicesOpen(false)}>{__("Cancel")}</Button>
              </div>
            </div>}
          </div>
        </Tab.Pane>
      )
    }
  ];

  return (
    <Segment className="CreateUpdateZone">
      <WithQueryStatus error={zonesError || devicesError} loading={zonesLoading || devicesLoading}>
        {!!error && <Message error content={error.message}/>}

        <div className="CreateUpdateZone-TopButtons">
          <Button onClick={() => cancel()}>
            <Icon name="cancel"/>{__("Cancel")}
          </Button>
          {!zone &&
          <>
            <Button disabled={activeTab <= 0} onClick={() => onActiveTabChange(activeTab - 1)}>
              <Icon name="arrow alternate circle left"/>{__("Back")}
            </Button>
            {activeTab < tabCount - 1 &&
            <Button positive onClick={() => onActiveTabChange(activeTab + 1)}>
              <Icon name="arrow alternate circle right"/>{__("Next")}
            </Button>}
          </>}
          {(!!zone || activeTab >= tabCount - 1) &&
          <Button
            positive
            disabled={!!zone && !hasChangesOnTab(activeTab)}
            onClick={() => onCreateUpdate(true)}
          >
            <Icon name={!zone ? "plus" : "check"}/>{!zone ? __("Create") : __("Save")}
          </Button>}
        </div>

        <Tab
          panes={panes}
          className="CreateUpdateZone-Tab"
          activeIndex={activeTab}
          onTabChange={(e, { activeIndex }) => onActiveTabChange(activeIndex)}
        />

        <Modal open={changesDialogOpen} onClose={() => setChangesDialogOpen(false)}>
          <Header>{__("Unsaved Changes")}</Header>
          <Modal.Content>{__("Zone has unsaved changes. Would you like to save the changes?")}</Modal.Content>
          <Modal.Actions>
            <Button positive onClick={() => {
              onCreateUpdate();
              setActiveTab(nextTabRef.current);
              setChangesDialogOpen(false);
            }}>
              <Icon name="check"/>{__("Save")}
            </Button>
            <Button negative onClick={() => {
              discardTabChanges(activeTab);
              setActiveTab(nextTabRef.current);
              setChangesDialogOpen(false);
            }}>
              <Icon name="undo"/>{__("Discard")}
            </Button>
            <Button onClick={() => setChangesDialogOpen(false)}>
              <Icon name="cancel"/>{__("Cancel")}
            </Button>
          </Modal.Actions>
        </Modal>

        {(createLoading || updateLoading) && <Loading text={__("Updating...")}/>}

        <Help markdown={HelpMD}/>
      </WithQueryStatus>
    </Segment>
  );
};

export default CreateUpdateZone;
