import React, { useCallback, useEffect, useMemo, useState } from "react";
import { Button, Checkbox, Icon, Popup, Table } from "semantic-ui-react";
import classNames from "classnames";
import { ObjectDescriptor, RealmObjectType, useSetsLazyQuery } from "@generated/graphql";
import { __ } from "@solid/libs/i18n";
import { DeviceSet } from "../Sets";
import WithQueryStatus from "components/WithQueryStatus";

import "./style.css";
import { FontAwesomeIcon } from "@fortawesome/react-fontawesome";

type ControlSetPanelsProps = {
  deviceSets?: ObjectDescriptor[];
  setDeviceSets: (sets: ObjectDescriptor[]) => void,
};

const ControlSetPanels = ({deviceSets, setDeviceSets}: ControlSetPanelsProps) => {
  const [getSets, { data, loading, error }] = useSetsLazyQuery();
  const [setList, setSetList] = useState<DeviceSet[]>([]);
  const [assignedSets, setAssignedSets] = useState<ObjectDescriptor[]>([]);
  const [selectedSetIdSet, setSelectedSetIdSet] = useState(new Set<string>());
  const [availSetOpen, setAvailSetOpen] = useState(false);
  const assignedSetIds = useMemo(() => assignedSets.map(set => set.id), [assignedSets]);
  const hiddenSetIds = useMemo(() => {
    const idSet = new Set<string>(assignedSetIds);
    return Array.from(idSet.keys());
  }, [data, assignedSetIds]);

  useEffect(() => {
    if (data && data.sets) {
      const setList = getSetList(data.sets, hiddenSetIds);
      setSetList(setList);
    }
  }, [data, assignedSets]);

  useEffect(() => {
    if (deviceSets && deviceSets.length !== 0) {
      setAssignedSets(deviceSets);
    }
  }, [deviceSets]);

  function getSetList(setList: DeviceSet[], hiddenIds: string[] = []): DeviceSet[] {
    const list = setList
      .filter(set => (!hiddenIds || !hiddenIds.includes(set.id)) && !set.isSystemManaged)
      .sort((a, b) => a.name.localeCompare(b.name, undefined, { sensitivity: "base" }))
      ?? [];
    return list;
  }

  const openAssigningSets = useCallback(() => {
    if (setList.length === 0) {
      getSets();
    }
    setAvailSetOpen(true);
  }, [setList]);

  const removeSet = (set: ObjectDescriptor) => {
    const newAssignedSets = assignedSets.filter(assignedSet => assignedSet.id !== set.id);
    setAssignedSets(newAssignedSets);
    setDeviceSets(newAssignedSets);
  };

  function getObjectSet(set: DeviceSet): ObjectDescriptor {
    return {
      __typename: "ObjectDescriptor",
      id: set.id,
      name: set.name,
      type: RealmObjectType.Set
    };
  }

  function selectUnselectSet(set: DeviceSet, select: boolean): void {
    setSelectedSetIdSet(value => {
      const newValue = new Set<string>(value);
      if (select) {
        newValue.add(set.id);
      } else {
        newValue.delete(set.id);
      }
      return newValue;
    });
  }

  function addSets(): void {
    const newAssignedSets: ObjectDescriptor[] = assignedSets
      .concat(setList
        .map(set => getObjectSet(set))
        .filter(set => selectedSetIdSet.has(set.id))
        .map(set => ({ ...set })));
    setAssignedSets(newAssignedSets);
    setDeviceSets(newAssignedSets);
    setSelectedSetIdSet(new Set<string>());
  }

  return (
    <div className={classNames("ControlSetPanel", { "AvailOpen": availSetOpen })}>

      <div className="ControlSetPanel-TopPanel">
        <div className="ControlSetPanel-TopPanel-Button">
          <Button onClick={() => openAssigningSets()}>{__("Assign Sets")}</Button>
        </div>
        {availSetOpen && <div className="ControlSetPanel-TopPanel-Header">{__("Select sets to assign")}</div>}
      </div>
      <div className="ControlSetPanel-SetsContent">
        <div className="ControlSetPanel-AssignedSets">
          <div className="ControlSetPanel-Table">
            <Table celled compact>
              <Table.Header>
                <Table.Row>
                  <Table.HeaderCell/>
                  <Table.HeaderCell width={16}>{__("Name")}</Table.HeaderCell>
                </Table.Row>
              </Table.Header>

              <Table.Body>
                {assignedSets.map(set =>
                  <Table.Row key={set.id}>
                    <Table.Cell collapsing>
                      {!set.isSystemManaged &&
                        <Popup trigger={
                          <Icon name="remove" className="ControlSetPanel-IconButton" onClick={() => removeSet(set)}/>
                              }
                          content={__("Remove set from device")}
                            />
                      }
                    </Table.Cell>
                    <Table.Cell>
                      <FontAwesomeIcon icon="circle" />
                      <span>&nbsp;{set.name}</span>
                    </Table.Cell>
                  </Table.Row>)}

                {assignedSets.length === 0 &&
                <Table.Row>
                  <Table.Cell colSpan={2} textAlign="center">
                    {__("No sets assigned")}
                  </Table.Cell>
                </Table.Row>}
              </Table.Body>
            </Table>
          </div>
        </div>
        {availSetOpen &&
        <div className="ControlSetPanel-AvailableSets">
          <WithQueryStatus error={error} loading={loading}>
            <div className="ControlSetPanel-Table">
              <Table celled compact>
                <Table.Header>
                  <Table.Row>
                    <Table.HeaderCell/>
                    <Table.HeaderCell width={16}>{__("Name")}</Table.HeaderCell>
                  </Table.Row>
                </Table.Header>
                <Table.Body>
                  {setList
                    .map(set =>
                      <Table.Row key={set.id}>
                        <Table.Cell collapsing>
                          <Checkbox checked={selectedSetIdSet.has(set.id)} onChange={(e, data) => selectUnselectSet(set, !!data.checked)}/>
                        </Table.Cell>
                        <Table.Cell className="ControlSetsPanel-SetCell">
                          <FontAwesomeIcon icon="circle" />
                          <span>&nbsp;{set.name}</span>
                        </Table.Cell>
                      </Table.Row>)}

                  {setList.length === 0 &&
                  <Table.Row>
                    <Table.Cell colSpan={2} textAlign="center">
                      {__("No available sets found")}
                    </Table.Cell>
                  </Table.Row>}
                </Table.Body>
              </Table>
            </div>
          </WithQueryStatus>
          <div className="ControlSetsPanel-BottomTableButtons">
            <Button positive disabled={selectedSetIdSet.size === 0 || loading} onClick={addSets}>{__("Select")}</Button>
            <Button onClick={() => setAvailSetOpen(false)}>{__("Cancel")}</Button>
          </div>
        </div>}
      </div>
    </div>
  );
};

export default ControlSetPanels;
