import React, { useEffect, useImperativeHandle, useMemo, useRef, useState } from "react";
import { Segment } from "semantic-ui-react";

import { __ } from "@solid/libs";

import { ComponentWithFormRef, DeviceProperties } from "components/Admin/Sensors/SensorSettings";
import { AutoForm, AutoLayout, FieldValues, FormSchema } from "components/AutoForm";
import { LatLng, LatLngInput } from "@generated/graphql";
import GEOMap from "components/GEOMap";

import "./style.css";

type SensorPositionProps = {
  sensor?: DeviceProperties;
  mountMap?: boolean;
  onChange: (property: DeviceProperties) => void;
};

const SensorPosition = React.forwardRef<ComponentWithFormRef, SensorPositionProps>(({ sensor, mountMap, onChange }, ref) => {
  const [position, setPosition] = useState<LatLngInput | null | undefined>(sensor?.position);

  const positionSchema = useMemo(() => {
    const schema: FormSchema = [
      {
        name: "lat",
        label: __("Lat"),
        type: "float",
        minValue: -90,
        maxValue: 90
      },
      {
        name: "lng",
        label: __("Lng"),
        type: "float",
        minValue: -180,
        maxValue: 180
      }
    ];
    return schema;
  }, [sensor]);

  const positionValues = useMemo(() => {
    return { ...sensor?.position };
  }, [sensor]);

  const positionFormRef = useRef<AutoForm>(null);

  useEffect(() => {
    sensor?.position && setPosition(sensor.position);
  }, [sensor]);

  useImperativeHandle(ref, () => ({
    getFormRef() {
      return positionFormRef.current;
    }
  }));

  function setNewPosition(position: LatLngInput) {
    positionFormRef.current?.setValue("lat", position.lat);
    positionFormRef.current?.setValue("lng", position.lng);
    setPosition(position);
    updatePosition(position);
  }

  function onGISChange(name: string, value: any, values: FieldValues, form: AutoForm): void {
    const newValue = parseFloat(value);
    const newPosition = { ...position, ...{ [name]: newValue } };
    form.setValue(name, newValue);
    updatePosition(newPosition);
    setPosition(prevPosition => ({ ...prevPosition, ...{ [name]: newValue } }));
  }

  function updatePosition(position: LatLngInput) {
    if (typeof position.lat === "number" && typeof position.lng === "number") {
      const latLng: LatLng = {
        lat: position.lat,
        lng: position.lng,
        alt: position.alt
      };
      onChange({ position: latLng });
    }
  }

  return (
    <Segment className="SensorPosition">
      <AutoForm
        ref={positionFormRef}
        className="SensorPosition-TemplateForm"
        schema={positionSchema}
        values={positionValues}
        onChange={onGISChange}
      >
        <div className="SensorPosition-FormLayout">
          <div className="SensorPosition-FormColumn1">
            <AutoLayout names={["lat"]}/>
          </div>
          <div className="SensorPosition-FormColumn2">
            <AutoLayout names={["lng"]}/>
          </div>
        </div>
      </AutoForm>

      { mountMap && <GEOMap id={sensor?.id} setPosition={setNewPosition} position={position ?? undefined} isView={false} /> }
    </Segment>
  );
});

export default SensorPosition;
