import React, { useEffect, useRef, useState } from "react";
import { Button, Icon, Popup, Ref, Segment } from "semantic-ui-react";

import { __ } from "@solid/libs";
import { SensorShort } from "@core/actions";
import { SensorListByAspectTypeQuery } from "@generated/graphql";
import { ApolloError } from "@apollo/client";

import { WidgetProps } from "components/Widgets";
import WidgetTitle from "components/WidgetTitle";
import ListFilter from "components/ListFilter";
import ButtonsWrapper from "../ButtonsWrapper";
import SensorTable from "./SensorTable";

import "./style.css";

type SensorListProps = {
  onSelect: (device: SensorShort) => void;
  onDelete: (device: SensorShort) => void;
  onCreate: () => void;
  data?: SensorListByAspectTypeQuery,
  loading?: boolean,
  error?: ApolloError,
  createdId?: string;
  avatarId?: string;
} & WidgetProps;

const SensorList = ({
  data,
  loading,
  error,
  onCreate,
  onSelect,
  onDelete,
  cellProps,
  setCellProps,
  createdId,
  avatarId
}: SensorListProps) => {
  const [filterVisible, setFilterVisible] = useState(false);
  const [searchText, setSearchText] = useState("");
  const [sensorList, setSensorList] = useState<SensorShort[]>([]);

  const rootRef = useRef<HTMLDivElement>(null);

  useEffect(() => {
    if (!setCellProps || avatarId) return;

    setCellProps({ title:
  <WidgetTitle
    title={cellProps?.title}
    controls={
      <Popup
        content={filterVisible ? __("Hide Filter") : __("Show Filter")}
        position="bottom right"
        trigger={
          <Icon
            name="search"
            onClick={() => setFilterVisible(visible => !visible)}
          />
        }
      />
    }
  />
    });
  }, [setCellProps, filterVisible, avatarId]);

  useEffect(() => {
    if (data && data.devicesByAspectTypes) {
      const sensorList = getSensorList(searchText);
      setSensorList(sensorList);
    }
  }, [data, searchText]);

  function getSensorList(searchText: string): SensorShort[] {
    const text = searchText?.toLocaleUpperCase();
    const list: SensorShort[] = data
      ?.devicesByAspectTypes
      ?.filter(sensor =>
        (!text
          || sensor.name.toLocaleUpperCase().includes(text)
          || sensor.location?.toLocaleUpperCase().includes(text)
          || sensor.platform?.name?.toLocaleUpperCase().includes(text)
          || sensor.zone?.name.toLocaleUpperCase().includes(text))
        &&
        (!avatarId || sensor.platform?.id === avatarId)
      )
      .sort((a, b) => {
        if (!createdId) {
          return a.name.localeCompare(b.name, undefined, { sensitivity: "base" });
        }
        return createdId === a.id ? -1 : 1;
      })
      ?? [];

    return list;
  }

  return (
    <Ref innerRef={rootRef}>
      <Segment className="SensorList" error={error}>
        { filterVisible && !avatarId &&
        <ListFilter
          filterTextPlaceholder={__("Filter by text")}
          searchText={searchText}
          nameFilter
          rootRef={rootRef}
          onSearchTextChange={text => setSearchText(text)}
        /> }

        <ButtonsWrapper>
          <Button className="create-device" icon="plus" content={__("Add")} onClick={onCreate}/>
        </ButtonsWrapper>

        { avatarId &&
        <ListFilter
          filterTextPlaceholder={__("Filter by text")}
          searchText={searchText}
          nameFilter
          rootRef={rootRef}
          onSearchTextChange={text => setSearchText(text)}
        /> }

        <SensorTable onDelete={onDelete} onSelect={onSelect} list={sensorList} createdDeviceId={createdId} loading={loading} />
      </Segment>
    </Ref>
  );
};

export default SensorList;
