import React, { useEffect, useMemo, useState } from "react";
import { Button, Dropdown, DropdownItemProps, Form, Icon, Message, Segment } from "semantic-ui-react";
import { CameraMetricsQueryVariables, useAvatarMetricsLazyQuery, useCameraMetricsLazyQuery } from "@generated/graphql";
import { __ } from "@solid/libs/i18n";
import { UUID, Type  } from "@solid/types";

import "./style.css";

type ReportMetricsProps = {
  metricType: Type.avatar | Type.camera,
  deviceId: UUID,
  isMetricsUpdated: boolean,
  updateMetrics: boolean,
  start: number,
  end: number,
  createReport: (value: string) => void,
};

const ReportMetrics = ({
  metricType,
  deviceId,
  isMetricsUpdated,
  updateMetrics,
  start,
  end,
  createReport,
}: ReportMetricsProps) => {
  const [getCameraMetrics, {data: cameraData, loading: cameraLoading, error: cameraError}] = useCameraMetricsLazyQuery({fetchPolicy: "network-only"});
  const [getAvatarMetrics, {data: avatarData, loading: avatarLoading, error: avatarError}] = useAvatarMetricsLazyQuery({fetchPolicy: "network-only"});
  const [metricValue, setMetricValue] = useState<string>();

  const metricOptions = useMemo(() => {
    if (!metricType) {
      return [];
    }

    if (metricType === Type.camera) {
      const options = cameraData?.cameraMetrics.reduce((acc: DropdownItemProps[], metric) => {
        acc.push({
          key: metric.key,
          text: metric.name,
          value: `${metric.__typename},${metric.key},${metric.name},${metric.dimension},${metric.dfaId}`
        });
        return acc;
      }, []);
      return options;
    }
    const options = avatarData?.avatarMetrics.reduce((acc: DropdownItemProps[], metric) => {
      acc.push({
        key: metric.key,
        text: metric.name,
        value: `${metric.__typename},${metric.key},${metric.name},${metric.dimension}`
      });
      return acc;
    }, []);

    return options;
  }, [cameraData, avatarData, metricType]);

  useEffect(() => {
    if (updateMetrics) {
      if (metricType === Type.camera) {
        const input: CameraMetricsQueryVariables = {
          id: deviceId,
          from: new Date(start),
          to: new Date(end),
        };

        getCameraMetrics({variables: input});
      } else {
        getAvatarMetrics();
      }
    }
  }, [updateMetrics]);

  return (
    <Segment className="ReportMetric" loading={avatarLoading || cameraLoading} error={avatarError || cameraError} key={`RM_${deviceId}`}>
      {metricOptions && metricOptions.length > 0 &&
        <Form className="ReportMetric-Form" onSubmit={(e) => { e.preventDefault(); }}>
          <Form.Field>
            <Dropdown
              placeholder={__("Select type of report")}
              selection
              disabled={!isMetricsUpdated}
              options={metricOptions}
              value={metricValue}
              onChange={(e, { value }) => {
                setMetricValue(value as string);
              }} />
          </Form.Field>
          <Form.Field>
            <Button
              positive
              disabled={!isMetricsUpdated || !metricValue}
              onClick={() => createReport(metricValue as string)}>
              <Icon name="plus" />
              {__("Create Report")}
            </Button>
          </Form.Field>
        </Form>}
      {metricOptions && metricOptions.length === 0 &&
      <Message warning>
        <Message.Header>{__("There are no reports available for this period")}</Message.Header>
        <p>{__("Try to choose a different period")}</p>
      </Message>}
    </Segment>
  );
};

export default ReportMetrics;
