import React, { useState, useEffect, useRef, useMemo } from "react";
import { Segment, Form, Button, Icon, Message, Popup, Tab, Table, Checkbox, Modal, Header } from "semantic-ui-react";
import { FontAwesomeIcon } from "@fortawesome/react-fontawesome";
import { IconProp } from "@fortawesome/fontawesome-svg-core";
import classNames from "classnames";
import {
  PoliciesQuery,
  ZonesQuery,
  useActionsQuery,
  useAccessLevelsQuery,
  useZonesQuery,
  useDevicesByAspectTypesShortQuery,
  useCreatePolicyStatementMutation,
  useUpdatePolicyStatementMutation,
  PoliciesDocument,
  PolicyStatement,
  StatementEffect,
  RealmObjectType,
  RealmObject,
  DeviceFunctionalAspectType,
  PolicyStatementInput,
  useSetsQuery,
  HealthStatus
} from "generated/graphql";
import { PropType } from "utils";
import { DeviceShort, getListItemIcon } from "@core/actions";
import WithQueryStatus from "components/WithQueryStatus";
import Loading from "components/Loading";
import {levelView, levelManage, levelControl} from "../utils";
import IconButton from "components/IconButton";
import CameraList, { ShowHideFilterEvent, SelectionStyle } from "components/CameraList";
import TreeView, { TreeNode } from "components/TreeView";
import { DeviceSet } from "components/Admin/Sets/Sets";
import Help from "components/Help";
import {Log} from "@solid/libs/log";
import {__} from "@solid/libs/i18n";
import HelpMD from "./help.md";
import {useApolloClient} from "@apollo/client";
import produce from "immer";

import "./style.css";

type CreateUpdateStatementProps = {
  policy: Policy;
  statement?: PolicyStatement;
  onBack?: () => void;
};

type Policy = PropType<PoliciesQuery, "policies">[0];
type Zone = PropType<ZonesQuery, "zones">[0];
type DeviceWithRealmObjectType = DeviceShort & {
  type: RealmObjectType
};

const deviceQueryTypes = [{ type: DeviceFunctionalAspectType.Media }, { type: DeviceFunctionalAspectType.Sensor }];

const CreateUpdateStatement = ({ policy, statement, onBack }: CreateUpdateStatementProps) => {
  const client = useApolloClient();
  const { data: levelsData, error: levelsError, loading: levelsLoading } = useAccessLevelsQuery();
  const { data: actionsData, error: actionsError, loading: actionsLoading } = useActionsQuery();
  const { data: zonesData, error: zonesError, loading: zonesLoading } = useZonesQuery();
  const { data: setsData, error: setsError, loading: setsLoading } = useSetsQuery();
  const { data: devicesData, error: devicesError, loading: devicesLoading } = useDevicesByAspectTypesShortQuery({ variables: { types: deviceQueryTypes } });
  const [createStatement, { error: createError, loading: createLoading }] = useCreatePolicyStatementMutation();
  const [updateStatement, { error: updateError, loading: updateLoading }] = useUpdatePolicyStatementMutation();
  const [name, setName] = useState(statement?.name ?? "");
  const [nameError, setNameError] = useState("");
  const [effect, setEffect] = useState(statement?.effect ?? StatementEffect.Allow);
  const [allResources, setAllResources] = useState<boolean>(!!statement?.allResources);
  const [availResourcesOpen, setAvailResourcesOpen] = useState(false);
  const [assignedResources, setAssignedResources] = useState<RealmObject[]>([]);
  const assignedResourceIdSet = useMemo(() => new Set<string>(assignedResources.map(res => res.id)), [assignedResources]);
  const availableZones = useMemo(() => {
    const zones = zonesData?.zones ?? [];
    return zones
      .filter(zone => !assignedResourceIdSet.has(zone.id))
      .sort((a, b) => a.name.localeCompare(b.name, undefined, {sensitivity: "base"}));
  }, [zonesData, assignedResourceIdSet]);
  const availableSets = useMemo(() => {
    const sets = setsData?.sets ?? [];
    return sets
      .filter(set => !assignedResourceIdSet.has(set.id))
      .sort((a, b) => a.name.localeCompare(b.name, undefined, {sensitivity: "base"}));
  }, [setsData, assignedResourceIdSet]);
  const [selectedZoneIdSet, setSelectedZoneIdSet] = useState(new Set<string>());
  const [selectedSetIdSet, setSelectedSetIdSet] = useState(new Set<string>());
  const [selectedAvailableDevices, setSelectedAvailableDevices] = useState<string[]>([]);
  const hiddenDeviceIds = useMemo(() => {
    let ids: string[] = assignedResources.filter(res => res.type === RealmObjectType.Device).map(res => res.id);
    const zones = zonesData?.zones.filter(zone => assignedResourceIdSet.has(zone.id)) ?? [];
    for (const zone of zones) {
      ids = ids.concat(zone.devices.map(dev => dev.id));
    }
    return ids;
  }, [assignedResources, zonesData, assignedResourceIdSet]);
  const showHideFilterEventRef = useRef(new ShowHideFilterEvent());
  const deviceMap = useMemo(() => {
    const map = new Map<string, DeviceWithRealmObjectType>();
    if (devicesData) {
      for (const device of devicesData.devicesByAspectTypes) {
        map.set(device.id, { ...device, type: RealmObjectType.Device });
      }
    }
    return map;
  }, [devicesData]);
  const levels = useMemo(() => levelsData ? Array.from(levelsData.accessLevels).sort((a, b) => a.order - b.order) : [], [levelsData]);
  const actions = useMemo(() => actionsData ?
    actionsData.actions
      .filter(action => action.resourceTypes.includes(RealmObjectType.Device) && action.deviceFunctionalAspectTypes.includes(DeviceFunctionalAspectType.Media))
      .sort((a, b) => a.description.localeCompare(b.description, undefined, { sensitivity: "base" })) : [], [actionsData]);
  // const [openLevels, setOpenLevels] = useState(new Set<string>());
  const [checkedActions, setCheckedActions] = useState(new Set<string>());
  const [activeTab, setActiveTab] = useState(0);
  const [changesDialogOpen, setChangesDialogOpen] = useState(false);
  const backOnUpdateRef = useRef(false);
  const nextTabRef = useRef(0);
  const prevTabRef = useRef(-1);
  const tabCount = 3;
  const treeNodes = useMemo(() => getTreeNodes(), [zonesData, assignedResources, assignedResourceIdSet, setsData]);

  useEffect(() => {
    setName(statement?.name ?? "");
    setEffect(statement?.effect ?? StatementEffect.Allow);
    setAllResources(!!statement?.allResources);
    setCheckedActions(statement ? new Set<string>(statement.actions) : new Set<string>());
    setAssignedResources(statement ? statement.resources : []);
  }, [policy]);

  useEffect(() => {
    if (updateError && prevTabRef.current >= 0) {
      setActiveTab(prevTabRef.current);
      prevTabRef.current = -1;
    }
  }, [updateError]);

  /*useEffect(() => {
    const viewActions = actions.filter(action => action.accessLevel.level === levelView);
    const viewChecked = viewActions.every(action => checkedActions.has(action.action));
    const manageActions = actions.filter(action => action.accessLevel.level === levelManage);
    const manageChecked = manageActions.every(action => checkedActions.has(action.action));

    if (effect === StatementEffect.Allow && manageChecked && !viewChecked) {
      checkUncheckLevel(levelView, true);
    }
    if (effect === StatementEffect.Deny && viewChecked && !manageChecked) {
      checkUncheckLevel(levelManage, true);
    }
  }, [effect]);*/

  function getNameError(name: string): string {
    let error = "";
    const id = statement?.id;
    if (!name.trim()) {
      error = __("Statement name should be not empty");
    } else
    if (policy.statements.find(stmt => stmt.id !== id && stmt.name.toLocaleUpperCase() === name.trim().toLocaleUpperCase())) {
      error = __("Statement with the same name already exists");
    }
    return error;
  }

  function onNameInput(e: React.FormEvent<HTMLInputElement>): void {
    const name = e.currentTarget.value;
    setName(name);
    setNameError(getNameError(name));
  }

  async function onCreateUpdate(backOnUpdate = false): Promise<void> {
    const policyId = policy.id;
    const statementName = name.trim();
    if (!statement) {
      for (let i = 0; i < tabCount; i++) {
        if (!validateTab(i)) {
          setActiveTab(i);
          return;
        }
      }

      const newStatement: PolicyStatementInput = {
        name: statementName,
        effect,
        actions: Array.from(checkedActions.keys()),
        addResourceIds: !allResources ? assignedResources.map(res => res.id) : [],
        allResources
      };
      const response = await createStatement({
        variables: {
          policyId,
          statement: newStatement
        }
      });

      // update cache
      const statementId = response.data?.createPolicyStatement.id;
      if (statementId) {
        const data = client.readQuery<PoliciesQuery>({
          query: PoliciesDocument
        });

        const policiesCache = data?.policies ?? [];
        const policiesIndex = policiesCache.findIndex((policy) => {
          return policy.id === policyId;
        });

        const policies = produce(policiesCache, (draft) => {
          draft[policiesIndex].statements.push({
            __typename: "PolicyStatement",
            id: statementId,
            name: statementName,
            effect,
            actions: Array.from(checkedActions.keys()),
            allResources,
            resources: !allResources ? assignedResources.map(res => { return  {__typename: "ObjectDescriptor", id: res.id, name: res.name, type: res.type}; }) : []
          });
        });

        client.writeQuery<PoliciesQuery>({
          query: PoliciesDocument,
          data: {
            policies
          }
        });

        if (response.data?.createPolicyStatement.warning) {
          Log.error(response.data.createPolicyStatement.warning);
        }
        onBack && onBack();
      }
    } else {
      if (!validateTab(activeTab)) {
        return;
      }

      backOnUpdateRef.current = backOnUpdate;
      prevTabRef.current = activeTab;

      const statementId = statement.id;

      let response;
      switch (activeTab) {
        case 0:
          response = await updateStatement({
            variables: {
              statementId,
              statement: {
                name: statementName,
                effect
              }
            }
          });

          // update cache
          if (response.data?.updatePolicyStatement) {
            const data = client.readQuery<PoliciesQuery>({
              query: PoliciesDocument
            });

            const policiesCache = data?.policies ?? [];
            const policiesIndex = policiesCache.findIndex((policy) => policy.id === policyId);
            const statementIndex = policiesCache[policiesIndex].statements.findIndex((statement) => statement.id === statementId);

            const policies = produce(policiesCache, (draft) => {
              draft[policiesIndex].statements[statementIndex].name = statementName;
              draft[policiesIndex].statements[statementIndex].effect = effect;
            });

            client.writeQuery<PoliciesQuery>({
              query: PoliciesDocument,
              data: {
                policies
              }
            });
          }
          break;
        case 1:
          response = await updateStatement({
            variables: {
              statementId,
              statement: {
                actions: Array.from(checkedActions.keys())
              }
            }
          });

          // update cache
          if (response.data?.updatePolicyStatement) {
            const data = client.readQuery<PoliciesQuery>({
              query: PoliciesDocument
            });

            const policiesCache = data?.policies ?? [];
            const policiesIndex = policiesCache.findIndex((policy) => policy.id === policyId);
            const statementIndex = policiesCache[policiesIndex].statements.findIndex((statement) => statement.id === statementId);

            const policies = produce(policiesCache, (draft) => {
              draft[policiesIndex].statements[statementIndex].actions = Array.from(checkedActions.keys());
            });

            client.writeQuery<PoliciesQuery>({
              query: PoliciesDocument,
              data: {
                policies
              }
            });
          }
          break;
        case 2:
          response = await updateStatement({
            variables: {
              statementId,
              statement: {
                allResources,
                addResourceIds: !allResources ?
                  assignedResources.filter(res => !statement.resources.some(r => r.id === res.id)).map(res => res.id) : [],
                removeResourceIds: allResources ?
                  statement.resources.map(res => res.id) :
                  statement.resources.filter(res => !assignedResources.some(r => r.id === res.id)).map(res => res.id)
              }
            }
          });

          // update cache
          if (response.data?.updatePolicyStatement) {
            const data = client.readQuery<PoliciesQuery>({
              query: PoliciesDocument
            });

            const policiesCache = data?.policies ?? [];
            const policiesIndex = policiesCache.findIndex((policy) => policy.id === policyId);
            const statementIndex = policiesCache[policiesIndex].statements.findIndex((statement) => statement.id === statementId);

            const policies = produce(policiesCache, (draft) => {
              draft[policiesIndex].statements[statementIndex].allResources = allResources;
              // draft[policiesIndex].statements[policiesIndex].resources = statement.resources.map(res => {__typename: "ObjectDescriptor", id: res.id, name: res.name, type: res.type })
              draft[policiesIndex].statements[statementIndex].resources = !allResources ? assignedResources.map(res => { return  {__typename: "ObjectDescriptor", id: res.id, name: res.name, type: res.type}; }) : [];
            });

            client.writeQuery<PoliciesQuery>({
              query: PoliciesDocument,
              data: {
                policies
              }
            });
          }
          break;
      }

      if (response?.data?.updatePolicyStatement) {
        prevTabRef.current = -1;
        // backOnUpdateRef.current && onBack && onBack();
      }
    }
  }

  function onActiveTabChange(index: number | string | undefined): void {
    if (typeof index !== "number") {
      return;
    }

    if (!statement) {
      if (index < activeTab || validateTab(activeTab)) {
        setActiveTab(index);
      }
    } else {
      if (!hasChangesOnTab(activeTab)) {
        setActiveTab(index);
        return;
      }

      if (!validateTab(activeTab)) {
        return;
      }

      nextTabRef.current = index;
      setChangesDialogOpen(true);
    }
  }

  function hasChangesOnTab(tabIndex: number): boolean {
    if (!statement) {
      switch (tabIndex) {
        case 0: return !!name.trim() || effect === StatementEffect.Deny;
        case 1: return checkedActions.size > 0;
        case 2: return assignedResources.length > 0 || allResources;
        default: return false;
      }
    } else {
      switch (tabIndex) {
        case 0: return name.trim() !== statement.name || effect !== statement.effect;
        case 1: return JSON.stringify(Array.from(statement.actions).sort()) !== JSON.stringify(Array.from(checkedActions.keys()).sort());
        case 2: return allResources !== statement.allResources ||
          statement.resources.some(resource => !assignedResources.some(res => res.id === resource.id)) ||
          assignedResources.some(resource => !statement.resources.some(res => res.id === resource.id));
        default: return false;
      }
    }
  }

  function validateTab(tabIndex: number): boolean {
    let result = true;
    switch (tabIndex) {
      case 0:
        const nameError = getNameError(name);
        if (nameError) {
          setNameError(nameError);
          result = false;
        }
        break;
    }
    return result;
  }

  function discardTabChanges(tabIndex: number): void {
    switch (tabIndex) {
      case 0:
        setName(statement?.name ?? "");
        setNameError("");
        setEffect(statement?.effect ?? StatementEffect.Allow);
        break;
      case 1:
        setCheckedActions(statement ? new Set<string>(statement.actions) : new Set<string>());
        break;
      case 2:
        setAllResources(!!statement?.allResources);
        setAssignedResources(statement ? statement.resources : []);
        break;
    }
  }

  function removeResource(resource: RealmObject): void {
    setAssignedResources(value => value.filter(res => res.id !== resource.id));
  }

  function selectUnselectZone(zone: Zone, select: boolean): void {
    setSelectedZoneIdSet(value => {
      const newValue = new Set<string>(value);
      if (select) {
        newValue.add(zone.id);
      } else {
        newValue.delete(zone.id);
      }
      return newValue;
    });
  }

  function selectUnselectSet(set: DeviceSet, select: boolean): void {
    setSelectedSetIdSet(value => {
      const newValue = new Set<string>(value);
      if (select) {
        newValue.add(set.id);
      } else {
        newValue.delete(set.id);
      }
      return newValue;
    });
  }

  function addSets(): void {
    setAssignedResources(value => value.concat(availableSets
      .filter(set => selectedSetIdSet.has(set.id))
      .map(set => ({ ...set, type: RealmObjectType.Set }))));
    setSelectedSetIdSet(new Set<string>());
    setAvailResourcesOpen(false);
  }

  function addZones(): void {
    setAssignedResources(value => value.concat(availableZones
      .filter(zone => selectedZoneIdSet.has(zone.id))
      .map(zone => ({ ...zone, type: RealmObjectType.Zone }))));
    setSelectedZoneIdSet(new Set<string>());
    setAvailResourcesOpen(false);
  }

  function addDevices(): void {
    setAssignedResources(value => value.concat(selectedAvailableDevices
      .map(id => deviceMap.get(id))
      .filter(dev => !!dev)
      .map(dev => dev!)));
    setSelectedAvailableDevices([]);
    setAvailResourcesOpen(false);
  }

  // function openCloseLevel(level: string): void {
  //   setOpenLevels(value => {
  //     const newValue = new Set<string>(value);
  //     if (newValue.has(level)) {
  //       newValue.delete(level);
  //     } else {
  //       newValue.add(level);
  //     }
  //     return newValue;
  //   });
  // }

  function checkUncheckLevel(level: string, check: boolean): void {
    setCheckedActions(value => {
      const newValue = new Set<string>(value);
      for (const action of actions.filter(action => action.accessLevel.level === level)) {
        if (check) {
          newValue.add(action.action);
        } else {
          newValue.delete(action.action);
        }
      }
      return newValue;
    });
    if (effect === StatementEffect.Allow) {
      if (level === levelView && check) {
        checkUncheckLevel(levelControl, false);
        checkUncheckLevel(levelManage, false);
      }
      if (level === levelControl && check) {
        checkUncheckLevel(levelView, false);
        checkUncheckLevel(levelManage, false);
      }
      if (level === levelManage && check) {
        checkUncheckLevel(levelView, false);
        checkUncheckLevel(levelControl, false);
      }
    } else
    if (level === levelView && check) {
      checkUncheckLevel(levelManage, true);
    }
  }

  // function checkUncheckAction(action: string, check: boolean): void {
  //   setCheckedActions(value => {
  //     const newValue = new Set<string>(value);
  //     if (check) {
  //       newValue.add(action);
  //     } else {
  //       newValue.delete(action);
  //     }
  //     return newValue;
  //   });
  // }

  // function getActionListContent(level: string): React.ReactNode {
  //   const levelActions = actions.filter(action => action.accessLevel.level === level);
  //   const columns: number[][] = [];
  //   const columnCount = 4;
  //   const rowCount = levelActions.length / columnCount + (levelActions.length % columnCount === 0 ? 0 : 1);
  //   for (let col = 0; col < columnCount; col++) {
  //     columns.push([]);
  //     for (let row = 0; row < rowCount; row++) {
  //       const index = row * columnCount + col;
  //       columns[col].push(index < levelActions.length ? index : -1);
  //     }
  //   }
  //   return (
  //     <div className="CreateUpdateStatement-ActionList">
  //       {columns.map((row, index) =>
  //         <div key={index} className="CreateUpdateStatement-ActionColumn">
  //           {row.map(index => {
  //             if (index < 0) {
  //               return undefined;
  //             }
  //             const action = levelActions[index];
  //             return (
  //               <Checkbox
  //                 key={action.action}
  //                 label={action.description}
  //                 checked={checkedActions.has(action.action)}
  //                 readOnly//={policy.isSystemManaged}
  //                 onChange={(e, { checked }) => checkUncheckAction(action.action, !!checked)}
  //               />
  //             );
  //           })}
  //         </div>)}
  //     </div>
  //   );
  // }

  function getTreeNodes(): TreeNode[] {
    const nodes: TreeNode[] = [];
    const zones = zonesData?.zones
      .filter(zone => assignedResourceIdSet.has(zone.id))
      .sort((a, b) => a.name.localeCompare(b.name, undefined, { sensitivity: "base" }));
    if (zones) {
      for (const zone of zones) {
        const node: TreeNode = {
          id: zone.id,
          content: getTreeNodeContent({ ...zone, type: RealmObjectType.Zone }, 0),
          nodes: zone.devices.map(dev => ({
            id: dev.id,
            content: getTreeNodeContent({ ...dev, type: RealmObjectType.Device }, 1),
            nodes: []
          }))
        };
        nodes.push(node);
      }
    }
    const sets = setsData?.sets
      .filter(set => assignedResourceIdSet.has(set.id))
      .sort((a, b) => a.name.localeCompare(b.name, undefined, { sensitivity: "base" }));
    if (sets) {
      for (const set of sets) {
        const node: TreeNode = {
          id: set.id,
          content: getTreeNodeContent({ ...set, type: RealmObjectType.Set}, 0),
          nodes: set.devices.map(dev => ({
            id: dev.id,
            content: getTreeNodeContent({ ...dev, type: RealmObjectType.Device }, 1),
            nodes: []
          }))
        };
        nodes.push(node);
      }
    }
    for (const resource of assignedResources
      .filter(res => res.type === RealmObjectType.Device)
      .sort((a, b) => a.name.localeCompare(b.name, undefined, { sensitivity: "base" }))) {
      nodes.push({
        id: resource.id,
        content: getTreeNodeContent(resource, 0),
        nodes: []
      });
    }
    return nodes;
  }

  function getTreeNodeContent(resource: RealmObject, level: number): React.ReactNode {
    const device = deviceMap.get(resource.id);
    let deviceIcon: IconProp = "video";
    if (device && device.deviceType) {
      const deviceStatus = !device.enabled || device.healthStatus !== HealthStatus.Normal;
      deviceIcon = getListItemIcon(device.deviceType, deviceStatus);
    }
    return (
      <div className="CreateUpdateStatement-TreeNodeContent">
        {resource.type === RealmObjectType.Device && <FontAwesomeIcon icon={deviceIcon} />}
        {resource.type === RealmObjectType.Zone && <Icon name="th large"/>}
        {resource.type === RealmObjectType.Set && <Icon name="circle outline"/>}
        <div className="name">&nbsp;{resource.name}</div>
        {!policy.isSystemManaged && level === 0 &&
        <div className="right">
          <Popup trigger={
            <Icon name="trash alternate" onClick={() => removeResource(resource)}/>
            }
            content={__("Remove Resource")}
          />
        </div>}
      </div>
    );
  }

  const error = createError ?? updateError;

  /* eslint-disable react/jsx-indent */
  const panesResources = [
    {
      menuItem: __("Zones"),
      pane:
        <Tab.Pane key="zones">
          <div className="CreateUpdateStatement-AvailableZones">
            <div className="CreateUpdateStatement-Header">{__("Select zones to assign")}</div>
            <div className="CreateUpdateStatement-Table">
              <Table celled compact>
                <Table.Header>
                  <Table.Row>
                    <Table.HeaderCell/>
                    <Table.HeaderCell width={16}>{__("Name")}</Table.HeaderCell>
                  </Table.Row>
                </Table.Header>

                <Table.Body>
                  {availableZones
                    .map(zone =>
                      <Table.Row key={zone.id}>
                        <Table.Cell collapsing>
                          <Checkbox checked={selectedZoneIdSet.has(zone.id)} onChange={(e, data) => selectUnselectZone(zone, !!data.checked)}/>
                        </Table.Cell>
                        <Table.Cell>
                          <div className="CreateUpdateStatement-ZoneCell">
                            <Icon name="th large"/>
                            <span>&nbsp;{zone.name}</span>
                          </div>
                        </Table.Cell>
                      </Table.Row>)}

                  {availableZones.length === 0 &&
                  <Table.Row>
                    <Table.Cell colSpan={2} textAlign="center">
                      {__("No available zones found")}
                    </Table.Cell>
                  </Table.Row>}
                </Table.Body>
              </Table>
            </div>
            <div className="CreateUpdateStatement-BottomTableButtons">
              <Button positive disabled={selectedZoneIdSet.size === 0} onClick={addZones}>{__("Select")}</Button>
              <Button onClick={() => setAvailResourcesOpen(false)}>{__("Cancel")}</Button>
            </div>
          </div>
        </Tab.Pane>
    },
    {
      menuItem: __("Sets"),
      pane:
      <Tab.Pane key="sets">
      <div className="CreateUpdateStatement-AvailableZones">
        <div className="CreateUpdateStatement-Header">{__("Select sets to assign")}</div>
        <div className="CreateUpdateStatement-Table">
          <Table celled compact>
            <Table.Header>
              <Table.Row>
                <Table.HeaderCell/>
                <Table.HeaderCell width={16}>{__("Name")}</Table.HeaderCell>
              </Table.Row>
            </Table.Header>

            <Table.Body>
              {availableSets
                .map(set =>
                  <Table.Row key={set.id}>
                    <Table.Cell collapsing>
                      <Checkbox checked={selectedSetIdSet.has(set.id)} onChange={(e, data) => selectUnselectSet(set, !!data.checked)}/>
                    </Table.Cell>
                    <Table.Cell>
                      <div className="CreateUpdateStatement-ZoneCell">
                        <Icon name="circle outline"/>
                        <span>&nbsp;{set.name}</span>
                      </div>
                    </Table.Cell>
                  </Table.Row>)}

              {availableSets.length === 0 &&
              <Table.Row>
                <Table.Cell colSpan={2} textAlign="center">
                  {__("No available sets found")}
                </Table.Cell>
              </Table.Row>}
            </Table.Body>
          </Table>
        </div>
        <div className="CreateUpdateStatement-BottomTableButtons">
          <Button positive disabled={selectedSetIdSet.size === 0} onClick={addSets}>{__("Select")}</Button>
          <Button onClick={() => setAvailResourcesOpen(false)}>{__("Cancel")}</Button>
        </div>
      </div>
      </Tab.Pane>
    },
    {
      menuItem: __("Devices"),
      pane:
        <Tab.Pane key="devices">
          <div className="CreateUpdateStatement-AvailableDevices">
            <div className="CreateUpdateStatement-CameraListHeader">
              {__("Select devices to assign")}
              <IconButton
                icon="search"
                hint={__("Show/Hide Devices Filter")}
                hintPosition="bottom right"
                onClick={() => showHideFilterEventRef.current.publish({})}
              />
            </div>
            <CameraList
              deviceTypes={[{ type: DeviceFunctionalAspectType.Media }, { type: DeviceFunctionalAspectType.Sensor }]}
              selectable
              selectionStyle={SelectionStyle.Checkbox}
              multiselect
              selectedIds={selectedAvailableDevices}
              onSelectedMultiple={selection => setSelectedAvailableDevices(selection)}
              hiddenIds={hiddenDeviceIds}
              allFilters
              allowLabelHierarchyEdit
              openMultipleBranches
              showHideFilterEvent={showHideFilterEventRef.current}
            />
            <div className="CreateUpdateStatement-BottomTableButtons">
              <Button positive disabled={selectedAvailableDevices.length === 0} onClick={() => addDevices()}>{__("Select")}</Button>
              <Button onClick={() => setAvailResourcesOpen(false)}>{__("Cancel")}</Button>
            </div>
          </div>
        </Tab.Pane>
    }
  ];
  /* eslint-enable react/jsx-indent */

  const panes = [
    {
      menuItem: __("Statement Properties"),
      render: () => (
        <Tab.Pane>
          <Form className="CreateUpdateStatement-Form" onSubmit={e => { e.preventDefault(); }}>
            <Form.Field
              control={Form.Input}
              label={__("Name")}
              placeholder={__("Name")}
              autoFocus
              value={name}
              error={nameError ? { content: nameError, pointing: "below" } : undefined}
              readOnly={policy.isSystemManaged}
              onInput={onNameInput}
            />
            <Form.Select
              label={__("Effect")}
              placeholder={__("Effect")}
              options={[
                { key: StatementEffect.Allow, value: StatementEffect.Allow, text: __("Allow") }
                //{ key: StatementEffect.Deny, value: StatementEffect.Deny, text: __("Deny") }, //TODO: Should be return
              ]}
              value={effect}
              readOnly={policy.isSystemManaged}
              onChange={(e, { value }) => setEffect(value === StatementEffect.Allow ? StatementEffect.Allow : StatementEffect.Deny)}
            />
          </Form>
        </Tab.Pane>
      )
    },
    {
      menuItem: __("Actions"),
      render: () => (
        <Tab.Pane>
          <div className="CreateUpdateStatement-Actions">
            {levels.map(level => {
              const levelActions = actions.filter(action => action.accessLevel.level === level.level);
              const checked = levelActions.every(action => checkedActions.has(action.action));
              const indeterminate = !checked && levelActions.some(action => checkedActions.has(action.action));
              return (
                <React.Fragment key={level.level}>
                  <div className="CreateUpdateStatement-AccessLevel">
                    {/* <Icon
                      name={openLevels.has(level.level) ? "triangle down" : "triangle right"}
                      onClick={() => openCloseLevel(level.level)}
                    /> */}
                    <Checkbox
                      label={level.description}
                      checked={checked}
                      indeterminate={indeterminate}
                      readOnly={policy.isSystemManaged}
                      onChange={(e, { checked }) => checkUncheckLevel(level.level, !!checked)}
                    />
                  </div>
                  {/* {openLevels.has(level.level) && getActionListContent(level.level)} */}
                </React.Fragment>
              );
            })}
          </div>
        </Tab.Pane>
      )
    },
    {
      menuItem: __("Resources"),
      render: () => (
        <Tab.Pane>
          <div className={classNames("CreateUpdateStatement-Resources", { "AvailOpen": availResourcesOpen })}>
            <div className="CreateUpdateStatement-ResourcesTop">
              <Checkbox
                label={__("All Resources")}
                checked={allResources}
                readOnly={policy.isSystemManaged}
                onChange={(e, { checked }) => setAllResources(!!checked)}
              />
            </div>
            {!allResources &&
            <div className="CreateUpdateStatement-ResourcesContent">
              <div className="CreateUpdateStatement-AssignedResources">
                {!policy.isSystemManaged &&
                <div className="CreateUpdateStatement-TopTableButtons">
                  <Button onClick={() => setAvailResourcesOpen(true)}>
                    {__("Assign Resources")}
                  </Button>
                </div>}
                {assignedResources.length === 0 ?
                  <div className="CreateUpdateStatement-NoResources">{__("No resources assigned")}</div> :
                  <TreeView
                    nodes={treeNodes}
                    editable={false}
                    selectable={false}
                  />}
              </div>
              {availResourcesOpen &&
              <Tab
                panes={panesResources}
                className="CreateUpdateStatement-AvailableResources"
                renderActiveOnly={false}
              />}
            </div>}
          </div>
        </Tab.Pane>
      )
    }
  ];

  return (
    <Segment className="CreateUpdateStatement">
      <WithQueryStatus
        error={levelsError || actionsError || zonesError || devicesError || setsError}
        loading={levelsLoading || actionsLoading || zonesLoading || devicesLoading || setsLoading}>
        {!!error && <Message error content={error.message}/>}

        <div className="CreateUpdateStatement-TopButtons">
          <Button onClick={() => onBack && onBack()}>
            <Icon name="cancel"/>{__("Cancel")}
          </Button>
          {!statement &&
          <>
            <Button disabled={activeTab <= 0} onClick={() => onActiveTabChange(activeTab - 1)}>
              <Icon name="arrow alternate circle left"/>{__("Back")}
            </Button>
            {activeTab < tabCount - 1 &&
            <Button positive onClick={() => onActiveTabChange(activeTab + 1)}>
              <Icon name="arrow alternate circle right"/>{__("Next")}
            </Button>}
          </>}
          {(!!statement || activeTab >= tabCount - 1) &&
          <Button
            positive
            disabled={!!statement && !hasChangesOnTab(activeTab)}
            onClick={() => onCreateUpdate(true)}
          >
            <Icon name={!statement ? "plus" : "check"}/>{!statement ? __("Create") : __("Save")}
          </Button>}
        </div>

        <Tab
          panes={panes}
          className="CreateUpdateStatement-Tab"
          activeIndex={activeTab}
          onTabChange={(e, { activeIndex }) => onActiveTabChange(activeIndex)}
        />

        <Modal open={changesDialogOpen} onClose={() => setChangesDialogOpen(false)}>
          <Header>{__("Unsaved Changes")}</Header>
          <Modal.Content>{__("Statement has unsaved changes. Would you like to save the changes?")}</Modal.Content>
          <Modal.Actions>
            <Button positive onClick={async () => {
              await onCreateUpdate();
              setActiveTab(nextTabRef.current);
              setChangesDialogOpen(false);
            }}>
              <Icon name="check"/>{__("Save")}
            </Button>
            <Button negative onClick={() => {
              discardTabChanges(activeTab);
              setActiveTab(nextTabRef.current);
              setChangesDialogOpen(false);
            }}>
              <Icon name="undo"/>{__("Discard")}
            </Button>
            <Button onClick={() => setChangesDialogOpen(false)}>
              <Icon name="cancel"/>{__("Cancel")}
            </Button>
          </Modal.Actions>
        </Modal>

        {(createLoading || updateLoading) && <Loading text={__("Updating...")}/>}

        <Help markdown={HelpMD}/>
      </WithQueryStatus>
    </Segment>
  );
};

export default CreateUpdateStatement;
