import React, { useState } from "react";
import { Modal, Button, Icon, List, SemanticICONS } from "semantic-ui-react";
import { IconProp } from "@fortawesome/fontawesome-svg-core";
import { FontAwesomeIcon } from "@fortawesome/react-fontawesome";
import { ReactSVG } from "react-svg";
import {__} from "@solid/libs/i18n";

import "./style.css";

export type ListItem = {
  id: string;
  name: string;
  icon?: SemanticICONS;
  faIcon?: IconProp;
  svgIcon?: string;
};

type ListTextProps = {
  items: ListItem[];
  maxItems?: number;
  modalHeader?: string;
  sort?: boolean;
  icons?: boolean;
};

const ListText = ({ items, maxItems = 3, modalHeader = "", sort = true, icons = false }: ListTextProps) => {
  const [modalOpen, setModalOpen] = useState(false);

  const listItems = sort ? Array.from(items)
    .sort((a, b) => a.name.localeCompare(b.name, undefined, { sensitivity: "base" })) : items;

  return (
    <>
      {!icons ?
        <>
          {listItems
            .slice(0, maxItems)
            .map((item, index) =>
              <span key={item.id}>
                {(index > 0 ? ", " : "") + item.name}
              </span>)}
          {items.length > maxItems && <span>, {__("and")} <a className="ListText-Link" onClick={() => setModalOpen(true)}>{items.length - maxItems} {__("more")}</a></span>}
        </> :
        <div className="ListText-IconsContent">
          {listItems
            .slice(0, maxItems)
            .map((item, index) =>
              <React.Fragment key={item.id}>
                {index > 0 && <span>,&nbsp;</span>}
                {!!item.icon && <Icon name={item.icon}/>}
                {!!item.faIcon && <FontAwesomeIcon icon={item.faIcon}/>}
                {!!item.svgIcon && <ReactSVG role="small-icon" src={item.svgIcon}/>}
                <span>&nbsp;{item.name}</span>
              </React.Fragment>)}
          {items.length > maxItems && <span>, {__("and")} <a className="ListText-Link" onClick={() => setModalOpen(true)}>{items.length - maxItems} {__("more")}</a></span>}
        </div>}

      {modalOpen &&
      <Modal closeIcon size="tiny" open={modalOpen} onClose={() => setModalOpen(false)}>
        <Modal.Header>{modalHeader}</Modal.Header>
        <Modal.Content scrolling>
          <List>
            {!icons ?
              listItems.map(item =>
                <List.Item key={item.id}>
                  {item.name}
                </List.Item>) :
              listItems.map(item =>
                <List.Item key={item.id}>
                  <div className="ListText-IconsContent">
                    {!!item.icon && <Icon name={item.icon}/>}
                    {!!item.faIcon && <FontAwesomeIcon icon={item.faIcon}/>}
                    {!!item.svgIcon && <ReactSVG role="small-icon" src={item.svgIcon}/>}
                    <span>&nbsp;{item.name}</span>
                  </div>
                </List.Item>)}
          </List>
        </Modal.Content>
        <Modal.Actions>
          <Button onClick={() => setModalOpen(false)}>
            <Icon name="close"/>{__("Close")}
          </Button>
        </Modal.Actions>
      </Modal>}
    </>
  );
};

export default ListText;
