import React, { FC, useEffect, useState } from "react";
import { AutoSizer, Column, Table } from "react-virtualized";
import { __ } from "@solid/libs";
import { Button, Checkbox, Dropdown, Segment } from "semantic-ui-react";
import { GatewayProbeEvent, SubscribedGatewayEventInput } from "@generated/graphql";
import ListFilter from "components/ListFilter";
import produce from "immer";
import "../../GatewayList/style.css";
import "./style.css";

export type EventType = GatewayProbeEvent & { type: string };

interface GatewayEventsProps {
  events: GatewayProbeEvent[];
  subscribedEvents?: SubscribedGatewayEventInput[];
  updateSubscribedEvents: (selectedEvent: SubscribedGatewayEventInput | SubscribedGatewayEventInput[], type: string) => void;
}

type EventTypeList = "NONE" | "ALERT" | "INFO";

type OptionType = {
  key: EventTypeList;
  value: EventTypeList;
  text: EventTypeList;
};

const optionList: OptionType[] = [
  {
    key: "NONE",
    value: "NONE",
    text: "NONE",
  },
  {
    key: "ALERT",
    value: "ALERT",
    text: "ALERT",
  },
  {
    key: "INFO",
    value: "INFO",
    text: "INFO",
  },
];

export const GatewayEvents: FC<GatewayEventsProps> = ({ events, subscribedEvents, updateSubscribedEvents }) => {
  const [mappedOnly, setMappedOnly] = useState<boolean>(false);
  const [formattedEvents, setFormattedEvents] = useState<EventType[]>([]);
  const [eventList, setEventList] = useState<EventType[]>([]);
  const [searchText, setSearchText] = useState<string>("");
  const [assignAll, setAssignAll] = useState<string>("NONE");

  useEffect(() => {
    setFormattedEvents(
      events
        .map((event) => {
          if (subscribedEvents) {
            const ev = subscribedEvents.find((subscribedEvent) => subscribedEvent.id === event.id);
            if (ev) {
              return {
                name: event.name,
                id: event.id,
                type: ev.type,
              };
            }
          }
          return {
            ...event,
            type: "NONE",
          };
        })
        .filter((event) => {
          if (mappedOnly && event.type !== "NONE") {
            return event;
          }
          if (!mappedOnly) {
            return event;
          }
          return null;
        })
    );
  }, [events, subscribedEvents, mappedOnly]);

  useEffect(() => {
    if (formattedEvents) filterEvents(searchText);
  }, [searchText, formattedEvents]);

  function filterEvents(searchText: string) {
    const text = searchText.toLocaleUpperCase();
    setEventList(
      formattedEvents
        .filter(
          (event) =>
            !text || event.name.toLocaleUpperCase().includes(text) || event.id.toLocaleUpperCase().includes(text) || event.type.toLocaleUpperCase().includes(text)
        )
        .sort((a, b) => a.id.localeCompare(b.id, undefined, { sensitivity: "base" }))
    );
  }

  function assignType(event: EventType | EventType[], type: string) {
    if (Array.isArray(event)) {
      setFormattedEvents(event.map((ev) => ({ ...ev, type })));
      updateSubscribedEvents(events.map((event) => ({id: event.id, type})), type);
    }
    else {
      produce(formattedEvents, (draft) => {
        const index = draft.findIndex((ev) => ev.id === event.id);
        draft[index].type = type;
      });
      updateSubscribedEvents({ id: event.id, type: event.type }, type);
    }
  }

  return (
    <Segment loading={false} className="GatewayTable-Wrapper">
      <div className="GatewayEvents__wrap">
        <div className="MenuWrap">
          <Checkbox toggle checked={mappedOnly} onChange={() => setMappedOnly((prev) => !prev)} />
          <span>{__("Show Mapped Events Only")}</span>
        </div>
        <div className="MenuWrap">
          <Dropdown
            options={optionList}
            value={assignAll}
            onChange={(e, { value }) => {
              typeof value === "string" && setAssignAll(value);
            }}
          />
          <Button
            onClick={() => {
              assignType(formattedEvents, assignAll);
            }}
          >
            {__("Assign to all")}
          </Button>
        </div>
      </div>
      <ListFilter
        searchText={searchText}
        nameFilter
        onSearchTextChange={(text) => {
          setSearchText(text);
        }}
        filterTextPlaceholder={__("Filter")}
      />
      <AutoSizer style={{ width: "100%", height: "100%", marginTop: "0.5rem" }}>
        {({ width, height }) => (
          <Table
            className="GatewayTable"
            width={width}
            height={height}
            rowHeight={34}
            headerHeight={58}
            rowCount={eventList.length}
            rowGetter={({ index }) => eventList[index]}
            rowClassName="GatewayRow"
          >
            <Column dataKey="id" label={__("ID")} width={10} flexGrow={1} />
            <Column dataKey="name" label={__("Name")} width={150} flexGrow={1} />
            <Column
              style={{ overflow: "initial" }}
              dataKey="mapped"
              label={__("Assigned Event Type")}
              width={10}
              flexGrow={1}
              cellRenderer={({ rowData }) => (
                <Dropdown
                  options={optionList}
                  value={rowData.type ? rowData.type : "NONE"}
                  onChange={(e, { value }) => {
                    typeof value === "string" && assignType(rowData, value);
                  }}
                />
              )}
            />
          </Table>
        )}
      </AutoSizer>
    </Segment>
  );
};
