import React, { FC, useEffect, useState } from "react";
import DevicesTable from "./DevicesTable/DevicesTable";
import { Segment } from "semantic-ui-react";
import { AssignCamera } from "./AssignCamera/AssignCamera";
import { AssociatedGatewayDevice, AssociatedGatewayDeviceInput, GatewayProbeDevice } from "@generated/graphql";
import { __ } from "@solid/libs";

import "../../GatewayList/style.css";

export type DeviceType = GatewayProbeDevice & { assignedCameras: string[] | undefined };

interface GatewayDevicesProps {
  devices: GatewayProbeDevice[];
  associatedDevices?: AssociatedGatewayDeviceInput[];
  updateAssociatedDevices: (selectedDevice: AssociatedGatewayDevice, cameras: string[]) => void;
}

const GatewayDevices: FC<GatewayDevicesProps> = ({ devices, associatedDevices, updateAssociatedDevices }) => {
  const [configActive, setConfigActive] = useState<AssociatedGatewayDevice | null>(/*{
    id: "",
    name: "test",
    associatedDevices: ["965dfbfa-babb-11ed-943b-4fd313210fa9", "755d898e-babb-11ed-943b-4fd313210fa9", "da28e4ba-4876-11ed-a714-836ca4775226"],
  }*/);

  const [deviceList, setDeviceList] = useState<AssociatedGatewayDevice[]>([]);

  function onAssign(device: AssociatedGatewayDevice) {
    setConfigActive(device);
  }

  useEffect(() => {
    setDeviceList(
      devices.map((device) => {
        if (associatedDevices) {
          const dev = associatedDevices.find((associatedDevices) => associatedDevices.id === device.id);
          if (dev) {
            return {
              name: device.name,
              id: device.id,
              associatedDevices: dev.associatedDevices,
            };
          }

          return {
            name: device.name,
            id: device.id,
            associatedDevices: [],
          };
        }
        return {
          name: device.name,
          id: device.id,
          associatedDevices: [],
        };
      })
    );
  }, [devices, associatedDevices]);


  function onUpdate(selectedDevice: AssociatedGatewayDevice, cameras: string[]) {
    setDeviceList((prev) =>
      prev.map((device) => {
        if (device.id === selectedDevice.id) {
          return {
            ...device,
            associatedDevices: cameras,
          };
        }
        return device;
      })
    );
    if (cameras) {
      updateAssociatedDevices(selectedDevice, cameras);
    }
  }

  return (
    <Segment className="GatewayList">
      {configActive ? (
        <AssignCamera device={configActive} onBack={() => setConfigActive(null)} updateDevice={(device, cameras) => onUpdate(device, cameras)} />
      ) : deviceList ? (
        <DevicesTable devices={deviceList} onAssign={(device) => onAssign(device)} />
      ) : (
        <span>{__("No devices")}</span>
      )}
    </Segment>
  );
};

export default GatewayDevices;
