import React, { FC, useEffect, useMemo, useRef, useState } from "react";
import { Button, Icon, Segment } from "semantic-ui-react";
import { __ } from "@solid/libs";
import { AssignedCameras } from "./AssignedCameras";
import ButtonsWrapper from "../../../ButtonsWrapper";
import { AssociatedGatewayDevice, BaseDeviceType, DeviceFunctionalAspectType, HealthStatus, useDeviceListByAspectTypesQuery } from "@generated/graphql";

import CameraList, { SelectionStyle, ShowHideFilterEvent } from "components/CameraList";
import IconButton from "components/IconButton";

import { useLocation } from "react-router-dom";
import "./style.css";

export type Device = {
  id: string;
  name: string;
  deviceType?: BaseDeviceType;
  status: boolean;
};

interface AssignCameraProps {
  device: AssociatedGatewayDevice;
  updateDevice: (device: AssociatedGatewayDevice, cameras: string[]) => void;
  onBack: () => void;
}

export const AssignCamera: FC<AssignCameraProps> = ({ device, onBack, updateDevice }) => {
  const { pathname } = useLocation();
  const showHideFilterEventRef = useRef(new ShowHideFilterEvent());
  const mountedRef = useRef<boolean>(true);

  const {
    data: devicesData,
    /*error: devicesError,
    loading: devicesLoading,*/
  } = useDeviceListByAspectTypesQuery({ variables: { types: [{ type: DeviceFunctionalAspectType.Media }] } });

  const [assignedDevices, setAssignedDevices] = useState<Device[]>([]);
  const assignedDeviceIds = useMemo(() => assignedDevices.map((dev) => dev.id), [assignedDevices]);
  const [selectedAvailableDevices, setSelectedAvailableDevices] = useState<string[]>([]);
  const [assignActive, setAssignActive] = useState<boolean>(false);
  const hiddenDeviceIds = useMemo(() => {
    const idSet = new Set<string>(assignedDeviceIds);
    return Array.from(idSet.keys());
  }, [device, assignedDeviceIds]);

  useEffect(() => {
    const assignedDevices = getAssignedDevices(device.associatedDevices);
    if (assignedDevices.length) setAssignedDevices(assignedDevices);

    if (!assignedDevices.length && pathname === "/view/gateway/add") setAssignActive(true);
  }, [device]);

  const deviceMap = useMemo(() => {
    const map = new Map<string, Device>();
    if (devicesData) {
      for (const device of devicesData.devicesByAspectTypes) {
        map.set(device.id, {
          id: device.id,
          name: device.name,
          deviceType: device.deviceType || undefined,
          status: !device.enabled || device.healthStatus !== HealthStatus.Normal,
        });
      }
    }
    return map;
  }, [devicesData]);

  function getAssignedDevices(devices: string[]): Device[] {
    const assignedDevices: Device[] = [];
    for (const dev of devices) {
      const device = deviceMap.get(dev);
      device && assignedDevices.push(device);
    }
    return assignedDevices;
  }

  function addDevices(): void {
    if (selectedAvailableDevices.length === 0) {
      return;
    }

    setAssignActive(false);

    setAssignedDevices((value) =>
      value.concat(
        selectedAvailableDevices
          .map((id) => deviceMap.get(id))
          .filter((dev) => !!dev)
          .map((dev) => dev!)
      )
    );
    setSelectedAvailableDevices([]);
  }

  function removeCamera(device: Device): void {
    setAssignedDevices((value) => value.filter((dev) => dev.id !== device.id));
  }

  useEffect(() => {
    if (mountedRef.current) {
      mountedRef.current = false;
    } else {
      updateDevice(device, assignedDeviceIds);
    }
  }, [assignedDevices]);

  return (
    <Segment className="Gateways AssignCamerasWrap">
      <ButtonsWrapper>
        <Button className="AssignCameras__back" onClick={onBack}>
          <Icon name="arrow alternate circle left" />
          {__("Back")}
        </Button>
        <Button className="AssignCameras__open" onClick={() => setAssignActive(true)}>{__("Assign Cameras")}</Button>
      </ButtonsWrapper>
      <div className="CameraToAssign">
        <AssignedCameras deviceName={device.name} cameras={assignedDevices} removeCamera={removeCamera} />
        {assignActive && (
          <div className="AssignCameras-AvailableDevices">
            <div className="AssignCameras-CameraListHeader">
              {__("Select devices to assign")}
              <IconButton icon="search" hint={__("Show/Hide Devices Filter")} hintPosition="top left" onClick={() => showHideFilterEventRef.current.publish({})} />
            </div>
            <CameraList
              deviceTypes={[{ type: DeviceFunctionalAspectType.Media }]}
              selectable
              selectionStyle={SelectionStyle.Checkbox}
              multiselect
              selectedIds={selectedAvailableDevices}
              onSelectedMultiple={(selection) => setSelectedAvailableDevices(selection)}
              hiddenIds={hiddenDeviceIds}
              allFilters
              allowLabelHierarchyEdit
              openMultipleBranches
              showHideFilterEvent={showHideFilterEventRef.current}
            />
            <div className="AssignCameras-BottomTableButtons">
              <Button positive disabled={selectedAvailableDevices.length === 0} onClick={addDevices}>
                {__("Select")}
              </Button>
              <Button onClick={() => setAssignActive(false)}>{__("Cancel")}</Button>
            </div>
          </div>
        )}
      </div>
    </Segment>
  );
};
