import React, { useEffect, useRef, useState } from "react";
import { WidgetProps } from "components/Widgets";
import { Button, Icon, Message, Popup, Ref, Segment } from "semantic-ui-react";
import { __ } from "@solid/libs";
import ButtonsWrapper from "../ButtonsWrapper";

import GatewayTable from "./GatewayTable";
import { GatewayListByAspectTypeQuery } from "@generated/graphql";
import WidgetTitle from "components/WidgetTitle";
import ListFilter from "components/ListFilter";
import { ApolloError } from "@apollo/client";
import { GatewayShort } from "@core/actions";

import "./style.css";
import {UUID} from "@solid/types";

type GatewayListProps = {
  data?: GatewayListByAspectTypeQuery;
  loading?: boolean;
  error?: ApolloError;
  onCreate: () => void;
  onSelect: (device: GatewayShort) => void;
  onDelete: (device: GatewayShort) => void;
  createdId?: UUID;
} & WidgetProps;

const GatewayList = ({ data, error, loading, onCreate, onSelect, onDelete, createdId, cellProps, setCellProps }: GatewayListProps) => {
  const rootRef = useRef<HTMLDivElement>(null);
  const [filterVisible, setFilterVisible] = useState(false);
  const [searchText, setSearchText] = useState<string>("");
  const [gatewayList, setGatewayList] = useState<GatewayShort[]>([]);

  useEffect(() => {
    if (data && data.devicesByAspectTypes) {
      const list = getGatewayList(searchText);
      setGatewayList(list);
    }
  }, [data, searchText]);

  function getGatewayList(searchText: string): GatewayShort[] {
    const text = searchText?.toLocaleUpperCase();
    const list: GatewayShort[] =
      data?.devicesByAspectTypes
        ?.filter((gateway) => !text || gateway.name.toLocaleUpperCase().includes(text) || gateway.platform?.name?.toLocaleUpperCase().includes(text))
        .sort((a, b) => {
          if (!createdId) {
            return a.name.localeCompare(b.name, undefined, { sensitivity: "base" });
          }
          return createdId === a.id ? -1 : 1;
        })
      ?? [];

    return list;
  }

  useEffect(() => {
    if (!setCellProps) return;
    setCellProps({
      title: (
        <WidgetTitle
          title={cellProps?.title}
          controls={
            <Popup
              content={filterVisible ? __("Hide Filter") : __("Show Filter")}
              position="bottom right"
              trigger={<Icon name="search" onClick={() => setFilterVisible((visible) => !visible)} />}
            />
          }
        />
      ),
    });
  }, [setCellProps, filterVisible]);

  return (
    <Ref innerRef={rootRef}>
      <Segment className="GatewayList" error={error}>
        {filterVisible && (
          <ListFilter
            filterTextPlaceholder={__("Filter by text")}
            searchText={searchText}
            nameFilter
            rootRef={rootRef}
            onSearchTextChange={(text) => setSearchText(text)}
          />
        )}
        <ButtonsWrapper>
          <Button className="create-device" icon="plus" content={__("Add")} onClick={onCreate} />
        </ButtonsWrapper>
        {!!error && <Message error content={error.message} />}
        {gatewayList ? <GatewayTable list={gatewayList} onSelect={onSelect} onDelete={onDelete} loading={loading} createdId={createdId}/> : <span>{__("There is no gateways")}</span>}
      </Segment>
    </Ref>
  );
};

export default GatewayList;
