import React, { useEffect, useCallback, Dispatch, SetStateAction, useState } from "react";
import { Button, Icon, Message, Modal, Segment } from "semantic-ui-react";
import {
  ChangeLocalStorageConfigMutationVariables,
  LocalStorageInput,
  useGetLocalStorageConfigQuery
} from "@generated/graphql";
import { UUID } from "@solid/types";
import { __ } from "@solid/libs";
import WithQueryStatus from "components/WithQueryStatus";
import LinkLocalStorageForm from "./LinkLocalStorageForm";

import "./style.css";

type LinkLocalStorageConfigProps = {
  deviceId: UUID;
  setIsLocalStorageModalOpen: Dispatch<SetStateAction<boolean>>;
  isModalOpen: boolean;
  setLocalStorage: (input: ChangeLocalStorageConfigMutationVariables) => void;
};

type LocalStorageInputKeys = keyof LocalStorageInput;
type LocalStorageInputValues = LocalStorageInput[LocalStorageInputKeys];

export type OnChangeLocalStorageValueProp = {
  [x: string]: LocalStorageInputValues;
};

export enum PoolSizeUnit {
  M = "MB",
  G = "GB",
  K = "KB",
}

const LinkLocalStorageConfig = ({deviceId, isModalOpen, setIsLocalStorageModalOpen, setLocalStorage}: LinkLocalStorageConfigProps) => {
  const { data, loading, error } = useGetLocalStorageConfigQuery({ fetchPolicy: "no-cache", variables: { deviceId } });
  const [input, setInput] = useState<LocalStorageInput>();
  const [confirm, setConfirm] = useState<boolean>(false);
  const [isValidLocalStorageConfig, setIsValidLocalStorageConfig] = useState<boolean>(true);

  useEffect(() => {
    if (data?.localStorage && !loading && !error) {
      const input: LocalStorageInput = {
        enabled: data.localStorage.enabled,
        strategy: data.localStorage.strategy,
        ver: data.localStorage.ver,
        pool: {
          size: data.localStorage.pool.size,
          enforce: data.localStorage.pool.enforce,
          path: data.localStorage.pool.path
        },
      };

      setInput(input);
    }
  }, [data, error, loading]);

  const saveLinkLocalStorage = useCallback(() => {
    if (!input || !deviceId) {
      return;
    }
    setConfirm(false);

    const localStorageConfigInput: ChangeLocalStorageConfigMutationVariables = {
      id: deviceId,
      input,
    };
    setLocalStorage(localStorageConfigInput);
    setIsLocalStorageModalOpen(false);
  }, [input, deviceId]);

  const onSaveDialog = useCallback(() => {
    if (!data?.localStorage || !input) {
      return;
    }
    if (
      (data?.localStorage!.enabled && !input.enabled) ||
      (data?.localStorage!.enabled && input.enabled && data?.localStorage!.pool.path !== input.pool.path)
    ) {
      setConfirm(true);
      return;
    }
    saveLinkLocalStorage();
  }, [input, data?.localStorage]);

  function onChangeLocalStorageValue(value: OnChangeLocalStorageValueProp) {
    if (!input) {
      return;
    }

    const newInput = {...input, ...value};
    const isValidLocalStorageConfig = validateLocalStorageConfig(newInput);
    if (isValidLocalStorageConfig) {
      setIsValidLocalStorageConfig(true);
    } else {
      setIsValidLocalStorageConfig(false);
    }
    setInput(newInput);
  }

  function validateLocalStorageConfig(input: LocalStorageInput): boolean {
    const { enabled, pool } = input;
    if (typeof enabled === "boolean" && !enabled) {
      return true;
    }

    const match = pool.size.match(/([0-9]+)([a-zA-Z]+)/);
    const isPoolSizeValid = match && Number(match[1]) > 0 && Object.keys(PoolSizeUnit).some(key => key === match[2]);
    const isValidPath = pool.path.length > 0;
    if (isPoolSizeValid && isValidPath) {
      return true;
    }

    return false;
  }

  return     (
    <Modal
      className="LinkLocalStorageConfig-Modal"
      onClose={() => setIsLocalStorageModalOpen(false)}
      onOpen={() => setIsLocalStorageModalOpen(true)}
      open={isModalOpen}>
      <Modal.Header>{__("Avatar Local Storage")}</Modal.Header>
      <Modal.Content className="LinkLocalStorageConfig-Content">
        {!confirm ?
          <Segment className="LinkLocalStorageConfig-Form-Wrapper" loading={loading}>
            <WithQueryStatus error={error}>
              {input && <LinkLocalStorageForm input={input} onChangeLocalStorageValue={onChangeLocalStorageValue}/>}
            </WithQueryStatus>
          </Segment> :
          <Message negative>
            <Message.Header>{__("All data stored before will be lost!")}</Message.Header>
            <p>{__("Are you sure want to continue?")}</p>
          </Message>
        }
      </Modal.Content>
      <Modal.Actions>
        <Button onClick={() => {
          setConfirm(false);
          setIsLocalStorageModalOpen(false);
        }}>
          <Icon name="cancel"/>
          {__("Cancel")}
        </Button>
        <Button
          disabled={!isValidLocalStorageConfig}
          positive
          onClick={() => !confirm ? onSaveDialog() : saveLinkLocalStorage()}>
          <Icon name="check"/>
          {!confirm ? __("Save") : __("Confirm")}
        </Button>
      </Modal.Actions>
    </Modal>
  );
};

export default LinkLocalStorageConfig;
