import React, {useCallback, useEffect, useMemo, useRef, useState} from "react";
import { useApolloClient } from "@apollo/client";
import { useNavigate, useLocation, useParams } from "react-router-dom";
import {
  Button,
  Checkbox,
  Dropdown,
  Form,
  Header,
  Icon,
  Input,
  List,
  Message,
  Modal,
  Popup,
  Ref,
  Segment,
  Tab,
  Table
} from "semantic-ui-react";
import classNames from "classnames";
import {DropTargetMonitor, useDrag, useDragLayer, useDrop} from "react-dnd";
import {getEmptyImage} from "react-dnd-html5-backend";
import {
  ChangeDeliveryPriorityDocument,
  ChangeDeliveryPriorityMutation,
  ChangeDeliveryPriorityMutationVariables,
  ChangeLocalStorageConfigMutationVariables,
  DeviceCommand,
  DeviceFunctionalAspectType,
  DeviceInput,
  Dfa_Avatar,
  DsConfiguredState,
  DsRunningState,
  HealthStatus,
  MediaArchiveOption,
  UpdateDeviceDocument,
  UpdateDeviceMutation,
  UpdateDeviceMutationVariables,
  useChangeLocalStorageConfigMutation,
  useCreateAvatarApiKeyMutation,
  useDeviceCommandMutation,
  useDevicesByAspectTypesShortQuery,
  usePurgeAvatarQueueMutation,
  useRegenerateAvatarApiKeyMutation,
  useRemoveAvatarApiKeyMutation,
  useSystemStatusQuery,
  VmxVideoStreamingMode
} from "@generated/graphql";
import {
  aspectToAspectInput,
  Device, DeviceShort,
  getLinkChannelBenchmarkText,
  getLinkChannelStatusText,
  getLinkEncryptionKeyText,
  getLinkSoftwareVersionText,
  getLinkStatusUpdateText,
  getLinkUpdateHistoryText,
  getLinkUptimeText,
  linkHealthStatusText,
  useDeviceActions,
  useLinkDeviceInfo,
  useSnapshot,
  getLinkDsRunningState,
  getLinkDsConfiguredState,
  getLinkLoadState,
  getLinkDeliveryQueue,
  getLinkDeliveryQueueUploadTime,
  getStorageUsageStateText,
  checkStorageUsageShortage,
  getStatTimeText
} from "@core/actions";
import Loading from "components/Loading";
import IconButton from "components/IconButton";
import FontAwesomeButton from "components/FontAwesomeButton";
import { queryToInput, removeTrailingSlash} from "utils";
import AdminCameraList, {DeviceDialog} from "components/Admin/AdminCameraList";
import {WidgetProps} from "components/Widgets";
import WithQueryStatus from "components/WithQueryStatus";
import EventInfoComponent from "components/EventInfo";
import EventList from "components/EventList";
import Report from "components/Admin/Report";
import {clone} from "@solid/libs/utils";
import {Log} from "@solid/libs/log";
import {__} from "@solid/libs/i18n";
import { Type } from "@solid/types";
import LinkLocalStorageConfig from "./LinkLocalStorageConfig";
import { useAccessability } from "@core/store/actions/accessability";
import { RouteParams } from "@core/types";
import { defaultActiveTab } from "../AdminLinkList";
import { searchText } from "components/SystemStatus/SystemStatus";
import Sensors from "components/Admin/Sensors";
import VaeUpdates from "./VaeUpdates";
import Gateways from "components/Admin/Gateways";
import ApiKey from "./ApiKey";

import "./style.css";

export type ConfigureLinkProps = WidgetProps & {
  link: Device;
  links: DeviceShort[];
  onBack?: () => void;
};

export enum ConfigureLinkTabRoute {
  Properties = "properties",
  Cameras = "cameras",
  DeliveryPriority = "deliveryPriority",
  Audit = "audit",
  Report = "report",
  Sensors = "sensors",
  Gateways = "gateways"
}

const refetchInterval = 60000;
const title = __("Configure Avatar");
const channelBenchmarkEnableInterval = 2 * 60 * 1000;
const routeTabIndices: Record<ConfigureLinkTabRoute, number> = {
  [ConfigureLinkTabRoute.Properties]: 0,
  [ConfigureLinkTabRoute.Cameras]: 1,
  [ConfigureLinkTabRoute.Sensors]: 2,
  [ConfigureLinkTabRoute.Gateways]: 3,
  [ConfigureLinkTabRoute.DeliveryPriority]: 4,
  [ConfigureLinkTabRoute.Audit]: 5,
  [ConfigureLinkTabRoute.Report]: 6,
};

const ConfigureLink = ({ link, links, onBack, cellProps, setCellProps }: ConfigureLinkProps) => {
  const client = useApolloClient();
  const { pathname } = useLocation();
  const { linkId = "", tabId = defaultActiveTab } = useParams<RouteParams>();
  const navigate = useNavigate();
  const { getLinkDevicesInfo, getLinkDevicesCounts } = useLinkDeviceInfo();
  const { data: statusData } = useSystemStatusQuery({ fetchPolicy: "no-cache" });
  const [deviceCommand, { data: commandData, error: commandError, loading: commandLoading }] = useDeviceCommandMutation();
  const [purgeAvatarQueue, { error: purgeError, loading: purgeLoading, data: purgeData }] = usePurgeAvatarQueueMutation({
    variables: {
      id: link.id
    },
  });
  const [changeLocalStorage, { data: changeLocalStorageData, loading: changeLocalStorageLoading, error: changeLocalStorageError }] = useChangeLocalStorageConfigMutation();
  const [createApiKey, { data: createApiKeyData, loading: createApiKeyLoading, error: createApiKeyError }] = useCreateAvatarApiKeyMutation({
    variables: {
      id: link.id
    },
  });
  const [removeApiKey, { data: removeApiKeyData, loading: removeApiKeyLoading, error: removeApiKeyError }] = useRemoveAvatarApiKeyMutation({
    variables: {
      id: link.id
    },
  });
  const [regenerateApiKey, { data: regenerateApiKeyData, loading: regenerateApiKeyLoading, error: regenerateApiKeyError }] = useRegenerateAvatarApiKeyMutation({
    variables: {
      id: link.id
    },
  });
  const [updateError, setUpdateError] = useState<Error | undefined>();
  const [updateLoading, setUpdateLoading] = useState(false);
  const { updateCachedDevice } = useDeviceActions({ skipSubscription: true });
  const { config: { limitedReportsAccess } } = useAccessability();
  const [name, setName] = useState("");
  const [nameError, setNameError] = useState("");
  const [statusUpdateInterval, setStatusUpdateInterval] = useState(0);
  const [vmxVideoStreamingMode, setVmxVideoStreamingMode] = useState<VmxVideoStreamingMode>(VmxVideoStreamingMode.Local);
  const [webRtcEnabled, setWebRtcEnabled] = useState(false);
  const [vaeImageUpdate, setVaeImageUpdate] = useState(false);
  const [uploadBandwidthCap, setUploadBandwidthCap] = useState("");
  const [valuesInitialized, setValuesInitialized] = useState(false);
  const [command, setCommand] = useState<DeviceCommand | undefined>();
  const [benchmarkConfirmOpen, setBenchmarkConfirmOpen] = useState(false);
  const [rebootConfirmOpen, setRebootConfirmOpen] = useState(false);
  const [benchmarkTime, setBenchmarkTime] = useState(0);
  const [benchmarkInProgress, setBenchmarkInProgress] = useState(false);
  const [pushEventEndPoint, setPushEventEndPoint] = useState<string>("");
  const refetchIntervalRef = useRef<NodeJS.Timeout | undefined>();
  const [/*refreshCounter*/, setRefreshCounter] = useState(0);
  const refreshTimerRef = useRef<NodeJS.Timeout | undefined>();
  const [activeTab, setActiveTab] = useState(routeTabIndices[tabId]);
  const [deviceDialog, setDeviceDialog] = useState<DeviceDialog | undefined>();
  const [getCsvFile, setGetCsvFile] = useState<boolean>(false);
  const { loading: devLoading, error: devError, data: devData, refetch: devRefetch } = useDevicesByAspectTypesShortQuery({ variables: { types: [{ type: DeviceFunctionalAspectType.Media }] } });
  const [devices, setDevices] = useState<DeviceShort[]>([]);
  const [selectedDeviceId, setSelectedDeviceId] = useState("");
  const [deliveryPriorityChanged, setDeliveryPriorityChanged] = useState(false);
  const imageRef = useRef<HTMLImageElement>(null);
  const { imageSrc, loading: snapshotLoading } = useSnapshot({ load: activeTab === 4 && !!selectedDeviceId, deviceId: selectedDeviceId, imageRef });
  const isDraggingRef = useRef(false);
  const selectedDevice = useMemo(() => devices.find(dev => dev.id === selectedDeviceId), [devices, selectedDeviceId]);
  const [wasOneActive, setWasOneActive] = useState(false);
  const [isLocalStorageModalOpen, setIsLocalStorageModalOpen] = useState<boolean>(false);
  const [vaeUpdatesModalOpen, setVaeUpdatesModalOpen] = useState<boolean>(false);
  const [closePopup, setClosePopup] = useState<boolean>(true);
  const [apiKeyPopupOpen, setApiKeyPopupOpen] = useState<boolean>(false);

  const aspect = link.aspects.find(aspect => aspect.type === DeviceFunctionalAspectType.Avatar);
  const linkAspect: Dfa_Avatar | undefined = aspect && aspect.__typename === "DFA_Avatar" ? aspect as Dfa_Avatar : undefined;

  const missingStoreCameraMessages = useMemo(() => {
    if (!devices) {
      return [];
    }
    const deviceIds = devices.map(dev => dev.id);

    return statusData
      ? statusData.systemStatus.reasons
        .filter(reason => reason.object?.type === "camera" && reason.message.startsWith(searchText) && deviceIds.includes(reason.object.id))
        .map(reason => reason.message)
      : [];
  }, [statusData, devices]);

  const cancel = () => {
    navigate(-1);
    onBack && onBack();
  };

  const pushTabHistory = (tabId: ConfigureLinkTabRoute) => {
    !pathname.includes(`/view/link/edit/${linkId}/${tabId}`) && navigate(`/view/link/edit/${linkId}/${tabId}`, { replace: true });
  };

  useEffect(() => {
    const tabRoute = getRouteTabByIndex(activeTab);
    if (tabRoute) {
      pushTabHistory(tabRoute);
      tabRoute === ConfigureLinkTabRoute.Audit && !wasOneActive && setWasOneActive(true);
    }
    else {
      console.error(`Tab index ${activeTab} doesn't have route`);
      pushTabHistory(defaultActiveTab);
    }
  }, [activeTab]);

  useEffect(() => {
    !apiKeyPopupOpen && setClosePopup(false);
  }, [apiKeyPopupOpen]);

  useEffect(() => {
    if (!purgeLoading && purgeData) {
      Log.info(__("Avatar queue purge in progress"));
    }
    if (!purgeLoading && purgeError) {
      Log.error(purgeError.message);
    }
  }, [purgeError, purgeLoading, purgeData]);

  useEffect(() => {
    if (!changeLocalStorageLoading && changeLocalStorageData) {
      Log.info(__("Saved Successfully"));
    }
  }, [changeLocalStorageData, changeLocalStorageLoading, changeLocalStorageError]);

  useEffect(() => {
    refetchIntervalRef.current = setInterval(() => !isDraggingRef.current && updateCachedDevice(link.id), refetchInterval);
    setCellProps && setCellProps({ title });

    return () => {
      setCellProps && setCellProps({ title: cellProps?.title ?? "" });
      if (refetchIntervalRef.current) {
        clearInterval(refetchIntervalRef.current);
        refetchIntervalRef.current = undefined;
      }
      if (refreshTimerRef.current) {
        clearTimeout(refreshTimerRef.current);
        refreshTimerRef.current = undefined;
      }
    };
  }, []);

  useEffect(() => {
    if (deviceDialog === undefined) {
      setCellProps && setCellProps({ title });
    }
  }, [deviceDialog]);

  useEffect(() => {
    if (!getValuesChanged()) {
      setName(link.name);
      setStatusUpdateInterval(linkAspect?.statusUpdateInterval ?? 5);
      setWebRtcEnabled(!!linkAspect?.webRtcEnabled);
      setVaeImageUpdate(!!linkAspect?.vaeImageUpdate);
      setVmxVideoStreamingMode(linkAspect?.vmxVideoStreamingMode ?? VmxVideoStreamingMode.Local);
      setUploadBandwidthCap(linkAspect?.uploadBandwidthCap ? linkAspect?.uploadBandwidthCap.toString() : "");
      setPushEventEndPoint(linkAspect?.pushEventEndPoint ?? "");
      setValuesInitialized(true);
    }

    const benchTime = linkAspect?.channelBenchmark?.startedAt ? new Date(linkAspect.channelBenchmark.startedAt).getTime() : 0;
    if (benchTime !== benchmarkTime) {
      if (benchmarkInProgress) {
        if (refreshTimerRef.current) {
          clearTimeout(refreshTimerRef.current);
        }
        // Force update component when it is time to enable Run Channel Benchmark button
        refreshTimerRef.current = setTimeout(() => setRefreshCounter(value => value + 1), channelBenchmarkEnableInterval);
      }
      setBenchmarkInProgress(false);
      setBenchmarkTime(benchTime);
    }
  }, [link]);

  useEffect(() => {
    if (!devData) {
      setDevices([]);
      return;
    }

    setDevices(devices => {
      const newDevices: DeviceShort[] = devData.devicesByAspectTypes
        .filter(dev => dev.platform?.id === link.id)
        .map(dev => {
          const result: DeviceShort = clone(dev);
          if (deliveryPriorityChanged) {
            const device = devices.find(d => d.id === dev.id);
            if (device) {
              result.deliveryPriority = device.deliveryPriority;
            }
          }
          return result;
        })
        .sort((a, b) => {
          if (!a.deliveryPriority && b.deliveryPriority) {
            return 1;
          }
          if (a.deliveryPriority && !b.deliveryPriority) {
            return -1;
          }
          const diff = (a.deliveryPriority && b.deliveryPriority) ? a.deliveryPriority - b.deliveryPriority : 0;
          return diff || a.name.localeCompare(b.name, undefined, { sensitivity: "base" });
        });
      for (let i = 0; i < newDevices.length; i++) {
        newDevices[i].deliveryPriority = i + 1;
      }
      return newDevices;
    });
  }, [link, devData]);

  useEffect(() => {
    if (commandData) {
      switch (command) {
        case DeviceCommand.AvatarUpdateSoftware:
          Log.info(__("Request to update Avatar software was successfully submitted. " +
            "It may take from few minutes up to an hour for updates to be downloaded and installed. " +
            "In the end of the update process, your Avatar box will restart automatically. " +
            "Currently, you can continue using software as usual, the update process is transparently executed in the background."));
          break;
        case DeviceCommand.AvatarUpdateKey:
          Log.info(__("Avatar channel encryption key update in progress."));
          break;
        case DeviceCommand.AvatarChannelBenchmark:
          Log.info(__("Avatar channel benchmark in progress."));
          break;
        case DeviceCommand.AvatarReboot:
          Log.info(__("Avatar reboot in progress."));
          break;
        case DeviceCommand.AvatarDsActivate:
          Log.info(__("DS activation in progress"));
          break;
        case DeviceCommand.AvatarDsDeactivate:
          Log.info(__("DS deactivation in progress"));
          break;
        case DeviceCommand.AvatarDsStart:
          Log.info(__("DS is starting"));
          break;
        case DeviceCommand.AvatarDsStop:
          Log.info(__("DS is stopping"));
          break;
      }
    }
  }, [commandData]);

  useEffect(() => {
    removeApiKeyData?.removeAvatarApiKey && Log.info(__("Deleting API key in progress"));
    removeApiKeyError && Log.error(removeApiKeyError.message);
  }, [removeApiKeyData, removeApiKeyLoading, removeApiKeyError]);

  useEffect(() => {
    createApiKeyData?.createAvatarApiKey && Log.info(__("Creating API key in progress"));
    createApiKeyError && Log.error(createApiKeyError.message);
  }, [createApiKeyData, createApiKeyLoading, createApiKeyError]);

  useEffect(() => {
    regenerateApiKeyData?.regenerateAvatarApiKey && Log.info(__("API key regeneration in progress"));
    regenerateApiKeyError && Log.error(regenerateApiKeyError.message);
  }, [regenerateApiKeyData, regenerateApiKeyLoading, regenerateApiKeyError]);

  useEffect(() => {
    if (commandError) {
      Log.error(commandError.message);
    }
  }, [commandError]);

  useEffect(() => {
    if (getCsvFile) {
      setGetCsvFile(false);
    }
  }, [getCsvFile]);

  function getValuesChanged(): boolean {
    if (!valuesInitialized) {
      return false;
    }
    return name.trim() !== link.name ||
      statusUpdateInterval !== linkAspect?.statusUpdateInterval ||
      webRtcEnabled !== linkAspect?.webRtcEnabled ||
      (linkAspect?.uploadBandwidthCap ?
        linkAspect?.uploadBandwidthCap.toString() !== uploadBandwidthCap :
        uploadBandwidthCap !== "" && uploadBandwidthCap !== "0") ||
        vaeImageUpdate !== linkAspect.vaeImageUpdate ||
        vmxVideoStreamingMode !== linkAspect.vmxVideoStreamingMode ||
        pushEventEndPoint !== linkAspect.pushEventEndPoint;
  }

  function getNameError(name: string): string {
    let error = "";
    if (!name.trim()) {
      error = __("Name should be not empty");
    }
    else if (links.some(lnk => lnk.id !== link.id && lnk.name.toLocaleUpperCase() === name.trim().toLocaleUpperCase())) {
      error = __("Avatar with the same name already exists");
    }
    return error;
  }

  function onNameInput(e: React.FormEvent<HTMLInputElement>): void {
    const name = e.currentTarget.value;
    setName(name);
    setNameError(getNameError(name));
  }

  function onUploadBandwidthCapInput(e: React.FormEvent<HTMLInputElement>): void {
    let value: string = e.currentTarget.value;
    let floatValue: number | undefined;
    if (value.length > 0 && value.charAt(value.length - 1) === ".") {
      value = value.replace(/\.+/g, ".");
      if (value === ".") {
        value = "0.";
      }
      if (value.split(".").length > 2) {
        floatValue = parseFloat(value);
      }
    }
    else {
      const valArr = value.split(".");
      if (valArr.length > 1) {
        value = valArr[0] + "." + valArr[1].slice(0, 1);
      }
      floatValue = parseFloat(value);
    }
    if (floatValue !== undefined) {
      if (floatValue < 0) {
        floatValue = 0;
      }
      setUploadBandwidthCap(!isNaN(floatValue) ? floatValue.toString() : "");
    }
    else {
      setUploadBandwidthCap(value);
    }
  }

  async function onUpdate(): Promise<void> {
    if (link.aspects.length === 0) {
      throw new Error("Avatar does not have functional aspects");
    }

    if (nameError) {
      setActiveTab(0);
      return;
    }

    const aspect = aspectToAspectInput(link.aspects[0]);

    let refetchCameras = link.name !== name.trim() || aspect.webRtcEnabled !== webRtcEnabled;

    aspect.statusUpdateInterval = statusUpdateInterval;
    aspect.vaeImageUpdate = vaeImageUpdate;
    aspect.vmxVideoStreamingMode = vmxVideoStreamingMode;
    aspect.webRtcEnabled = webRtcEnabled;
    aspect.uploadBandwidthCap = parseFloat(uploadBandwidthCap) || 0;
    aspect.pushEventEndPoint = pushEventEndPoint;

    const input: DeviceInput = {
      name: name.trim(),
      enabled: true,
      config: queryToInput(link.config),
      aspects: [aspect]
    };

    setUpdateLoading(true);
    setUpdateError(undefined);
    try {
      try {
        const { data, errors } = await client.mutate<UpdateDeviceMutation, UpdateDeviceMutationVariables>({
          mutation: UpdateDeviceDocument,
          variables: { id: link.id, device: input },
          errorPolicy: "all"
        });

        if (data) {
          if (data.updateDevice.warning) {
            Log.error(data.updateDevice.warning);
          }
        }
        else if (errors && errors.length > 0) {
          throw errors[0];
        }
        else {
          throw new Error(__("Could not update Avatar"));
        }
      }
      catch (e: any) {
        setUpdateError(e);
        return;
      }

      await updateCachedDevice(link.id);

      if (deliveryPriorityChanged && devices.length > 0) {
        try {
          const { data, errors } = await client.mutate<ChangeDeliveryPriorityMutation, ChangeDeliveryPriorityMutationVariables>({
            mutation: ChangeDeliveryPriorityDocument,
            variables: {
              devices: devices.map(({ id, deliveryPriority }, index) => ({ deviceId: id, deliveryPriority: deliveryPriority ?? index + 1 }))
            },
            errorPolicy: "all"
          });

          if (!data) {
            throw errors && errors.length > 0 ? errors[0] : new Error(__("Could not update delivery priority"));
          }

          refetchCameras = true;
        }
        catch (e: any) {
          Log.error(__("Error updating delivery priority: {{message}}", {message: e.message}));
        }
      }

      if (refetchCameras) {
        try {
          await devRefetch();
        }
        catch { /*ignore error*/ }
      }

      // onBack && onBack();
    }
    finally {
      setUpdateLoading(false);
    }
  }

  function onDeviceCommand(command: DeviceCommand): void {
    setCommand(command);
    deviceCommand({ variables: { id: link.id, command }});
  }

  function openTerminal(): void {
    let url = process.env.API_BACKEND ? removeTrailingSlash(process.env.API_BACKEND) : window.location.protocol + "//" + window.location.host;
    url += "/wetty/ssh/" + link.id;
    window.open(url, "_blank");
  }

  function moveDeliveryPriorityItem(dragIndex: number, hoverIndex: number): void {
    setDevices(devices => {
      const device = devices[dragIndex];
      const newDevices = Array.from(devices);
      newDevices.splice(dragIndex, 1);
      newDevices.splice(hoverIndex, 0, device);
      for (let i = 0; i < newDevices.length; i++) {
        newDevices[i].deliveryPriority = i + 1;
      }
      return newDevices;
    });
    setDeliveryPriorityChanged(true);
  }

  function deliveryPriorityItemClick(id: string): void {
    setSelectedDeviceId(id);
  }

  function setLocalStorage(input: ChangeLocalStorageConfigMutationVariables) {
    changeLocalStorage({ variables: input });
  }

  const onScrollProperties = useCallback((event: React.BaseSyntheticEvent) => {
    apiKeyPopupOpen && setClosePopup(true);
  }, [apiKeyPopupOpen]);

  const panes = [
    {
      menuItem: __("Properties"),
      pane: (
        <Tab.Pane
          key="properties"
          className="ConfigureLink-TabProperties"
          onScroll={onScrollProperties}
          onClick={() => apiKeyPopupOpen && setClosePopup(true)}
        >
          <Form>
            <Header as="h4">{__("General")}</Header>
            <Table compact>
              <Table.Body>
                <Table.Row>
                  <Table.Cell width={4}>{__("Name")}</Table.Cell>
                  <Table.Cell collapsing/>
                  <Table.Cell width={12}>
                    <Form.Field
                      control={Form.Input}
                      placeholder={__("Name")}
                      value={name}
                      error={nameError ? { content: nameError, pointing: "below" } : undefined}
                      onInput={(e: React.FormEvent<HTMLInputElement>) => onNameInput(e)}
                    />
                  </Table.Cell>
                </Table.Row>
                <Table.Row>
                  <Table.Cell>{__("Status Update Interval")}</Table.Cell>
                  <Table.Cell collapsing/>
                  <Table.Cell>
                    <Dropdown
                      selection
                      options={[
                        { key: 5, value: 5, text: __("5 sec") },
                        { key: 15, value: 15, text: __("15 sec") },
                        { key: 30, value: 30, text: __("30 sec") },
                        { key: 60, value: 60, text: __("1 min") },
                        { key: 120, value: 120, text: __("2 min") },
                        { key: 300, value: 300, text: __("5 min") },
                        { key: 600, value: 600, text: __("10 min") },
                        { key: 900, value: 900, text: __("15 min") },
                        { key: 1800, value: 1800, text: __("30 min") },
                      ]}
                      value={statusUpdateInterval}
                      onChange={(e, { value }) => {
                        if (typeof value === "number") {
                          setStatusUpdateInterval(value);
                        }
                      }}
                    />
                  </Table.Cell>
                </Table.Row>
                <Table.Row>
                  <Table.Cell>{__("Local Media Streaming")}</Table.Cell>
                  <Table.Cell collapsing/>
                  <Table.Cell>
                    <Checkbox
                      toggle
                      checked={webRtcEnabled}
                      onChange={(e, { checked }) => setWebRtcEnabled(!!checked)}
                    />
                  </Table.Cell>
                </Table.Row>
                <Table.Row>
                  <Table.Cell>{__("Video Wall Video Streaming Mode")}</Table.Cell>
                  <Table.Cell collapsing/>
                  <Table.Cell>
                    <Dropdown
                      selection
                      options={[
                        { key: VmxVideoStreamingMode.Auto, value: VmxVideoStreamingMode.Auto, text: __("Auto-select local/cloud") },
                        { key: VmxVideoStreamingMode.Local, value: VmxVideoStreamingMode.Local, text: __("Local only") },
                      ]}
                      value={vmxVideoStreamingMode}
                      onChange={(e, { value }) => {
                        if (typeof value === "string") {
                          setVmxVideoStreamingMode(value as VmxVideoStreamingMode);
                        }
                      }}
                    />
                  </Table.Cell>
                </Table.Row>
                <Table.Row>
                  <Table.Cell>{__("Enable Video Analytics Auto Update")}</Table.Cell>
                  <Table.Cell collapsing/>
                  <Table.Cell>
                    <Checkbox
                      toggle
                      checked={vaeImageUpdate}
                      onChange={(e, { checked }) => setVaeImageUpdate(!!checked)}
                    />
                  </Table.Cell>
                </Table.Row>
                <Table.Row>
                  <Table.Cell>{__("Status")}</Table.Cell>
                  <Table.Cell collapsing/>
                  <Table.Cell className={classNames({ "ConfigureLink-Red": link.healthStatus === HealthStatus.Off })}>
                    {linkHealthStatusText[link.healthStatus]}
                  </Table.Cell>
                </Table.Row>
                <Table.Row>
                  <Table.Cell>{__("Connection Status")}</Table.Cell>
                  <Table.Cell collapsing/>
                  <Table.Cell>{getLinkChannelStatusText(link) || __("N/A")}</Table.Cell>
                </Table.Row>
                <Table.Row>
                  <Table.Cell>{__("DS Running State")}</Table.Cell>
                  <Table.Cell collapsing>
                    {getLinkDsRunningState(link) !== "N/A" &&
                      <IconButton
                        icon="power off"
                        hint={getLinkDsRunningState(link) === DsRunningState.Running ? __("Stop") : __("Start")}
                        onClick={
                          getLinkDsRunningState(link) === DsRunningState.Running ?
                            () => onDeviceCommand(DeviceCommand.AvatarDsStop) :
                            () => onDeviceCommand(DeviceCommand.AvatarDsStart)
                        }
                    />
                    }
                  </Table.Cell>
                  <Table.Cell>{getLinkDsRunningState(link)}</Table.Cell>
                </Table.Row>
                <Table.Row>
                  <Table.Cell>{__("DS Configured State")}</Table.Cell>
                  <Table.Cell collapsing>
                    {getLinkDsConfiguredState(link) !== "N/A" &&
                      <IconButton
                        icon="power off"
                        hint={getLinkDsConfiguredState(link) === DsConfiguredState.On ? __("Deactivate") : __("Activate")}
                        onClick={
                          getLinkDsConfiguredState(link) === DsConfiguredState.On ?
                            () => onDeviceCommand(DeviceCommand.AvatarDsDeactivate) :
                            () => onDeviceCommand(DeviceCommand.AvatarDsActivate)
                        }
                    />
                    }
                  </Table.Cell>
                  <Table.Cell>{getLinkDsConfiguredState(link)}</Table.Cell>
                </Table.Row>
                <Table.Row>
                  <Table.Cell>{__("Last Status Update")}</Table.Cell>
                  <Table.Cell collapsing/>
                  <Table.Cell>{getLinkStatusUpdateText(link)}</Table.Cell>
                </Table.Row>
                <Table.Row>
                  <Table.Cell>{__("Local IP Address")}</Table.Cell>
                  <Table.Cell collapsing/>
                  <Table.Cell>{link.config.connect.host ?? ""}</Table.Cell>
                </Table.Row>
                <Table.Row>
                  <Table.Cell>{__("Devices")}</Table.Cell>
                  <Table.Cell collapsing/>
                  <Table.Cell>
                    {getLinkDevicesCounts(link).total === 0 ? __("No devices registered") : getLinkDevicesInfo(link)}
                  </Table.Cell>
                </Table.Row>
                <Table.Row>
                  <Table.Cell>{__("Avatar Load")}</Table.Cell>
                  <Table.Cell collapsing/>
                  <Table.Cell>{getLinkLoadState(link) ? `${getLinkLoadState(link)} %` : "N/A"}</Table.Cell>
                </Table.Row>
                <Table.Row>
                  <Table.Cell>{__("Avatar Delivery Queue (Scheduled + Delivery Requests)")}</Table.Cell>
                  <Table.Cell collapsing>
                    <FontAwesomeButton
                      icon="broom"
                      hint={__("Purge")}
                      disabled={benchmarkInProgress || Date.now() - benchmarkTime < channelBenchmarkEnableInterval}
                      onClick={() => purgeAvatarQueue()}
                    />
                  </Table.Cell>
                  <Table.Cell>{getLinkDeliveryQueue(link) ? `${getLinkDeliveryQueue(link)} MB` : "N/A"}</Table.Cell>
                </Table.Row>
                <Table.Row>
                  <Table.Cell>{__("Expected Time To Upload Delivery Queue")}</Table.Cell>
                  <Table.Cell collapsing/>
                  <Table.Cell>{getLinkDeliveryQueueUploadTime(link) ? `${getLinkDeliveryQueueUploadTime(link)} min` : "N/A"}</Table.Cell>
                </Table.Row>
                <Table.Row>
                  <Table.Cell>{__("Software Version")}</Table.Cell>
                  <Table.Cell collapsing>
                    <IconButton
                      icon="sync alternate"
                      hint={__("Update Software")}
                      disabled={!linkAspect?.software?.updateAvailable}
                      onClick={() => onDeviceCommand(DeviceCommand.AvatarUpdateSoftware)}
                    />
                  </Table.Cell>
                  <Table.Cell>{getLinkSoftwareVersionText(link)}</Table.Cell>
                </Table.Row>
                <Table.Row>
                  <Table.Cell>{__("VAE Updates")}</Table.Cell>
                  <Table.Cell collapsing>
                    <IconButton
                      icon="file alternate outline"
                      hint={__("View Updates")}
                      disabled={!linkAspect?.vaeUpdates}
                      onClick={() => setVaeUpdatesModalOpen(true)}
                    />
                  </Table.Cell>
                  <Table.Cell/>
                </Table.Row>
                <Table.Row>
                  <Table.Cell>{__("Object ID")}</Table.Cell>
                  <Table.Cell collapsing/>
                  <Table.Cell>{link.id}</Table.Cell>
                </Table.Row>
                {linkAspect?.isLink &&
                  <Table.Row>
                    <Table.Cell width={4}>{__("Push event endpoint")}</Table.Cell>
                    <Table.Cell collapsing/>
                    <Table.Cell width={12}>
                      <Input
                        placeholder={__("HTTP URL")}
                        value={pushEventEndPoint}
                        onInput={(e: React.FormEvent<HTMLInputElement>) => setPushEventEndPoint(e.currentTarget.value)}
                    />
                    </Table.Cell>
                  </Table.Row>}
                <Table.Row>
                  <Table.Cell>{__("API key")}</Table.Cell>
                  <Table.Cell collapsing >
                    <ApiKey
                      apiKey={linkAspect?.apiKey}
                      onCreate={() => createApiKey()}
                      onRegenerate={() => regenerateApiKey()}
                      onRemove={() => removeApiKey()}
                      closePopup={closePopup}
                      onOpen={(state) => {
                        setApiKeyPopupOpen(state);
                      }}
                    />
                  </Table.Cell>
                  <Table.Cell>
                    { linkAspect?.apiKey &&
                    <>
                      <Input
                        value={linkAspect?.apiKey}
                        disabled />
                      <IconButton
                        icon="copy"
                        hint={__("Copy to Clipboard")}
                        onClick={() => linkAspect?.apiKey && navigator.clipboard.writeText(linkAspect.apiKey)} />
                    </> }
                  </Table.Cell>
                </Table.Row>
              </Table.Body>
            </Table>

            {linkAspect?.isLink &&
            <>
              <Header as="h4">{__("Storage")}</Header>
              <Table compact>
                <Table.Body>
                  <Table.Row>
                    <Table.Cell width={4}>{__("Local Storage Volume")}</Table.Cell>
                    <Table.Cell collapsing />
                    <Table.Cell width={12}>
                      { getStorageUsageStateText(linkAspect.storageUsage?.localStorageVolume) }
                    </Table.Cell>
                  </Table.Row>
                  <Table.Row>
                    <Table.Cell>{__("Free")}</Table.Cell>
                    <Table.Cell collapsing />
                    <Table.Cell>
                      { getStorageUsageStateText(linkAspect.storageUsage?.free) }
                    </Table.Cell>
                  </Table.Row>
                  <Table.Row>
                    <Table.Cell>{__("Required Volume")}</Table.Cell>
                    <Table.Cell collapsing />
                    <Table.Cell>
                      <Popup
                        flowing
                        content={
                          [...missingStoreCameraMessages, __("Required Volume exceeds Local Storage Volume by more than 10%")]
                            .map(message =><List.Item as='li' key={message}>{message}</List.Item>)
                        }
                        disabled={
                          !checkStorageUsageShortage(linkAspect.storageUsage?.requiredVolume, linkAspect.storageUsage?.localStorageVolume)
                          && missingStoreCameraMessages.length === 0}
                        trigger={
                          <span
                            className={
                              checkStorageUsageShortage(linkAspect.storageUsage?.requiredVolume, linkAspect.storageUsage?.localStorageVolume)
                              || missingStoreCameraMessages.length > 0
                                ? "RequiredVolume-Warning"
                                : ""}>
                            { getStorageUsageStateText(linkAspect.storageUsage?.requiredVolume) }
                          </span>}/>
                    </Table.Cell>
                  </Table.Row>
                  <Table.Row>
                    <Table.Cell>{__("Local Storage")}</Table.Cell>
                    <Table.Cell collapsing>
                      <IconButton
                        icon="hdd"
                        hint={__("Open Avatar Local Storage Config")}
                        onClick={() => setIsLocalStorageModalOpen(true)}
                    />
                    </Table.Cell>
                    <Table.Cell/>
                  </Table.Row>
                </Table.Body>
              </Table>
            </>}

            <Header as="h4">{__("Channel")}</Header>
            <Table compact>
              <Table.Body>
                <Table.Row>
                  <Table.Cell width={4}>{__("Channel Turnaround Time")}</Table.Cell>
                  <Table.Cell collapsing/>
                  <Table.Cell width={12}>
                    {getStatTimeText(linkAspect?.channelTurnaroundTime)}
                  </Table.Cell>
                </Table.Row>
                <Table.Row>
                  <Table.Cell>{__("Packet Arrival Jitter Time")}</Table.Cell>
                  <Table.Cell collapsing/>
                  <Table.Cell>
                    {getStatTimeText(linkAspect?.packetArrivalJitterTime)}
                  </Table.Cell>
                </Table.Row>
                <Table.Row>
                  <Table.Cell >{__("Upload Bandwidth Cap (Mbps)")}</Table.Cell>
                  <Table.Cell collapsing/>
                  <Table.Cell>
                    <Input
                      placeholder={__("UNLIMITED")}
                      value={uploadBandwidthCap}
                      onInput={(e: React.FormEvent<HTMLInputElement>) => onUploadBandwidthCapInput(e)}
                      onBlur={() => {
                        if (!parseFloat(uploadBandwidthCap)) {
                          setUploadBandwidthCap("");
                        }
                      }}
                      onFocus={() => {
                        if (!uploadBandwidthCap) {
                          setUploadBandwidthCap("0");
                        }
                      }}
                    />
                  </Table.Cell>
                </Table.Row>
                <Table.Row>
                  <Table.Cell>{__("Channel Benchmark")}</Table.Cell>
                  <Table.Cell collapsing>
                    <FontAwesomeButton
                      icon="running"
                      hint={__("Run Channel Benchmark")}
                      disabled={benchmarkInProgress || Date.now() - benchmarkTime < channelBenchmarkEnableInterval}
                      onClick={() => setBenchmarkConfirmOpen(true)}
                    />
                  </Table.Cell>
                  <Table.Cell>{getLinkChannelBenchmarkText(link)}</Table.Cell>
                </Table.Row>
                <Table.Row>
                  <Table.Cell>{__("Channel Encryption Key")}</Table.Cell>
                  <Table.Cell collapsing>
                    <IconButton
                      icon="sync alternate"
                      hint={__("Regenerate Channel Encryption Key")}
                      onClick={() => onDeviceCommand(DeviceCommand.AvatarUpdateKey)}
                    />
                  </Table.Cell>
                  <Table.Cell>{getLinkEncryptionKeyText(link)}</Table.Cell>
                </Table.Row>
                {!linkAspect?.isLink &&
                <Table.Row>
                  <Table.Cell>{__("Migrate To LINK")}</Table.Cell>
                  <Table.Cell collapsing>
                    <IconButton
                      icon="angle double up"
                      hint={__("Migrate")}
                      onClick={() => onDeviceCommand(DeviceCommand.AvatarUpgradeSoftware)}
                    />
                  </Table.Cell>
                  <Table.Cell>{__("Migrate")}</Table.Cell>
                </Table.Row>
                }
              </Table.Body>
            </Table>

            <Header as="h4">{__("Maintenance")}</Header>
            <Table compact>
              <Table.Body>
                <Table.Row>
                  <Table.Cell width={4}>{__("Platform")}</Table.Cell>
                  <Table.Cell collapsing/>
                  <Table.Cell width={12}>{linkAspect?.platform ?? ""}</Table.Cell>
                </Table.Row>
                <Table.Row>
                  <Table.Cell>{__("Update History")}</Table.Cell>
                  <Table.Cell collapsing>
                    <IconButton
                      icon="file alternate outline"
                      hint={__("View Logs")}
                      disabled={!linkAspect?.software?.lastUpdateTime}
                    />
                  </Table.Cell>
                  <Table.Cell>{getLinkUpdateHistoryText(link)}</Table.Cell>
                </Table.Row>
                <Table.Row>
                  <Table.Cell>{__("Uptime")}</Table.Cell>
                  <Table.Cell collapsing>
                    <IconButton
                      icon="power off"
                      hint={__("Reboot")}
                      disabled={!linkAspect?.supportsReboot}
                      onClick={() => setRebootConfirmOpen(true)}
                    />
                  </Table.Cell>
                  <Table.Cell>{getLinkUptimeText(link)}</Table.Cell>
                </Table.Row>
                <Table.Row>
                  <Table.Cell>{__("Support Terminal")}</Table.Cell>
                  <Table.Cell collapsing>
                    <IconButton
                      icon="terminal"
                      hint={__("Open Terminal")}
                      onClick={() => openTerminal()}
                    />
                  </Table.Cell>
                  <Table.Cell/>
                </Table.Row>
              </Table.Body>
            </Table>
          </Form>

          {link && isLocalStorageModalOpen &&
            <LinkLocalStorageConfig
              deviceId={link.id}
              isModalOpen={isLocalStorageModalOpen}
              setIsLocalStorageModalOpen={setIsLocalStorageModalOpen}
              setLocalStorage={setLocalStorage}
            />}

          { linkAspect?.vaeUpdates &&
            <Modal
              className="VaeUpdates-Modal"
              onClose={() => setVaeUpdatesModalOpen(false)}
              onOpen={() => setVaeUpdatesModalOpen(true)}
              open={vaeUpdatesModalOpen}
            >
              <Modal.Header>{__("VAE Updates")}</Modal.Header>
              <Modal.Content className="VaeUpdatesModal-Content">
                <VaeUpdates vaeUpdates={linkAspect.vaeUpdates} />
              </Modal.Content>
              <Modal.Actions>
                <Button onClick={() => setVaeUpdatesModalOpen(false)}>
                  <Icon name="check"/>{__("Close")}
                </Button>
              </Modal.Actions>
            </Modal> }
        </Tab.Pane>
      )
    },
    {
      menuItem: __("Cameras"),
      pane: (
        <Tab.Pane key="cameras" className={classNames("ConfigureLink-TabDevices", { "ConfigureLink-TabModal": deviceDialog !== undefined })}>
          <AdminCameraList
            link={link}
            showFilter
            onDialogOpen={dialog => setDeviceDialog(dialog)}
            onDialogClose={() => setDeviceDialog(undefined)}
            setCellProps={deviceDialog !== undefined ? setCellProps : undefined}
          />
        </Tab.Pane>
      )
    },
    {
      menuItem: __("Sensors"),
      pane: (
        <Tab.Pane key="sensors" className={classNames("ConfigureLink-TabSensors", { "ConfigureLink-TabModal": deviceDialog !== undefined })}>
          <Sensors avatarId={link.id} />
        </Tab.Pane>
      )
    },
    {
      menuItem: __("Gateways"),
      pane: (
        <Tab.Pane key="gateways" className={classNames("ConfigureLink-TabGateways", { "ConfigureLink-TabModal": deviceDialog !== undefined })}>
          <Gateways />
        </Tab.Pane>
      )
    },
    {
      menuItem: __("Delivery Priority"),
      pane: (
        <Tab.Pane key="deliveryPriority" className="ConfigureLink-TabDeliveryPriority">
          <WithQueryStatus loading={devLoading} error={devError}>
            <div className="ConfigureLink-DeliveryPriority">
              <div className="ConfigureLink-DeliveryPriorityList">
                <List>
                  <List.Item>
                    <List.Content>#</List.Content>
                    <List.Content>{__("Name")}</List.Content>
                    <List.Content>{__("Location")}</List.Content>
                    <List.Content>{__("Zone")}</List.Content>
                    <List.Content>{__("State")}</List.Content>
                    <List.Content>{__("Cloud Archive")}</List.Content>
                  </List.Item>
                  {devices.map((device, index) =>
                    <DeliveryPriorityItem
                      key={device.id}
                      device={device}
                      index={index}
                      selected={device.id === selectedDeviceId}
                      moveItem={moveDeliveryPriorityItem}
                      onClick={() => deliveryPriorityItemClick(device.id)}
                      onDragging={isDragging => { isDraggingRef.current = isDragging; }}
                    />)}
                </List>
                <CustomDragLayer/>
              </div>
              <div className={classNames("ConfigureLink-DeliveryPriorityRight", { "hidden": !selectedDevice })}>
                <div className="ConfigureLink-DeliveryPrioritySnapshot">
                  <img
                    ref={imageRef}
                    src={imageSrc}
                    alt=""
                    className={classNames({ "hidden": !imageSrc })}
                  />
                  {snapshotLoading && <Loading/>}
                </div>
                {!!selectedDevice &&
                <div className="ConfigureLink-DeliveryPriorityInfo">
                  <div>{__("Name")}:</div>
                  <div>{selectedDevice.name}</div>
                  <div>{__("Location")}:</div>
                  <div>{selectedDevice.location}</div>
                  <div>{__("Zone")}:</div>
                  <div>{selectedDevice.zone?.name}</div>
                  <div>{__("State")}:</div>
                  <div>{selectedDevice.enabled ? __("On") : __("Off")}</div>
                </div>}
              </div>
            </div>
          </WithQueryStatus>
        </Tab.Pane>
      )
    },
    {
      menuItem: __("Audit"),
      pane: (
        <Tab.Pane key="audit">
          {(activeTab === 5 || wasOneActive) &&
            <div className="audit">
              <div className="event-control">
                <div className="event-list-control">
                  <Header size="tiny">{__("Events: {{name}}", {name: link.name})}</Header>
                  <Button
                    className="export-audit"
                    size="mini"
                    onClick={() => setGetCsvFile(true)}>
                    <Icon name="download"/>
                    {__("Export CSV")}
                  </Button>
                </div>
                <div className="event-info-control">
                  <Header size="tiny">{__("Event details")}</Header>
                </div>
              </div>
              <div className="audit-content">
                <div className="event-list">
                  <div className="events">
                    <EventList witnessesList={[link.id]} exportCsvFile={getCsvFile}/>
                  </div>
                </div>
                <div className="event-info">
                  <div className="info">
                    <EventInfoComponent isCell/>
                  </div>
                </div>
              </div>
            </div>
          }
        </Tab.Pane>
      )
    },
    {
      menuItem: !limitedReportsAccess ? __("Report") : null,
      pane: (
        <Tab.Pane key="report" className="Link-Reports">
          <Report deviceId={link.id} reportType={Type.avatar}/>
        </Tab.Pane>
      )
    }
  ];
  /* eslint-enable react/jsx-indent */

  return (
    <Segment className={classNames("ConfigureLink", { "ConfigureLink-ModalOpen": deviceDialog !== undefined })}>
      {!!updateError && <Message error content={updateError.message}/>}

      <div className="ConfigureLink-TopButtons">
        <Button onClick={() => cancel()}>
          <Icon name="arrow left"/>{__("Back")}
        </Button>
        <Button positive disabled={(!getValuesChanged() && !deliveryPriorityChanged) || !!nameError} onClick={() => onUpdate()}>
          <Icon name="check"/>{__("Save")}
        </Button>
      </div>

      <Tab
        panes={panes}
        className="ConfigureLink-Tab"
        activeIndex={activeTab}
        renderActiveOnly={false}
        onTabChange={(e, { activeIndex }) => typeof activeIndex === "number" && setActiveTab(activeIndex)}
      />

      {updateLoading && <Loading text={__("Updating...")}/>}
      {commandLoading && <Loading text={__("Sending command...")}/>}

      <Modal open={benchmarkConfirmOpen} onClose={() => setBenchmarkConfirmOpen(false)}>
        <Header>{__("Channel Benchmark")}</Header>
        <Modal.Content>
          {__("Warning! Communication between this avatar and the cloud will be suspended for 1 minute.")}<br/>
          {__("All types of the cloud content will not be available for users during this time period.")}
        </Modal.Content>
        <Modal.Actions>
          <Button negative onClick={() => {
            setBenchmarkConfirmOpen(false);
            setBenchmarkInProgress(true);
            onDeviceCommand(DeviceCommand.AvatarChannelBenchmark);
          }}>
            <Icon name="check"/>{__("Run Channel Benchmark")}
          </Button>
          <Button positive onClick={() => setBenchmarkConfirmOpen(false)}>
            <Icon name="cancel"/>{__("Cancel")}
          </Button>
        </Modal.Actions>
      </Modal>

      <Modal open={rebootConfirmOpen} size="tiny" onClose={() => setRebootConfirmOpen(false)}>
        <Header>{__("Reboot Avatar")}</Header>
        <Modal.Content>
          {__("The Avatar will be rebooted.")}<br/>
          {__("Do you want to continue?")}
        </Modal.Content>
        <Modal.Actions>
          <Button negative onClick={() => {
            setRebootConfirmOpen(false);
            onDeviceCommand(DeviceCommand.AvatarReboot);
          }}>
            <Icon name="power off"/>{__("Reboot")}
          </Button>
          <Button positive onClick={() => setRebootConfirmOpen(false)}>
            <Icon name="cancel"/>{__("Cancel")}
          </Button>
        </Modal.Actions>
      </Modal>
    </Segment>
  );
};

type DeliveryPriorityItemProps = {
  device: DeviceShort;
  index: number;
  selected: boolean;
  moveItem: (dragIndex: number, hoverIndex: number) => void;
  onClick: () => void;
  onDragging: (isDragging: boolean) => void;
};

const dragItemType = "DeliveryPriorityItem";

const DeliveryPriorityItem = ({
  device: { id, name, location, zone, enabled, deliveryPriority, aspects },
  index,
  selected,
  moveItem,
  onClick,
  onDragging
}: DeliveryPriorityItemProps) => {
  type DragItem = {
    type: string;
    index: number;
    id: string;
    name: string;
    location: string;
    zone: string;
    state: string;
    cloudArchive: boolean;
  };

  const cloudArchive = aspects.some(aspect => aspect.__typename === "DFA_Media" && aspect.cloudArchive !== MediaArchiveOption.None);

  const itemRef = useRef<HTMLDivElement>(null);

  const [, drop] = useDrop({
    accept: dragItemType,
    hover(item: DragItem, monitor: DropTargetMonitor) {
      if (!itemRef.current) {
        return;
      }
      const dragIndex = item.index;
      const hoverIndex = index;

      // Don't replace items with themselves
      if (dragIndex === hoverIndex) {
        return;
      }

      // Time to actually perform the action
      moveItem(dragIndex, hoverIndex);

      // Note: we're mutating the monitor item here!
      // Generally it's better to avoid mutations,
      // but it's good here for the sake of performance
      // to avoid expensive index searches.
      item.index = hoverIndex;
    },
  });

  const [{ isDragging }, drag, preview] = useDrag({
    item: { type: dragItemType, index, id, name, location: location ?? "", zone: zone?.name ?? "", state: enabled ? __("On") : __("Off"),  cloudArchive },
    collect: (monitor: any) => ({
      isDragging: monitor.isDragging(),
    }),
  });

  useEffect(() => {
    preview(getEmptyImage(), { captureDraggingState: true });
  }, []);

  useEffect(() => {
    onDragging(isDragging);
  }, [isDragging]);

  drag(drop(itemRef));

  return (
    <Ref innerRef={itemRef}>
      <List.Item
        className={classNames("DeliveryPriorityItem", { "DeliveryPriorityItem_dragging": isDragging, "DeliveryPriorityItem_selected": selected })}
        onMouseDown={onClick}
      >
        <List.Content>{deliveryPriority}</List.Content>
        <List.Content>{name}</List.Content>
        <List.Content>{location}</List.Content>
        <List.Content>{zone?.name}</List.Content>
        <List.Content>{enabled ? __("On") : __("Off")}</List.Content>
        <List.Content>{cloudArchive ? <Icon name="checkmark" color="green"/> : ""}</List.Content>
      </List.Item>
    </Ref>
  );
};

const CustomDragLayer = () => {
  const {
    itemType,
    isDragging,
    item,
    currentOffset,
  } = useDragLayer((monitor) => ({
    item: monitor.getItem(),
    itemType: monitor.getItemType(),
    currentOffset: monitor.getSourceClientOffset(),
    isDragging: monitor.isDragging(),
  }));

  const layerRef = useRef<HTMLDivElement>(null);

  function renderItem() {
    switch (itemType) {
      case dragItemType:
        const { index, name, location, zone, state, cloudArchive } = item;
        return (
          <div className="DeliveryPriorityItem-DragPreview">
            <div>{index + 1}</div>
            <div>{name}</div>
            <div>{location}</div>
            <div>{zone}</div>
            <div>{state}</div>
            <div>{cloudArchive ? <Icon name="checkmark" color="green"/> : ""}</div>
          </div>
        );
      default:
        return null;
    }
  }

  if (!isDragging || !currentOffset) {
    return null;
  }

  let y = -1;
  if (layerRef.current) {
    y = currentOffset.y - layerRef.current.getBoundingClientRect().top;
  }

  return (
    <div ref={layerRef} className="DeliveryPriority-CustomDragLayer">
      {y >= 0 &&
      <div style={{ transform: `translate(0px, ${y}px)` }}>
        {renderItem()}
      </div>}
    </div>
  );
};

export default ConfigureLink;

const getRouteTabByIndex = (tabIndex: number): ConfigureLinkTabRoute | undefined => {
  const tabValues = Object.entries(routeTabIndices);
  const tabRoute = tabValues.find(value => tabIndex === value[1])?.[0];
  return tabRoute as ConfigureLinkTabRoute;
};
