import React, { useCallback, useEffect, useMemo, useRef, useState} from "react";
import { Button, DropdownProps, Form, Header, Icon, Segment, Table } from "semantic-ui-react";
import { ApolloError } from "@apollo/client";

import {
  AuthType,
  DeviceBaseConfigInput,
  DeviceBaseConfigType,
  DeviceFunctionalAspectType,
  DeviceInput,
  DeviceProbeInput,
  MediaStreamType,
  NetworkConnectDescriptorInput,
  ProbeDeviceQuery,
  StoragePool,
  ZonesQuery,
  useCreateDeviceMutation,
  useDevicesByAspectTypesShortQuery,
  useProbeDeviceLazyQuery,
  useStoragePoolsQuery,
  useZonesQuery
} from "@generated/graphql";
import { API } from "@solid/libs/api";
import { formatDateTime } from "@core/utils";
import { UUID } from "@solid/types";
import { __ } from "@solid/libs/i18n";
import {DeviceShort, aspectToAspectInput, useDeviceActions} from "@core/actions";
import { useStore } from "@core/store";
import { Log } from "@solid/libs";

import Help from "components/Help";
import ImportDataRender  from "./ImportDataRender";

import "./style.css";
import HelpMD from "./help.md";

const api = new API();
const currentVersion = "1";
export const indexShift = 2;
export const cameraEntry = [
  "deviceName",
  "deviceLocation",
  "deviceZone",
  "cloudDirect",
  "storagePool",
  "avatarId",
  "deviceAddress",
  "username",
  "password",
  "deviceType",
  "deviceUrl",
  "onvifProfile",
  "httpPort",
  "rtcpPort"
];

export type IsCorrectType = {
  correct: boolean,
  isCorrectZone: boolean,
  isCloudDirect: boolean,
  isName: boolean,
  isAddress: boolean,
  isCorrectType: boolean,
  isCorrectStorage: boolean
};

export type ImportCameraEntry = {
  deviceName?: string,
  deviceLocation?: string,
  deviceZone?: string,
  cloudDirect?: boolean | string,
  storagePool?: string,
  avatarId?: string,
  deviceAddress?: string,
  username?: string,
  password?: string,
  deviceType?: string,
  deviceUrl?: string,
  onvifProfile?: string,
  httpPort?: number | string,
  rtcpPort?: number | string,
  valid?: boolean,
  correct?: boolean,
  id?: UUID | null
};

export enum DeviceConfigType {
  Onvif = "ONVIF-camera",
  Url = "URL-camera"
}

type ProbeResult = {
  data: ProbeDeviceQuery["probeDevice"] | undefined;
  error: ApolloError | undefined;
};

type ImportJob = {
  [id: string]: ImportCameraEntry[]
};

type ImportJobs = {
  version: string;
  list: ImportJob
};

const ImportCameraList = () => {
  const { store: { session: { info } } } = useStore();
  const { data: zonesData, error: zonesError, loading: zonesLoading } = useZonesQuery();
  const { data: storagesData, error: storagesError, loading: storagesLoading } = useStoragePoolsQuery();
  const { data: deviceData, loading: deviceLoading, error: deviceError } = useDevicesByAspectTypesShortQuery({ variables: { types: [{ type: DeviceFunctionalAspectType.Avatar }] } });
  const [probeDevice] = useProbeDeviceLazyQuery({ fetchPolicy: "network-only" });
  const [createDevice] = useCreateDeviceMutation();
  const { updateCachedDevice } = useDeviceActions({ skipSubscription: true });
  const [probeResultMap, setProbeResultMap] = useState<Map<number, ProbeResult>>(new Map());
  const [realmId, setRealmId] = useState<UUID>();
  const [currentList, setCurrentList] = useState<ImportCameraEntry[]>([]);
  const [fileResult, setFileResult] = useState<string>();
  const [uploadsLength, setUploadsLength] = useState<number[]>([]);
  const [deviceList, setDeviceList] = useState<DeviceShort[]>([]);
  const [zonesList, setZonesList] = useState<string[]>([]);
  const [storageList, setStorageList] = useState<StoragePool[]>([]);
  const [currentJob, setCurrentJob] = useState<string>();
  const [jobIdList, setJobIdList] = useState<string[]>([]);
  const [jobList, setJobList] = useState<ImportJob>();
  const [probMessageMap, setProbMessageMap] = useState<Map<number, string>>(new Map());
  const [isProbing, setIsProbing] = useState<boolean>(false);
  const [isCreating, setIsCreating] = useState<boolean>(false);
  const [createMessageMap, setCreateMessageMap] = useState<Map<number, string>>(new Map());
  const [processingJob, setProcessingJob] = useState<boolean>(false);
  const [selectedJob, setSelectedJob] = useState<string>();
  const [disposeJob, setDisposeJob] = useState<string>();
  const fileInputRef = useRef<HTMLInputElement>(null);

  const jobOptions = useMemo(() => {
    const options = jobIdList.map((item: string) => {
      const option = {};
      option["key"] = item;
      option["text"] = `Import job ${formatDateTime(new Date(Number(item)))}`;
      option["value"] = item;
      return option;
    });
    return options;
  }, [jobIdList]);

  const download = (downloadUrl: string, fileName: string) => {
    const a = document.createElement("a");
    a.href = downloadUrl;
    a.setAttribute("download", fileName);
    a.click();
  };

  const onFileChange = (e: React.ChangeEvent<HTMLInputElement>): void => {
    if (e.target.files) {
      setProcessingJob(true);
      const currentFile = e.target.files[0];
      const reader = new FileReader();
      reader.onloadstart = () => setProcessingJob(true);
      reader.onerror = () => setProcessingJob(false);
      reader.onabort = () => setProcessingJob(false);
      reader.onload = () => {
        if (typeof reader.result === "string") {
          setFileResult(reader.result);
        }
      };
      reader.readAsText(currentFile);
    }
  };

  const getImportJobsList = async () => {
    let list: { [id: string]: ImportCameraEntry[] } = {};
    try {
      const response = await api.getAttribute({ obj: realmId, attribute: "IMPORT_JOBS" });
      const importJobsValue: ImportJobs = JSON.parse(response.value);
      if (importJobsValue.version === currentVersion) {
        list = importJobsValue.list;
        return list;
      }
      return list;
    } catch (e) {
      console.error(e);
      return list;
    }
  };

  const setImportJobsList = (newImportJobs: any) => {
    const IMPORT_JOBS = {
      version: currentVersion,
      list: newImportJobs
    };
    api.setAttributes({obj: realmId, attributes: {IMPORT_JOBS}});
  };

  const handleChangeItem = useCallback((index: number, listItem: ImportCameraEntry) => {
    const newCurrentList = currentList;
    newCurrentList[index] = listItem;

    if (jobList && currentJob && currentJob in jobList) {
      const newJobList = jobList;
      newJobList[currentJob] = newCurrentList;
      setJobList(newJobList);
    }

    getImportJobsList()
      .then((list) => {
        if (currentJob && currentJob in list) {
          const newList = list;
          newList[currentJob] = newCurrentList;
          setImportJobsList(newList);
        }
      });
  }, [currentJob, currentList, jobList]);

  const getStorageId = (name?: string) => {
    const currentStoragePool = storageList
      .filter(item => item.name === name)
      .map(name => name.id)
      .toString();
    if (currentStoragePool) {
      return currentStoragePool;
    }
    return "";
  };

  const isCorrect = (entry: ImportCameraEntry): IsCorrectType => {
    const cloudDirect = entry.cloudDirect === "1" || !!entry.cloudDirect;
    const name = entry.deviceName ?? "";
    const address = entry.deviceAddress ?? "";
    const avatarId = entry.avatarId ?? "";
    const deviceZone = entry.deviceZone ?? "";
    const deviceUrl = entry.deviceUrl ?? "";
    const onvifProfile = entry.onvifProfile ?? "";
    const storagePool = entry.storagePool ?? "";

    const isName = name.length > 0;
    const isAddress = address.length > 0;
    const isCorrectZone = (deviceZone.length > 0 && zonesList.some(zone => zone === deviceZone)) || deviceZone.length === 0;
    const isCorrectStorage = storageList.some(storage => storage.id === getStorageId(storagePool) || storage.id === storagePool);
    const isCorrectDeviceId = deviceList.some(dev => dev.id === avatarId);
    const isCloudDirect = (cloudDirect && avatarId.length === 0) || (cloudDirect === false && isCorrectDeviceId);
    const isCorrectType = (entry.deviceType === DeviceConfigType.Onvif && onvifProfile.length > 0 && deviceUrl.length === 0) ||
                          (entry.deviceType === DeviceConfigType.Url && deviceUrl.length > 0 && onvifProfile.length === 0);
    const correct = entry.id ? true : isName && isAddress && isCloudDirect && isCorrectType && isCorrectStorage && isCorrectZone;

    return {
      correct,
      isCorrectZone,
      isCloudDirect,
      isName,
      isAddress,
      isCorrectType,
      isCorrectStorage
    };
  };

  useEffect(() => {
    if (info) {
      setRealmId(info.realm.id);
    }
  }, [info]);

  useEffect(() => {
    if (realmId) {
      getImportJobsList()
        .then((list) => {
          if (Object.keys(list).length > 0) {
            setJobIdList(Object.keys(list));
            setJobList(list);
          }
        });
    }
  }, [realmId]);

  useEffect(() => {
    if (!deviceLoading && !deviceError && deviceData) {
      const devices = deviceData.devicesByAspectTypes;
      setDeviceList(devices);
    }
  }, [deviceData, deviceLoading, deviceError]);

  useEffect(() => {
    if (!zonesLoading && !zonesError && zonesData) {
      const zones = zonesData.zones.map(zone => zone.name);
      setZonesList(zones);
    }
  }, [zonesData, zonesLoading, zonesError]);

  useEffect(() => {
    if (!storagesLoading && !storagesError && storagesData) {
      const storages = storagesData.storagePools.map(storage => storage);
      setStorageList(storages);
    }
  }, [storagesData, storagesLoading, storagesError]);

  useEffect(() => {
    if (!fileResult) return;

    const processResult = processFile(fileResult);
    const [newImportJobId, newList] = Object.entries(processResult.job)[0];
    const currentJobList = jobList ?? {};
    currentJobList[newImportJobId] = newList;
    setJobList(currentJobList);
    setCurrentList(newList);
    getImportJobsList()
      .then((list) => {
        if (Object.keys(list).length === 0) {
          setImportJobsList({[newImportJobId]: newList});
        } else {
          const allImportJobs = list;
          allImportJobs[newImportJobId] = newList;
          setImportJobsList(allImportJobs);
        }
      })
      .then(() => {
        setProbMessageMap(new Map());
        setCreateMessageMap(new Map());
        setProbeResultMap(new Map());
        setJobIdList(jobIdList => [...jobIdList, newImportJobId]);
        setCurrentJob(newImportJobId);
        setUploadsLength(processResult.uploadsResult);
        setProcessingJob(false);
      })
      .catch((e) => {
        setProcessingJob(false);
      })
      .finally(() => {
        setProcessingJob(false);
      });
  }, [fileResult]);

  useEffect(() => {
    if (!selectedJob) return;
    if (selectedJob === currentJob) {
      setProcessingJob(false);
      return;
    }
    if (jobList && selectedJob in jobList) {
      setProbMessageMap(new Map());
      setCreateMessageMap(new Map());
      setProbeResultMap(new Map());
      const currentList = jobList[selectedJob];

      const invalidIndexes = [];
      const validIndexes = [];
      currentList.forEach((listItem: ImportCameraEntry, index: number) => {
        if (listItem.correct) {
          validIndexes.push(index);
        } else {
          invalidIndexes.push(index);
        }
      });

      setUploadsLength([validIndexes.length, invalidIndexes.length]);
      setCurrentJob(selectedJob);
      setCurrentList(currentList);
      setProcessingJob(false);
    }
    else {
      setProcessingJob(false);
    }
  }, [selectedJob]);

  useEffect(() => {
    if (!disposeJob) {
      setProcessingJob(false);
      return;
    }
    if (jobList && disposeJob in jobList) {
      const newJobList = jobList;
      delete newJobList[disposeJob];
      setJobList(newJobList);
      getImportJobsList()
        .then((list) => {
          if (disposeJob in list) {
            const allJobsList = list;
            delete allJobsList[disposeJob];

            setImportJobsList(allJobsList);
            setJobIdList(Object.keys(allJobsList));
            setUploadsLength([]);
            setProbeResultMap(new Map());
            setProbMessageMap(new Map());
            setCreateMessageMap(new Map());
            setCurrentList([]);
            setCurrentJob(undefined);
            setProcessingJob(false);
          }
        })
        .catch((e) => {
          console.error(e);
          setProcessingJob(false);
        })
        .finally(() => {
          setProcessingJob(false);
        });
    }
    else {
      setProcessingJob(false);
    }
  }, [disposeJob]);

  useEffect(() => {
    if (isProbing || probeResultMap.size === 0) return;
    const newList = Array.from(currentList);
    let isProbingSuccessful = true;

    for (let i = 0; i < probeResultMap.size; i++) {
      const probeResult = probeResultMap.get(i);
      const entry = newList[i];
      if (!probeResult || !entry) continue;
      const { data: probeData, error } = probeResult;
      if (!probeData || error) {
        isProbingSuccessful = false;
        continue;
      }
      const isEntryOnvifProfileExist = probeData.aspects.some(aspect =>
        aspect.__typename === "DFA_Media" &&
        aspect.streamType === MediaStreamType.Video &&
        aspect.onvifProfile === entry.onvifProfile
      );
      if (!isEntryOnvifProfileExist) {
        entry.onvifProfile = "";
        entry.correct = isCorrect(entry).correct;
        isProbingSuccessful = false;
      }
      newList[i] = entry;
    }

    if (!isProbingSuccessful) {
      setCurrentList(newList);
      updateImportJobList(newList);
    }
  }, [isProbing, probeResultMap]);

  function processFile(file: string): { job: ImportJob, uploadsResult: [number, number] } {
    const newImportJobId = new Date().getTime().toString();
    const lines = file.split("\n");
    const invalidIndexes = [];
    const validIndexes = [];
    const newList: ImportCameraEntry[] = [];

    for (let i = 1; i < lines.length; i++) {
      const listItem = {};
      const currentLine = lines[i].split(",");
      if (currentLine.length === cameraEntry.length) {
        validIndexes.push(i);
        listItem["valid"] = true;
        listItem["correct"] = true;
        listItem["id"] = null;
        for (let j = 0; j < cameraEntry.length; j++) {
          listItem[cameraEntry[j]] = currentLine[j];
        }
        newList.push(listItem);
      } else {
        invalidIndexes.push(i);
        listItem["valid"] = false;
        listItem["correct"] = false;
        listItem["id"] = null;
        for (let u = 0; u < currentLine.length; u++) {
          listItem[u] = currentLine[u];
        }
        newList.push(listItem);
      }
    }

    for (let i = 0; i < newList.length; i++) {
      if (newList[i].valid) {
        newList[i].correct = isCorrect(newList[i]).correct;
      }
    }

    return {
      job: { [newImportJobId]: newList },
      uploadsResult: [validIndexes.length, invalidIndexes.length]
    };
  }

  async function createCameras() {
    setProbMessageMap(new Map());
    setCreateMessageMap(new Map());
    setIsCreating(true);
    const newList = Array.from(currentList);
    try {
      for (let i = 0; i < currentList.length; i++) {
        const entry = currentList[i];
        if (!isValidCreateEntry(entry, i)) continue;
        const probeData = probeResultMap.get(i)?.data;
        const input = getDeviceInput(entry, probeData, zonesData?.zones);
        if (!input) continue;
        const createResult = await createDevice({ variables: { device: input } });
        const { data, errors } = createResult;
        const rowIndex = i + indexShift;
        if (data?.createDevice) {
          data.createDevice.warning && Log.error(data.createDevice.warning);
          const id = data.createDevice.id;
          newList[i] = { ...newList[i], id };
          await updateImportJobList(newList);
          setCreateMessageMap(prevMap => {
            const newMap = new Map(prevMap);
            newMap.set(i, __("line {{number}}: camera '{{name}}' added with id: {{id}}", { number: rowIndex, name: entry.deviceName, id }));
            return newMap;
          });
          await updateCachedDevice(data.createDevice.id);
        }
        if (errors) {
          const message: string = errors.map(error => error.message).join(". ");
          setCreateMessageMap(prevMap => {
            const newMap = new Map(prevMap);
            newMap.set(i, __("line {{number}}: {{message}}", { number: rowIndex, message }));
            return newMap;
          });
        }
      }
      setCurrentList(newList);
      setIsCreating(false);
    }
    catch (e) {
      setCurrentList(newList);
      setIsCreating(false);
      console.error("create error:", e);
    }
    finally {
      setIsCreating(false);
    }
  }

  async function probeEntries() {
    setProbMessageMap(new Map());
    setCreateMessageMap(new Map());
    setIsProbing(true);
    try {
      for (let i = 0; i < currentList.length; i++) {
        const entry = currentList[i];
        if (!isValidProbeEntry(entry, i)) continue;
        const input = getProbeInput(entry);
        const { data, error } = await probeDevice({ variables: { input } });
        setProbeResultMap(prevMap => {
          const newMap = new Map(prevMap);
          newMap.set(i, { data: data?.probeDevice, error });
          return newMap;
        });
        const rowIndex = i + indexShift;
        if (error) {
          setProbMessageMap(prevMap => {
            const newMap = new Map(prevMap);
            newMap.set(i, __("line {{number}}: {{message}}", { number: rowIndex, message: error.message }));
            return newMap;
          });
        }
        if (data) {
          setProbMessageMap(prevMap => {
            const newMap = new Map(prevMap);
            newMap.set(i, __("line {{number}}: Camera '{{name}}' probing is successful", { number: rowIndex, name: entry.deviceName}));
            return newMap;
          });
        }
      }
      setIsProbing(false);
    }
    catch (e) {
      setIsProbing(false);
      console.error(e);
    }
    finally {
      setIsProbing(false);
    }
  }

  function isValidProbeEntry(entry: ImportCameraEntry, index: number): boolean {
    const rowIndex = index + indexShift;
    if (entry.id) {
      setProbMessageMap(prevMap => {
        const newMap = new Map(prevMap);
        newMap.set(index, __("line {{number}}: Camera '{{name}}' previously added", { number: rowIndex, name: entry.deviceName }));
        return newMap;
      });
      return false;
    }
    if (!entry.correct || !entry.valid) {
      setProbMessageMap(prevMap => {
        const newMap = new Map(prevMap);
        newMap.set(index, __("line {{number}}: Invalid entry", { number: rowIndex }));
        return newMap;
      });

      return false;
    }

    return true;
  }

  function isValidCreateEntry(entry: ImportCameraEntry, index: number): boolean {
    const rowIndex = index + indexShift;
    if (entry.id) {
      setCreateMessageMap(prevMap => {
        const newMap = new Map(prevMap);
        newMap.set(index, __("line {{number}}: Camera '{{name}}' previously added", { number: rowIndex, name: entry.deviceName }));
        return newMap;
      });
      return false;
    }
    if (!entry.correct) {
      setCreateMessageMap(prevMap => {
        const newMap = new Map(prevMap);
        newMap.set(index, __("line {{number}}: Invalid entry", { number: rowIndex }));
        return newMap;
      });
      return false;
    }
    const probeData = probeResultMap.get(index)?.data;
    if (!probeData) {
      setCreateMessageMap(prevMap => {
        const newMap = new Map(prevMap);
        newMap.set(index, __("line {{number}}: Skipped. Try to probe again", { number: rowIndex }));
        return newMap;
      });
      return false;
    }

    return true;
  }

  async function updateImportJobList(entries: ImportCameraEntry[]) {
    await getImportJobsList()
      .then((list) => {
        if (currentJob && currentJob in list) {
          const newList = list;
          newList[currentJob] = entries;
          setImportJobsList(newList);
        }
      });
  }

  return (
    <Segment className="Import-Devices" loading={!currentList || deviceLoading || zonesLoading || storagesLoading || processingJob}>
      <div className="Import-Panel">
        <Header as="h3" className="Import-Header">
          {__("Import Jobs")}
        </Header>
        <Form.Select
          disabled={jobOptions.length === 0 || isProbing || isCreating}
          options={jobOptions}
          placeholder={__("Select Import job")}
          value={currentJob}
          className="Import-Select-Job"
          onChange={(event: React.SyntheticEvent<HTMLElement, Event>, { value }: DropdownProps) => {
            event.preventDefault();
            if (typeof value === "string") {
              setProcessingJob(true);
              setSelectedJob(value);
            }
          }}
        />
        <Form onSubmit={(e) => e.preventDefault()} className="Import-Control">
          <Form.Field>
            <Button
              disabled={isProbing || isCreating}
              onClick={() => {
                setFileResult(undefined);
                fileInputRef.current?.click(); }}>
              <Icon name="plus"/>
              {__("New Import job")}
            </Button>
            <label>
              <input
                ref={fileInputRef}
                type="file"
                id="import_control_input_file_hidden"
                hidden
                accept=".csv"
                onChange={onFileChange}
                onClick={(e: any) => {
                  e.target.value = "";
                }}
              />
            </label>
          </Form.Field>
          <Form.Field>
            <Button
              disabled={!currentJob || isProbing || isCreating}
              onClick={(e) => {
                e.preventDefault();
                if (currentJob) {
                  setProcessingJob(true);
                  setDisposeJob(currentJob);
                }
              }}>
              <Icon name="minus"/>
              {__("Dispose Import job")}
            </Button>
          </Form.Field>
          <Form.Field>
            <Button onClick={() => download("cameras.csv", "cameras.csv")}>
              <Icon name="download" />
              {__("Download example file")}
            </Button>
          </Form.Field>
        </Form>
        <Help markdown={HelpMD}/>
      </div>

      <>
        {currentList.length > 0 &&
          <div className="Import-Add-Button">
            <Button
              className="play-test-btn"
              positive
              disabled={isProbing || isCreating}
              onClick={() => probeEntries()}
            >
              <Icon name="play"/>{__("Probe")}
            </Button>

            <Button
              className="final-create-btn"
              positive
              disabled={probeResultMap.size === 0 || isProbing || isCreating}
              onClick={() => createCameras()}
            >
              <Icon name="plus"/>{__("Add")}
            </Button>
          </div>}

        {probMessageMap.size > 0 ?
          <div className="Import-Messages">
            {[...probMessageMap.values()].map((message, index) =>
              <p key={index}>
                {message}
              </p>)
            }
          </div> :
          createMessageMap.size > 0 ?
            <div className="Import-Messages">
              {[...createMessageMap.values()].map((message, index) =>
                <p key={index}>
                  {message}
                </p>)}
            </div> : null
          }

        {uploadsLength.length > 0 &&
        <div className="Import-Message">
          { __("{{positiveCount}} entries accepted, {{negativeCount}} rejected", {positiveCount: uploadsLength[0], negativeCount: uploadsLength[1]})}
        </div>}

        <Segment className="Import-Result" loading={isProbing || isCreating}>
          {currentList.length > 0 &&
            <Table celled compact className="Import-Table">
              <Table.Header>
                <Table.Row>
                  <Table.HeaderCell>{__("Line")}</Table.HeaderCell>
                  <Table.HeaderCell>{__("Action")}</Table.HeaderCell>
                  <Table.HeaderCell>{__("Device name")}</Table.HeaderCell>
                  <Table.HeaderCell>{__("Device location")}</Table.HeaderCell>
                  <Table.HeaderCell>{__("Device zone")}</Table.HeaderCell>
                  <Table.HeaderCell>{__("Cloud direct")}</Table.HeaderCell>
                  <Table.HeaderCell>{__("Storage Pool")}</Table.HeaderCell>
                  <Table.HeaderCell>{__("Avatar ID")}</Table.HeaderCell>
                  <Table.HeaderCell>{__("Device address")}</Table.HeaderCell>
                  <Table.HeaderCell>{__("User name")}</Table.HeaderCell>
                  <Table.HeaderCell>{__("User password")}</Table.HeaderCell>
                  <Table.HeaderCell>{__("Device type")}</Table.HeaderCell>
                  <Table.HeaderCell>{__("Device URL")}</Table.HeaderCell>
                  <Table.HeaderCell>{__("ONVIF profile")}</Table.HeaderCell>
                  <Table.HeaderCell>{__("Http port")}</Table.HeaderCell>
                  <Table.HeaderCell>{__("Rtsp port")}</Table.HeaderCell>
                </Table.Row>
              </Table.Header>
              <Table.Body>
                {currentList.map((item, index) =>
                  <ImportDataRender
                    key={`${currentJob}_${index}`}
                    startIndex={index}
                    listItem={item}
                    deviceList={deviceList}
                    zonesList={zonesList}
                    storageList={storageList}
                    handleChangeItem={handleChangeItem}
                    isCorrect={isCorrect}
                    getStorageId={getStorageId}
                    aspects={probeResultMap.get(index)?.data?.aspects}
                 />)}
              </Table.Body>
            </Table>}
        </Segment>
      </>
    </Segment>
  );
};

export default ImportCameraList;

function getProbeInput(entry: ImportCameraEntry): DeviceProbeInput {
  const currentConfigType: DeviceBaseConfigType = entry.deviceType === DeviceConfigType.Onvif ? DeviceBaseConfigType.Onvif : DeviceBaseConfigType.Url;
  const configType: DeviceBaseConfigType = currentConfigType;
  const connect: NetworkConnectDescriptorInput = { authType: AuthType.NoAuth };
  const port = isNaN(Number(entry.httpPort)) ? null : Number(entry.httpPort);
  const rtcpPort = isNaN(Number(entry.rtcpPort)) ? null : Number(entry.rtcpPort);

  switch (configType) {
    case DeviceBaseConfigType.Onvif:
      connect.host = entry.deviceAddress;
      connect.port = port;
      connect.rtspPort = rtcpPort;
      connect.user = entry.username;
      connect.pass = entry.password;
      break;
    case DeviceBaseConfigType.Url:
      connect.URL = entry.deviceUrl;
      break;
  }

  const probeInput: DeviceProbeInput = {
    configType,
    connect,
    platformId: entry.avatarId
  };

  return probeInput;
}

function getDeviceInput(entry: ImportCameraEntry, probeData?: ProbeDeviceQuery["probeDevice"], zones?: ZonesQuery["zones"]): DeviceInput | undefined {
  if (!probeData) return undefined;

  const currentConfigType: DeviceBaseConfigType = entry.deviceType === DeviceConfigType.Onvif ? DeviceBaseConfigType.Onvif : DeviceBaseConfigType.Url;
  const configType: DeviceBaseConfigType = currentConfigType;
  const connect: NetworkConnectDescriptorInput = { authType: AuthType.NoAuth };
  const config: DeviceBaseConfigInput = { configType, connect };
  const port = isNaN(Number(entry.httpPort)) ? null : Number(entry.httpPort);
  const rtcpPort = isNaN(Number(entry.rtcpPort)) ? null : Number(entry.rtcpPort);
  const aspects = probeData.aspects.map(aspect => {
    if (aspect.__typename === "DFA_Media" && entry.onvifProfile === aspect.onvifProfile && aspect.streamType === MediaStreamType.Video) {
      aspect.enabled = true;
    }
    else {
      aspect.enabled = false;
    }
    return aspect;
  });

  switch (configType) {
    case DeviceBaseConfigType.Onvif:
      connect.host = entry.deviceAddress;
      connect.port = port;
      connect.rtspPort = rtcpPort;
      connect.user = entry.username;
      connect.pass = entry.password;
      config.make = probeData.config.make ??  null;
      config.model = probeData.config.model ?? null;
      config.firmware = probeData.config.firmware ??  null;
      config.onvifServices = probeData.config.onvifServices ?? null;
      break;
    case DeviceBaseConfigType.Url:
      connect.URL = entry.deviceUrl;
      connect.user = entry.username;
      connect.pass = entry.password;
      break;
  }

  const input: DeviceInput = {
    name: entry.deviceName ?? "",
    enabled: true,
    location: entry.deviceLocation,
    zoneId: zones?.find(zone => zone.name === entry.deviceZone)?.id ?? entry.deviceZone,
    platformId: entry.avatarId,
    config,
    storageConfig: {
      storagePoolId: entry.storagePool?.toString() ?? "",
    },
    aspects: aspects.map(aspect => aspectToAspectInput(aspect))
  };

  return input;
}
