import React, {useState, useEffect, useRef, useMemo, useCallback} from "react";
import {
  Segment,
  Button,
  Icon,
  Tab,
  Table,
  Checkbox,
  Dropdown,
  DropdownItemProps,
  Message,
  Popup,
  Modal,
  Header,
  Radio
} from "semantic-ui-react";
import classNames from "classnames";
import diff from "microdiff";
import { useLocation } from "react-router-dom";
import {
  DeviceBaseConfigType,
  DeviceFunctionalAspectType,
  useZonesQuery,
  useProbeDeviceLazyQuery,
  MediaArchiveOption,
  MediaStreamType,
  AuthType,
  useStoragePoolsQuery,
  DeviceInput,
  useCreateDeviceMutation,
  useUpdateDeviceMutation,
  useDeviceTemplatesQuery,
  NetworkConnectDescriptorInput,
  DeviceBaseConfigInput,
  DeviceFunctionalAspectInput, LatLngInput, useDevicesByAspectTypesShortLazyQuery, Dfa_Ptz, Dfa_Media, ObjectDescriptor, DevicesByAspectTypesShortQuery, HealthStatus, WarningField, StreamShare
} from "@generated/graphql";
import {Device, Aspect, aspectToAspectInput, useDeviceActions, DeviceShort, getEntityWithoutExceptions} from "@core/actions";
import WithQueryStatus from "components/WithQueryStatus";
import {AutoForm, AutoLayout, FieldSchema, FieldValues, FormSchema} from "components/AutoForm";
import Loading from "components/Loading";
import EventList from "components/EventList";
import EventInfoComponent from "components/EventInfo";
import {queryToInput} from "utils";
import {clone} from "@solid/libs/utils";
import {Log} from "@solid/libs/log";
import {__} from "@solid/libs/i18n";
import GEOMap from "components/GEOMap";
import ControlSetPanels from "components/Admin/Sets/ControlSetPanels/ControlSetPanels";
import Report from "components/Admin/Report";
import { Type } from "@solid/types";
import CameraLogs from "components/Admin/CameraLogs";
import { useAccessability } from "@core/store/actions/accessability";
import IconButton from "components/IconButton";
import StreamSnapshot from "./StreamSnapshot";
import SameDevicesModal from "components/Admin/SameDevicesModal";
import SharingStream from "./SharingStream";

import "./style.css";

export type CreateUpdateCameraProps = {
  cameras?: DevicesByAspectTypesShortQuery;
  device?: Device;
  onBack?: (createdDeviceIds?: string[]) => void;
};

export type MediaAspect = Aspect & {
  snapshot: string;
};

const edgeArchiveOptions: DropdownItemProps[] = [
  // {key: MediaArchiveOption.None, value: MediaArchiveOption.None, text: __("None")},
  {key: MediaArchiveOption.Continuous, value: MediaArchiveOption.Continuous, text: __("Continuous")},
  // {key: MediaArchiveOption.Events, value: MediaArchiveOption.Events, text: __("Events")},
];

const cloudArchiveOptions: DropdownItemProps[] = [
  {key: MediaArchiveOption.None, value: MediaArchiveOption.None, text: __("None")},
  {key: MediaArchiveOption.Continuous, value: MediaArchiveOption.Continuous, text: __("Continuous")},
  {key: MediaArchiveOption.Events, value: MediaArchiveOption.Events, text: __("Events")},
  {key: MediaArchiveOption.Live, value: MediaArchiveOption.Live, text: __("Live")},
];

const captureFrameRateOptions: DropdownItemProps[] = [
  {key: 1, value: 1, text: "1"},
  {key: 2, value: 2, text: "2"},
  {key: 3, value: 3, text: "3"},
  {key: 4, value: 4, text: "4"},
  {key: 5, value: 5, text: "5"},
  {key: 6, value: 6, text: "6"},
  {key: 7, value: 7, text: "7"},
  {key: 8, value: 8, text: "8"},
  {key: 9, value: 9, text: "9"},
  {key: 10, value: 10, text: "10"},
  {key: 11, value: 11, text: "11"},
  {key: 12, value: 12, text: "12"},
  {key: 15, value: 15, text: "15"},
  {key: 20, value: 20, text: "20"},
  {key: 25, value: 25, text: "25"},
  {key: 30, value: 30, text: "30"},
];

type UrlComponents = {
  protocol: string;
  user: string;
  pass: string;
  host: string;
  port: string;
  path: string;
  parameters: string;
};

type LinkIdFromLocation = {
  linkId?: string;
};

const CreateUpdateCamera = ({device, onBack, cameras}: CreateUpdateCameraProps) => {
  const [getAvatars, {data: avatarData, error: avatarError, loading: avatarLoading}] = useDevicesByAspectTypesShortLazyQuery({variables: {types: [{type: DeviceFunctionalAspectType.Avatar}]}});
  const {data: zoneData, error: zoneError, loading: zoneLoading} = useZonesQuery();
  const [probeDevice, {data: probeData, error: probeError, loading: probeLoading}] = useProbeDeviceLazyQuery({fetchPolicy: "no-cache"});
  const {data: storageData, error: storageError, loading: storageLoading} = useStoragePoolsQuery();
  const {data: devTemplData, error: devTemplError, loading: devTemplLoading} = useDeviceTemplatesQuery();
  const [error, setError] = useState<Error | undefined>();
  const { config: accessConfig } = useAccessability();
  const [activeTab, setActiveTab] = useState(0);
  const [mediaAspects, setMediaAspects] = useState<MediaAspect[]>([]);
  const [getCsvFile, setGetCsvFile] = useState<boolean>(false);
  const [ptzAspect, setPtzAspect] = useState<Dfa_Ptz | undefined>();
  const [vaeAspects, setVaeAspects] = useState<DeviceFunctionalAspectInput[]>([]);
  const [deviceSets, setDeviceSets] = useState<ObjectDescriptor[]>([]);
  const [createDevice, {/*data: createData,*/ error: createError, loading: createLoading}] = useCreateDeviceMutation();
  const [updateDevice, {/*data: updateData,*/ error: updateError, loading: updateLoading}] = useUpdateDeviceMutation();
  const {updateCachedDevice} = useDeviceActions({skipSubscription: true});
  const [refetching, setRefetching] = useState(false);
  const [mediaError, setMediaError] = useState("");
  const [paramsChanged, setParamsChanged] = useState(false);
  const [templateFormOpen, setTemplateFormOpen] = useState(false);
  const [templateValues, setTemplateValues] = useState<FieldValues>({});
  const [isChanged, setIsChanged] = useState(!device);
  const [isValid, setIsValid] = useState(true);
  const [position, setPosition] = useState<LatLngInput | null | undefined>(device?.position);
  const probingValues = useRef<FieldValues | undefined>();
  const probedValues = useRef<FieldValues | undefined>();
  const isDiscoveryDevice = !!device && device.config.configType !== DeviceBaseConfigType.Onvif && device.config.configType !== DeviceBaseConfigType.Url;
  const isSupportedDevice = !device || device.version === "2.0";
  const isDemoDevice = !!device && device.config.configType === DeviceBaseConfigType.Demo;
  const [isCloudDirect, setIsCloudDirect] = useState(false);
  const [mediaAspectWithStats, setMediaAspectWithStats] = useState<Dfa_Media>();
  const [auditTabWasActive, setAuditTabWasActive] = useState(false);
  const [gisTabWasActive, setGisTabWasActive] = useState(false);
  const [cameraExist, setCameraExist] = useState<boolean>(false);
  const [sameCamerasName, setSameCamerasName] = useState<string[]>();
  const [isCameraLogOpen, setIsCameraLogOpen] = useState<boolean>(false);
  const [rtpOverTcp, setRtpOverTcp] = useState<boolean>(false);
  const [rtpTime, setRtpTime] = useState<boolean>(false);
  const [captureFrameRate, setCaptureFrameRate] = useState<number>();
  const [avatarOnline, setAvatarOnline] = useState(false);
  const [updatedDevice, setUpdatedDevice] = useState<Device>();
  const [sharedStream, setSharedStream] = useState<StreamShare[] | null>();
  const { state } = useLocation();
  const linkId: string | undefined = (state as LinkIdFromLocation)?.linkId;

  const isCameraUrl:boolean = useMemo(() => {
    return (probeData?.probeDevice.config.configType === DeviceBaseConfigType.Url) ||
      (updatedDevice?.config.configType === DeviceBaseConfigType.Url || device?.config.configType === DeviceBaseConfigType.Url);
  }, [device, updatedDevice, probeData]);

  const audioAspects = useMemo(() => {
    return mediaAspects.filter(aspect => checkIsAspectAudio(aspect));
  }, [mediaAspects]);

  const mediaAspectUrl = useMemo(() => {
    const enabledAspect = mediaAspects.find(aspect => aspect.enabled);
    if (enabledAspect && enabledAspect.__typename === "DFA_Media") {
      return enabledAspect.onvifProfiles?.attributes?.DEVURL;
    }

    return undefined;
  }, [mediaAspects]);

  const isMediaDevice: boolean = useMemo(() => {
    const currentDevice = updatedDevice ? updatedDevice : device;
    return !!currentDevice?.aspects.find(aspect => isDfaMedia(aspect));
  }, [device, updatedDevice]);

  const schema = useMemo(() => {
    const devicePlatformId = device?.platform ? device.platform.id : "";
    const devicePlatformName = device?.platform ? device.platform.name : "";

    const destinationDataSource = device && !device.platform ?
      [{name: "Cloud Direct", id: "cloudDirect"}] :
      [{name: "Avatar", id: "avatar"}, {name: "Cloud Direct", id: "cloudDirect"}];

    const platformIdDataSource = isCloudDirect ?
      [{id: "", name: ""}] :
      device ?
        Array.from([{id: devicePlatformId, name: devicePlatformName}] ?? []) :
        Array.from(avatarData?.devicesByAspectTypes ?? []).sort((a, b) => a.name.localeCompare(b.name, undefined, {sensitivity: "base"}));

    const schema: FormSchema = [
      {
        name: "destination",
        label: __("Destination"),
        type: "dropdown",
        required: true,
        dataSource: Array.from(destinationDataSource ?? []),
        disableCondition: () => !!device
      },
      {
        name: "platformId",
        label: __("Avatar"),
        type: "dropdown",
        required: true,
        dataSource: Array.from(platformIdDataSource ?? []),
        disableCondition: () => isDiscoveryDevice || !!linkId || !isSupportedDevice || !!device?.platform?.id,
        hideCondition: values => values["destination"] === "cloudDirect",
      },
      {
        name: "name",
        label: __("Name"),
        required: true
      },
      {
        name: "state",
        label: __("State"),
        type: "dropdown",
        required: true,
        dataSource: [
          {id: "on", name: __("On")},
          {id: "off", name: __("Off")},
        ]
      },
      {
        name: "location",
        label: __("Location"),
      },
      !accessConfig?.limitedZonesAccess ?
        {
          name: "zoneId",
          label: __("Zone"),
          type: "dropdown",
          dataSource: Array.from(zoneData?.zones ?? []).sort((a, b) => a.name.localeCompare(b.name, undefined, {sensitivity: "base"}))
        } :
        undefined,
      {
        name: "configDiscovery",
        label: __("Config via"),
        readOnlyCondition: () => true,
        hideCondition: () => !isDiscoveryDevice
      },
      {
        name: "configType",
        label: __("Config via"),
        type: "dropdown",
        required: true,
        dataSource: [
          {id: DeviceBaseConfigType.Onvif, name: __("ONVIF")},
          {id: DeviceBaseConfigType.Url, name: __("URL")},
        ],
        hideCondition: () => isDiscoveryDevice,
        disableCondition: () => !isSupportedDevice
      },
      {
        name: "templateButton",
        label: __("Get URL From..."),
        type: "button",
        hideCondition: values => isDiscoveryDevice || values["configType"] !== DeviceBaseConfigType.Url || !isSupportedDevice,
        action: () => setTemplateFormOpen(true)
      },
      {
        name: "host",
        label: __("Camera IP/Host"),
        required: true,
        hideCondition: values => isDiscoveryDevice || values["configType"] !== DeviceBaseConfigType.Onvif,
        disableCondition: () => !isSupportedDevice
      },
      {
        name: "httpPort",
        label: __("HTTP Port"),
        type: "integer",
        required: true,
        minValue: 1,
        maxValue: 65535,
        hideCondition: values => isDiscoveryDevice || values["configType"] !== DeviceBaseConfigType.Onvif,
        disableCondition: () => !isSupportedDevice
      },
      {
        name: "rtspPort",
        label: __("RTSP Port"),
        type: "integer",
        required: true,
        minValue: 1,
        maxValue: 65535,
        hideCondition: values => isDiscoveryDevice || values["configType"] !== DeviceBaseConfigType.Onvif,
        disableCondition: () => !isSupportedDevice
      },
      {
        name: "url",
        label: __("URL"),
        required: true,
        hideCondition: values => isDiscoveryDevice || values["configType"] !== DeviceBaseConfigType.Url,
        disableCondition: () => !isSupportedDevice,
        validate: (schema, value, values) => {
          const {protocol, host} = parseUrl(value);
          if (!protocol || !host) {
            return __("Invalid URL");
          }
          return "";
        }
      },
      {
        name: "user",
        label: __("Username"),
        hideCondition: () => isDiscoveryDevice,
        disableCondition: () => !isSupportedDevice
      },
      {
        name: "pass",
        label: __("Password"),
        type: "password",
        hideCondition: () => isDiscoveryDevice,
        disableCondition: () => !isSupportedDevice
      },
    ];

    if (!accessConfig?.limitedZonesAccess) {
      const zonesDropdown: FieldSchema = {
        name: "zoneId",
        label: __("Zone"),
        type: "dropdown",
        dataSource: Array.from(zoneData?.zones ?? []).sort((a, b) => a.name.localeCompare(b.name, undefined, {sensitivity: "base"}))
      };
      schema.splice(4, 0, zonesDropdown);
    }

    return schema;
  }, [avatarData, zoneData, isDiscoveryDevice, linkId, isSupportedDevice, isCloudDirect, device, accessConfig]);

  const connectFields = ["platformId", "configType", "host", "httpPort", "rtspPort", "user", "pass", "url"];

  const values = useMemo(() => {
    if (device) {
      const currentDevice = updatedDevice ? updatedDevice : device;
      const {name, enabled, location, zone, platform, config: {configType, connect: {URL, host, port, rtspPort, user, pass}}} = currentDevice;
      return {
        name,
        state: enabled ? "on" : "off",
        location,
        zoneId: zone?.id ?? "",
        platformId: platform?.id,
        configDiscovery: __("DISCOVERY"),
        configType,
        host,
        httpPort: port ?? 80,
        rtspPort: rtspPort ?? 554,
        url: URL,
        user,
        pass,
      };
    }
    return {
      state: "on",
      platformId: linkId,
      httpPort: 80,
      rtspPort: 554,
    };
  }, [device, linkId, updatedDevice]);

  const formRef = useRef<AutoForm>(null);

  const templateSchema = useMemo(() => {
    const schema: FormSchema = [
      {
        name: "make",
        label: __("Camera Make"),
        type: "dropdown",
        required: true,
        dataSource: Array.from(devTemplData?.deviceTemplates ?? [])
          .sort((a, b) => a.make.localeCompare(b.make, undefined, {sensitivity: "base"}))
          .map(({make}) => ({id: make, name: make}))
      },
      {
        name: "model",
        label: __("Camera Model"),
        type: "dropdown",
        required: true,
        dataSource: values => devTemplData?.deviceTemplates
          .find(templ => templ.make === values["make"])
          ?.models
          .map(({model}) => ({id: model, name: model})) ?? []
      },
      {
        name: "host",
        label: __("Camera IP/Host"),
        required: true
      },
      {
        name: "port",
        label: __("Port"),
        type: "integer",
        minValue: 1,
        maxValue: 65535
      },
      {
        name: "user",
        label: __("Username"),
      },
      {
        name: "pass",
        label: __("Password"),
        type: "password",
      },
      {
        name: "computedUrl",
        label: __("URL"),
        required: true,
        readOnlyCondition: () => true
      },
    ];
    return schema;
  }, [devTemplData]);

  const templateFormRef = useRef<AutoForm>(null);

  const storageSchema = useMemo(() => {
    const schema: FormSchema = [
      {
        name: "storagePoolId",
        label: __("Storage Pool"),
        type: "dropdown",
        required: true,
        dataSource: Array.from(storageData?.storagePools ?? []).sort((a, b) => a.name.localeCompare(b.name, undefined, {sensitivity: "base"}))
      },
    ];
    return schema;
  }, [storageData]);

  const storageValues = useMemo(() => {
    return {storagePoolId: updatedDevice?.storageConfig?.storagePool.id || device?.storageConfig?.storagePool.id};
  }, [device, updatedDevice]);

  const storageFormRef = useRef<AutoForm>(null);

  const ptzSchema = useMemo(() => {
    const schema: FormSchema = [
      {
        name: "enabled",
        label: __("Enable PTZ"),
        type: "checkbox",
        disableCondition: () => !ptzAspect
      },
      {
        name: "ptzPanInverted",
        label: "Invert Pan",
        type: "checkbox",
        required: true,
        hideCondition: (values) => values["enabled"] !== true
      },
      {
        name: "ptzTiltInverted",
        label: "Invert Tilt",
        type: "checkbox",
        required: true,
        hideCondition: (values) => values["enabled"] !== true
      }
    ];
    return schema;
  }, [ptzAspect]);

  const ptzValues = useMemo(() => {
    if (ptzAspect) {
      return {
        enabled: !!ptzAspect?.enabled,
        ptzPanInverted: ptzAspect.ptzPanInverted !== null && ptzAspect.ptzPanInverted !== undefined ? ptzAspect.ptzPanInverted : false,
        ptzTiltInverted: ptzAspect.ptzTiltInverted !== null && ptzAspect.ptzTiltInverted !== undefined ? ptzAspect.ptzTiltInverted : false,
      };
    }
    return {enabled: false};
  }, [ptzAspect]);

  const ptzFormRef = useRef<AutoForm>(null);

  useEffect(() => {
    if (avatarData && device) {
      setAvatarOnline(avatarData.devicesByAspectTypes.find(avatar => avatar.id === device.platform?.id)?.healthStatus === HealthStatus.Normal);
    }
  }, [avatarData, device]);

  const onRtpValuesChanged = ({ property, value }: { property: "rtpOverTcp" | "rtpTime", value: boolean }) => {
    switch (property) {
      case "rtpOverTcp":
        setRtpOverTcp(value);
        break;
      case "rtpTime":
        setRtpTime(value);
    }
  };

  const onFrameRateChange = useCallback((value: number) => {
    if (!isDemoDevice) {
      return;
    }
    setCaptureFrameRate(value);
    const newMediaAspects = mediaAspects.map(aspect => {
      if (aspect.__typename === "DFA_Media" &&
          aspect.type === DeviceFunctionalAspectType.Media &&
          aspect.streamType === MediaStreamType.Video)
      {
        aspect.frameRate = value;
      }
      return aspect;
    });
    setMediaAspects(newMediaAspects);
  }, [mediaAspects, isDemoDevice]);

  const cancel = () => {
    onBack && onBack();
  };

  const positionSchema = useMemo(() => {
    const schema: FormSchema = [
      {
        name: "lat",
        label: __("Lat"),
        type: "float",
        minValue: -90,
        maxValue: 90
      },
      {
        name: "lng",
        label: __("Lng"),
        type: "float",
        minValue: -180,
        maxValue: 180
      }
    ];
    return schema;
  }, [position]);

  const positionValues = useMemo(() => {
    return {...position};
  }, [position]);

  const positionFormRef = useRef<AutoForm>(null);

  useEffect(() => {
    if (!device) {
      getAvatars();
      setMediaAspects([]);
      setPtzAspect(undefined);
      setVaeAspects([]);
      setRtpOverTcp(false);
      setRtpTime(true);
      setUpdatedDevice(undefined);
      return;
    }

    const currentDevice = updatedDevice ? updatedDevice : device;

    setRtpOverTcp(!!currentDevice.rtpOverTcp);
    setRtpTime(!!currentDevice.rtpTime);

    if (currentDevice.aspects.some(aspect => aspect.__typename === "DFA_Media")) {
      const dfaMediaAspectWithStats = device.aspects
        .find(aspect => aspect.__typename === "DFA_Media" &&
          aspect.type === DeviceFunctionalAspectType.Media &&
          aspect.streamType === MediaStreamType.Video);
      if ((dfaMediaAspectWithStats as Dfa_Media)?.frameRate) {
        const { frameRate } = dfaMediaAspectWithStats as Dfa_Media;
        if (frameRate) {
          setCaptureFrameRate(frameRate);
        }
      }
      if (dfaMediaAspectWithStats) {
        setMediaAspectWithStats(dfaMediaAspectWithStats as Dfa_Media);
      }
    }

    const mediaAspects = currentDevice.aspects
      .filter(aspect => aspect.type === DeviceFunctionalAspectType.Media)
      .map(aspect => ({...aspect, snapshot: ""}));
    setMediaAspects(mediaAspects);

    setPtzAspect(currentDevice.aspects.find(aspect => aspect.type === DeviceFunctionalAspectType.Ptz) as Dfa_Ptz);

    setVaeAspects(currentDevice.aspects
      .filter(aspect => aspect.type === DeviceFunctionalAspectType.Vae)
      .map(aspect => aspectToAspectInput(aspect)));

    if (currentDevice.set) {
      setDeviceSets(currentDevice.set as ObjectDescriptor[]);
    }

    if (currentDevice.position) {
      setPosition(currentDevice.position);
    }

    const aspect = currentDevice.aspects.find(aspect => aspect.hasOwnProperty("sharedStream"));
    if (aspect && isDfaMedia(aspect) && isMediaDevice) {
      const deviceSharedStream = aspect.sharedStream;
      setSharedStream(deviceSharedStream);
    }
  }, [device, updatedDevice]);

  useEffect(() => {
    if (device) {
      probedValues.current = clone(values);
    }
  }, [values]);

  useEffect(() => {
    if (!probeData) {
      return;
    }

    if (probingValues.current) {
      probedValues.current = clone(probingValues.current);
    }
    setParamsChanged(!!formRef.current?.areValuesChanged(connectFields, probedValues.current));

    if (!probeData.probeDevice) {
      setMediaAspects([]);
      setPtzAspect(undefined);
      return;
    }

    const mediaAspects = probeData.probeDevice.aspects.filter(aspect => aspect.type === DeviceFunctionalAspectType.Media);
    const enabledMediaAspectIndex: number = isCloudDirect ? mediaAspects.findIndex(aspect => aspect.__typename === "DFA_Media" && aspect.codec === "h264") : 0;

    if (typeof probeData.probeDevice.rtpTime === "boolean") {
      setRtpTime(probeData.probeDevice.rtpTime);
    }

    setMediaAspects(values => {
      const newValues = mediaAspects.map((aspect, index) => {
        if (isCloudDirect && index !== enabledMediaAspectIndex) {
          aspect.enabled = false;
        } else if (index !== enabledMediaAspectIndex) {
          aspect.enabled = false;
        }
        if (aspect.__typename === "DFA_Media") {
          aspect.edgeArchive = MediaArchiveOption.Continuous;
        }
        const snapshot = index < probeData.probeDevice.snapshots.length ? probeData.probeDevice.snapshots[index] : "";
        if (aspect.__typename !== "DFA_Media") {
          return {...aspect, snapshot};
        }
        const prevValue = values.find(value =>
          value.__typename === "DFA_Media" &&
          value.name === aspect.name &&
          value.streamType === aspect.streamType &&
          (value.URL ?? "") === (aspect.URL ?? "") &&
          (value.videoWidth ?? 0) === (aspect.videoWidth ?? 0) &&
          (value.videoHeight ?? 0) === (aspect.videoHeight ?? 0) &&
          (value.codec ?? "") === (aspect.codec ?? "") &&
          value.enabled === aspect.enabled);
        if (prevValue && aspect) {
          const {onvifProfile, onvifProfiles} = aspect;
          return {...prevValue, onvifProfile, onvifProfiles, snapshot};
        }
        return {...aspect, snapshot};
      });
      return newValues;
    });

    setPtzAspect(value => {
      const newValue = probeData.probeDevice.aspects.find(aspect => aspect.type === DeviceFunctionalAspectType.Ptz);
      if (newValue && value) {
        return value as Dfa_Ptz;
      }
      return newValue as Dfa_Ptz;
    });

    setActiveTab(1);
  }, [probeData, isCloudDirect]);

  useEffect(() => {
    const defaultInput = mediaAspects.find(aspect => aspect.template.id === "video" && aspect.enabled);
    if (!defaultInput) {
      return;
    }
    setVaeAspects(values => {
      const newValues = values.map(value => {
        const devAspect = device?.aspects.find(aspect => aspect.id === value.id);
        if (devAspect && devAspect.inputAspects.length > 0 && mediaAspects.some(aspect => aspect.id === devAspect.inputAspects[0].id && aspect.enabled)) {
          return {...value, inputAspectIDs: [devAspect.inputAspects[0].id]};
        }
        if (value.inputAspectIDs.length > 0 && !mediaAspects.some(aspect => aspect.id === value.inputAspectIDs[0] && aspect.enabled)) {
          return {...value, inputAspectIDs: [defaultInput.id]};
        }
        return value;
      });
      return newValues;
    });
  }, [mediaAspects]);

  useEffect(() => {
    if (probeError) {
      setError(probeError);
    }
  }, [probeError]);

  useEffect(() => {
    validateMedia();
    updateIsChanged();
  }, [mediaAspects]);

  useEffect(() => {
    updateIsValid();
  }, [mediaError]);

  function getUpdatedDevice(): Device | undefined {
    if (!device) {
      return undefined;
    }
    const updatedDevice = {...device};
    if (formRef.current) {
      const values = formRef.current.getValues();
      const connect = {
        URL: values["url"],
        host: values["host"],
        httpPort: values["port"],
        rtspPort: values["rtspPort"],
        user: values["user"],
        pass: values["pass"],
      };
      Object.assign(updatedDevice, {
        name: values["name"],
        enabled: values["state"] === "on",
        location: values["location"],
        zone: Array.from(zoneData?.zones ?? []).find(zone => zone.id === values["zoneId"]),
        config: {
          configType: values["configType"],
          connect
        },
        rtpOverTcp,
        rtpTime
      });
    }

    if (storageFormRef.current) {
      const values = storageFormRef.current.getValues();
      const storagePool =  Array.from(storageData?.storagePools ?? []).find(pool => pool.id === values["storagePoolId"]);
      Object.assign(updatedDevice, {
        storageConfig: {...device.storageConfig, storagePool}
      });
    }
    if (position && typeof position.lat === "number" && typeof position.lng === "number") {
      updatedDevice["position"] = { lat: position.lat, lng: position.lng, alt: position.alt };
    }

    updatedDevice.aspects = mediaAspects.map(aspect => {
      const { snapshot, ...fields } = aspect;
      return {...fields, sharedStream};
    });

    if (ptzFormRef.current && ptzAspect) {
      const ptzValues = ptzFormRef.current.getValues();
      updatedDevice.aspects.push({
        ...ptzAspect,
        enabled: ptzValues["enabled"],
        ptzTiltInverted: ptzValues["ptzTiltInverted"],
        ptzPanInverted: ptzValues["ptzPanInverted"]
      });
    }

    const vaeAspects = device.aspects.filter(aspect => aspect.type === DeviceFunctionalAspectType.Vae);
    updatedDevice.aspects?.push(...vaeAspects);

    if (deviceSets) {
      updatedDevice["set"] = deviceSets;
    }

    return updatedDevice;
  }

  function updateIsChanged(): void {
    if (!device) {
      setIsChanged(true);
      return;
    }
    const currentDevice = updatedDevice ? updatedDevice : device;

    let isChanged = false;
    if (!!formRef.current?.getIsChanged() ||
        !!storageFormRef.current?.getIsChanged() ||
        !!ptzFormRef.current?.getIsChanged() ||
        currentDevice.rtpOverTcp !== rtpOverTcp ||
        currentDevice.rtpTime !== rtpTime) {
      isChanged = true;
      setIsChanged(isChanged);
      return;
    }

    if (position && (position.lng !== currentDevice.position?.lng || position.lat !== currentDevice.position?.lat)) {
      isChanged = true;
      setIsChanged(isChanged);
      return;
    }

    const prevMediaAspects = currentDevice.aspects.filter(aspect => aspect.type === DeviceFunctionalAspectType.Media);
    const nextMediaAspects = mediaAspects.map(aspect => {
      const {snapshot, ...fields} = aspect;
      return {...fields};
    });
    if (diff(prevMediaAspects, nextMediaAspects).length !== 0) {
      isChanged = true;
      setIsChanged(isChanged);
      return;
    }

    if (currentDevice.set || deviceSets) {
      if (currentDevice.set?.length !== deviceSets?.length) {
        isChanged = true;
        setIsChanged(isChanged);
        return;
      }
      const initialSets = currentDevice.set?.sort((a, b) => a.id.localeCompare(b.id));
      const newSets = deviceSets?.sort((a, b) => a.id.localeCompare(b.id));
      if (JSON.stringify(initialSets) !== JSON.stringify(newSets)) {
        isChanged = true;
        return;
      }
    }

    const aspectWithSharedStream = currentDevice.aspects.find(aspect => isDfaMedia(aspect) && aspect.hasOwnProperty("sharedStream"));
    if (aspectWithSharedStream && isDfaMedia(aspectWithSharedStream)) {
      const prevStreams = aspectWithSharedStream.sharedStream?.sort((a, b) => a.id - b.id) ?? [];
      const newStreams = sharedStream?.sort((a, b) => a.id - b.id) ?? [];
      if (diff(prevStreams, newStreams).length !== 0) {
        isChanged = true;
        setIsChanged(isChanged);
        return;
      }
    }

    setIsChanged(isChanged);
  }

  function updateIsValid(): void {
    setIsValid(!!formRef.current?.getIsValid() &&
      !!storageFormRef.current?.getIsValid() &&
      !!ptzFormRef.current?.getIsValid() &&
      !mediaError &&
      !position || !!position && ((position.lat !== null && position.lng !== null) || (position.lat === null && position.lng === null))
    );
  }

  function onFormChange(name: string, value: any, values: FieldValues, form: AutoForm): void {
    updateIsChanged();

    if (!isDiscoveryDevice && isSupportedDevice && connectFields.includes(name)) {
      setParamsChanged(form.areValuesChanged(connectFields, probedValues.current));
    }

    if (name === "destination" && value === "cloudDirect") {
      form.setValue("platformId", "");
      setIsCloudDirect(true);
    }
    if (name === "destination" && value === "avatar") {
      setIsCloudDirect(false);
    }

    if (name === "configType") {
      form.setValue("host", "");
      form.setValue("httpPort", 80);
      form.setValue("rtspPort", 554);
      form.setValue("user", "");
      form.setValue("pass", "");
      form.setValue("url", "");
    }

    if (values["configType"] === DeviceBaseConfigType.Url) {
      const url = values["url"];
      if (!url) {
        return;
      }
      const {protocol, host, user, pass} = parseUrl(url);
      if (!protocol || !host) {
        return;
      }

      if (name === "url") {
        form.setValue("user", user);
        form.setValue("pass", pass);
      } else if (["user", "pass"].includes(name)) {
        form.setValue("url", changeUrlUserPass(url, values));
      }
    }
  }

  function setNewPosition(position: LatLngInput) {
    positionFormRef.current?.setValue("lat", position.lat);
    positionFormRef.current?.setValue("lng", position.lng);
    setPosition(position);
  }

  useEffect(() => {
    updateIsChanged();
    updateIsValid();
  }, [position]);

  function onTemplateFormChange(name: string, value: any, values: FieldValues, form: AutoForm): void {
    const urlTemplate = devTemplData?.deviceTemplates
      .find(templ => templ.make === values["make"])
      ?.models
      .find(m => m.model === values["model"])?.url ?? "";

    if (!urlTemplate) {
      return;
    }

    if (["make", "model"].includes(name)) {
      const {host, port, user, pass} = parseUrl(urlTemplate);
      form.setValue("host", host);
      form.setValue("port", port);
      form.setValue("user", user);
      form.setValue("pass", pass);
      form.setValue("computedUrl", urlTemplate);
    } else if (["host", "port", "user", "pass"].includes(name)) {
      form.setValue("computedUrl", computeUrl(urlTemplate, values));
    }
  }

  useEffect(() => {
    if (getCsvFile) {
      setGetCsvFile(false);
    }
  }, [getCsvFile]);

  useEffect(() => {
    updateIsChanged();
  }, [deviceSets]);

  useEffect(() => {
    updateIsChanged();
  }, [rtpOverTcp, rtpTime, sharedStream]);

  function checkIsAspectAudio(aspect?: MediaAspect | null): boolean {
    return !!(aspect && aspect.__typename === "DFA_Media" && aspect.streamType === MediaStreamType.Audio);
  }

  function parseUrl(url: string): UrlComponents {
    /* eslint-disable no-useless-escape */
    const urlRegEx = /(http|https|rtsp|udp|rtp):\/\/(?:([^\s@\/]+?)[@])?([^\s\/:]+)(?:[:]([0-9]+))?(?:(\/[^\s?#]+)([?][^\s#]+)?)?([#]\S*)?/;
    /* eslint-enable no-useless-escape */
    const urlMatch = urlRegEx.exec(url);
    if (!urlMatch) {
      return {protocol: "", user: "", pass: "", host: "", port: "", path: "", parameters: ""};
    }
    const [, protocol, credentials, host, port, path, parameters] = urlMatch;
    let user = "";
    let pass = "";
    if (credentials) {
      const credMatch = /(.*):(.*)/.exec(credentials);
      if (credMatch) {
        user = decodeURIComponent(credMatch[1]);
        pass = decodeURIComponent(credMatch[2]);
      } else {
        user = decodeURIComponent(credentials);
      }
    }
    return {protocol, user, pass, host, port, path, parameters};
  }

  function computeUrl(urlTemplate: string, values: FieldValues): string {
    const host = values["host"] || "ip";
    const port = values["port"];
    const user = values["user"];
    const pass = values["pass"];
    const hostAndPort = port ? `${host}:${port}` : host;
    const {protocol, path, parameters} = parseUrl(urlTemplate);
    let url = "";
    if (user && pass) {
      url = `${protocol}://${encodeURIComponent(user)}:${encodeURIComponent(pass)}@${hostAndPort}`;
    } else if (user) {
      url = `${protocol}://${encodeURIComponent(user)}@${hostAndPort}`;
    } else {
      url = `${protocol}://${hostAndPort}`;
    }
    if (path) {
      url += path;
      if (parameters) {
        url += parameters;
      }
    }
    return url;
  }

  function changeUrlUserPass(sourceUrl: string, values: FieldValues): string {
    const user = values["user"];
    const pass = values["pass"];
    const {protocol, host, port, path, parameters} = parseUrl(sourceUrl);
    if (!protocol || !host) {
      return sourceUrl;
    }
    const hostAndPort = port ? `${host}:${port}` : host;
    let url = "";
    if (user && pass) {
      url = `${protocol}://${encodeURIComponent(user)}:${encodeURIComponent(pass)}@${hostAndPort}`;
    } else if (user) {
      url = `${protocol}://${encodeURIComponent(user)}@${hostAndPort}`;
    } else {
      return sourceUrl;
    }
    if (path) {
      url += path;
      if (parameters) {
        url += parameters;
      }
    }
    return url;
  }

  function onTemplateFormClose(): void {
    const form = templateFormRef.current;
    if (!form) {
      setTemplateFormOpen(false);
      return;
    }

    if (!form.validate()) {
      return;
    }

    const values = form.getValues();
    if (formRef.current) {
      formRef.current.setValue("url", values["computedUrl"]);
      formRef.current.setValue("user", values["user"]);
      formRef.current.setValue("pass", values["pass"]);
      updateIsChanged();
      setParamsChanged(formRef.current.areValuesChanged(connectFields, probedValues.current));
    }
    setTemplateFormOpen(false);
    setTemplateValues(values);
  }

  function onActiveTabChange(tabIndex: number | string | undefined): void {
    if (typeof tabIndex !== "number") {
      return;
    }
    if (tabIndex > 0 && !device && (!probeData || paramsChanged)) {
      formRef.current?.validate();
      return; // disable switch to another tab if device is not probed
    }
    switch (activeTab) {
      case 0:
        if (!formRef.current?.validate()) {
          return;
        }
        break;
      case 3:
        if (!storageFormRef.current?.validate()) {
          return;
        }
        break;
      case 4:
        if (!ptzFormRef.current?.validate()) {
          return;
        }
        break;
      case 5:
        !auditTabWasActive && setAuditTabWasActive(true);
      case 6:
        !gisTabWasActive && setGisTabWasActive(true);
    }
    setActiveTab(tabIndex);
  }

  function onProbe(): void {
    if (!formRef.current?.validate()) {
      return;
    }

    setError(undefined);

    const values = formRef.current.getValues();
    probingValues.current = clone(values);
    const configType: DeviceBaseConfigType = values["configType"];
    const connect: NetworkConnectDescriptorInput = {authType: AuthType.NoAuth};
    switch (configType) {
      case DeviceBaseConfigType.Onvif:
        connect.host = values["host"];
        connect.port = values["httpPort"];
        connect.rtspPort = values["rtspPort"];
        connect.user = values["user"];
        connect.pass = values["pass"];
        break;
      case DeviceBaseConfigType.Url:
        connect.URL = values["url"];
        break;
    }
    probeDevice({
      variables: {
        input: {
          configType,
          connect,
          platformId: values["platformId"]
        }
      }
    });
  }

  function changeMediaAspect(id: string, action: (aspect: MediaAspect) => void): void {
    setMediaAspects(aspects => {
      const changedAspect = aspects.find(aspect => aspect.id === id);

      if (checkIsAspectAudio(changedAspect)) {
        aspects.forEach(aspect => {
          if (aspect.__typename === "DFA_Media" && aspect.streamType === MediaStreamType.Audio) {
            action(aspect);
          }
        });
        const newAspects = Array.from(aspects);
        return newAspects;
      }

      aspects.forEach(aspect => {
        if (aspect.id === id) {
          aspect.enabled = true;
        }
        else if (!checkIsAspectAudio(aspect)) {
          aspect.enabled = false;
        }
      });
      const index = aspects.findIndex(aspect => aspect.id === id);
      if (index < 0) {
        return aspects;
      }
      const newAspects = Array.from(aspects);
      newAspects[index] = {...newAspects[index]};
      action(newAspects[index]);
      return newAspects;
    });
  }

  function onCreateUpdate(): void {
    if (!formRef.current?.validate()) {
      setActiveTab(0);
      return;
    }
    if (mediaError) {
      setActiveTab(1);
      return;
    }
    if (!storageFormRef.current?.validate()) {
      setActiveTab(3);
      return;
    }
    if (!ptzFormRef.current?.validate()) {
      setActiveTab(4);
      return;
    }
    setError(undefined);
    const input = getDeviceInput();
    const sameCameras = getSameDevice();
    if (!device) {
      if (sameCameras) {
        setCameraExist(true);
        setSameCamerasName(sameCameras.map(cam => cam.name));
        return;
      }
      createCamera(input);
    } else {
      if (sameCameras) {
        if ((device.config.connect.URL && input.config.connect.URL !== device.config.connect.URL)
        || (device.config.connect.host && input.config.connect.host !== device.config.connect.host)) {
          setCameraExist(true);
          setSameCamerasName(sameCameras.map(cam => cam.name));
          return;
        }
      }
      updateCamera(device.id, input);
    }
  }

  function createUpdateAnyway() {
    setCameraExist(false);
    const input = getDeviceInput();
    if (!device) {
      createCamera(input);
    } else {
      updateCamera(device.id, input);
    }
  }

  function cancelCreating() {
    setCameraExist(false);
    setSameCamerasName(undefined);
  }

  function getDeviceInput(): DeviceInput {
    if (!formRef.current || !storageFormRef.current || !ptzFormRef.current) {
      throw new Error("Invalid form reference");
    }
    const values = formRef.current.getValues();
    const storageValues = storageFormRef.current.getValues();
    const ptzValues = ptzFormRef.current.getValues();
    const configType: DeviceBaseConfigType = values["configType"];
    const connect: NetworkConnectDescriptorInput = {authType: AuthType.NoAuth};
    const config: DeviceBaseConfigInput = {configType, connect};
    switch (configType) {
      case DeviceBaseConfigType.Onvif:
        connect.host = values["host"];
        connect.port = values["httpPort"];
        connect.rtspPort = values["rtspPort"];
        connect.user = values["user"];
        connect.pass = values["pass"];
        config.make = probeData?.probeDevice.config.make ?? device?.config.make ?? null;
        config.model = probeData?.probeDevice.config.model ?? device?.config.model ?? null;
        config.firmware = probeData?.probeDevice.config.firmware ?? device?.config.firmware ?? null;
        config.onvifServices = probeData?.probeDevice.config.onvifServices ?? device?.config.onvifServices ?? null;
        break;
      case DeviceBaseConfigType.Url:
        connect.URL = values["url"];
        connect.user = values["user"];
        connect.pass = values["pass"];
        break;
    }
    const input: DeviceInput = {
      name: values["name"],
      enabled: values["state"] === "on",
      location: values["location"],
      zoneId: values["zoneId"],
      rtpOverTcp,
      rtpTime,
      platformId: isSupportedDevice ? values["platformId"] ? values["platformId"] : device?.platform?.id : device?.platform?.id,
      config: device && isDiscoveryDevice ? queryToInput(device.config) : config,
      storageConfig: {
        storagePoolId: storageValues["storagePoolId"]
      },
      position: position ? {lat: position.lat, lng: position.lng, alt: position.alt} : null,
      aspects: mediaAspects.map(aspect => aspectToAspectInput(aspect)),
    };

    if (isMediaDevice && sharedStream && input.aspects) {
      const currentDevice = updatedDevice ? updatedDevice : device;
      const aspectWithSharedStream = currentDevice?.aspects.find(aspect => aspect.hasOwnProperty("sharedStream"));
      if (aspectWithSharedStream && isDfaMedia(aspectWithSharedStream)) {
        const prevSharedStreams = aspectWithSharedStream.sharedStream?.map(stream => stream.url).sort() ?? [];
        const currentSharedStreams = sharedStream.map(stream => stream.url).sort();
        if (diff(prevSharedStreams, currentSharedStreams).length !== 0) {
          for (const aspect of input.aspects) {
            aspect.sharedStream = currentSharedStreams;
          }
        }
      }
      else {
        const currentSharedStreams = sharedStream.map(stream => stream.url);
        if (currentSharedStreams.length > 0) {
          for (const aspect of input.aspects) {
            aspect.sharedStream = currentSharedStreams;
          }
        }
      }
    }

    if (ptzAspect) {
      input.aspects?.push({
        ...aspectToAspectInput(ptzAspect),
        enabled: ptzValues["enabled"],
        ptzTiltInverted: ptzValues["ptzTiltInverted"],
        ptzPanInverted: ptzValues["ptzPanInverted"]
      });
    }
    for (const aspect of vaeAspects) {
      input.aspects?.push(aspect);
    }

    if (deviceSets) {
      input["setId"] = deviceSets.map(set => set.id);
    }

    return input;
  }

  function validateMedia(): boolean {
    if (!mediaAspects.some(aspect => aspect.template.id === "video")) {
      setMediaError(__("A device must have a video stream"));
      return false;
    }
    if (!mediaAspects.some(aspect => aspect.template.id === "video" && aspect.enabled)) {
      setMediaError(__("At least one video stream must be selected"));
      return false;
    }
    setMediaError("");
    return true;
  }

  function onGISChange(name: string, value: any, values: FieldValues, form: AutoForm): void {
    const newValue = parseFloat(value);
    form.setValue(name, newValue);
    setPosition(prevPosition => ({...prevPosition, ...{ [name]: newValue} }));
  }

  function getSameDevice(): DeviceShort[] | undefined {
    if (!cameras || !cameras.devicesByAspectTypes || !formRef.current) {
      return undefined;
    }

    const values = formRef.current.getValues();
    const platformId = values["platformId"];
    if (!cameras.devicesByAspectTypes.some(dev => dev.platform?.id === platformId)) {
      return undefined;
    }

    const configType: DeviceBaseConfigType = values["configType"];
    let sameDevices: DeviceShort[] | undefined;
    switch (configType) {
      case DeviceBaseConfigType.Url:
        const url = values["url"];
        sameDevices = cameras.devicesByAspectTypes
          .filter(dev =>
            (dev.platform?.id === platformId && dev.config.connect.URL === url && dev.id !== device?.id));
        break;
      case DeviceBaseConfigType.Onvif:
        const host = values["host"];
        const port = values["httpPort"];
        sameDevices = cameras.devicesByAspectTypes
          .filter(dev =>
            (dev.platform?.id === platformId && dev.config.connect.host === host && port === dev.config.connect.port && dev.id !== device?.id));
        break;
    }

    if (sameDevices?.length === 0) {
      return undefined;
    }

    return sameDevices;
  }

  async function createCamera(device: DeviceInput): Promise<void> {
    setRefetching(true);
    try {
      const { data } = await createDevice({ variables: { device } });
      if (data) {
        data.createDevice.warning && Log.error(data.createDevice.warning);
        try {
          await updateCachedDevice(data.createDevice.id);
        }
        catch (e) {
          console.error("Updating cache error:", e);
        }
        finally {
          setRefetching(false);
          onBack?.([data.createDevice.id]);
        }
      }
      else {
        setRefetching(false);
      }
    }
    catch (e) {
      setRefetching(false);
    }
  }

  async function updateCamera(id: string, input: DeviceInput): Promise<void> {
    setRefetching(true);
    try {
      const { data } = await updateDevice({ variables: { id, device: input } });
      if (data) {
        data.updateDevice.warning && Log.error(data.updateDevice.warning);
        data.updateDevice.warningFields && updateWarningFields(data.updateDevice.warningFields);
        try {
          await updateCachedDevice(device?.id ?? "");
        }
        catch (e) {
          console.error("Updating cache error", e);
        }
        finally {
          const updatedDevice = getUpdatedDevice();
          updatedDevice && setUpdatedDevice(updatedDevice);
          setIsChanged(false);
          setRefetching(false);
        }
      }
      else {
        setRefetching(false);
      }
    }
    catch (e) {
      setRefetching(false);
    }
  }

  function updateWarningFields(warningFields?: WarningField[]) {
    if (!warningFields || warningFields.length === 0 || !device) {
      return;
    }

    const zoneUpdated = !warningFields.some(ef => ef.field === "zoneId");
    !zoneUpdated && formRef.current?.setValue("zoneId", device?.zone?.id);

    const exceptionSetIdField = warningFields.find(ef => ef.field === "setId");
    if (exceptionSetIdField) {
      const exceptionSetIds: string[] = JSON.parse(exceptionSetIdField.value || "");
      const newDeviceSets = getEntityWithoutExceptions<ObjectDescriptor[]>(exceptionSetIds, deviceSets, device.set || []);
      setDeviceSets(newDeviceSets);
    }
  }

  function onSharingStreamChange(streams: StreamShare[]) {
    setSharedStream(streams);
  }

  const getAudioAspectForVideoStream = useCallback((aspect: MediaAspect): MediaAspect | undefined => {
    if (isCameraUrl) {
      return audioAspects[0];
    }
    //@ts-ignore
    if (!aspect.onvifProfile) {
      return undefined;
    }
    //@ts-ignore
    return audioAspects.find(audio => audio.__typename === "DFA_Media" && audio.onvifProfile === aspect.onvifProfile);
  }, [mediaAspects, audioAspects]);

  /* eslint-disable react/jsx-indent */
  const panes = [
    {
      menuItem: __("Device Properties"),
      pane:
        <Tab.Pane key="properties" className="CreateUpdateCamera__tab_properties">
          {!!device && paramsChanged &&
          <Message warning content={__("Device parameters have been changed. Click 'Test' to be able to save the changes.")}/>}
          {!!error && <Message error content={error.message}/>}

          <AutoForm ref={formRef} schema={schema} values={values} onChange={onFormChange} onValidChange={() => updateIsValid()}>
            <div className="CreateUpdateCamera-FormLayout">
              <div className="CreateUpdateCamera-FormColumn1">
                <AutoLayout names={["destination", "platformId", "name", "state", "location", "zoneId"]}/>
              </div>
              <div className="CreateUpdateCamera-FormColumn2">
                <AutoLayout
                  names={["configDiscovery", "configType", "templateButton", "host", "httpPort", "rtspPort", "url", "user", "pass"]}/>
              </div>
            </div>
          </AutoForm>

          { device &&
            <>
              <Header as="h4" />
                <Table compact>
                  <Table.Body>
                    <Table.Row>
                      <Table.Cell width={4}>{__("Camera Log")}</Table.Cell>
                      <Table.Cell collapsing>
                        <Modal
                          onClose={() => setIsCameraLogOpen(false)}
                          onOpen={() => setIsCameraLogOpen(true)}
                          open={isCameraLogOpen}
                          trigger={
                            <IconButton icon="file alternate"/>
                          }
                        >
                        <Header>{__("Camera log")}</Header>
                        <CameraLogs deviceId={device.id} />
                        <Modal.Actions>
                          <Button onClick={() => setIsCameraLogOpen(false)}>
                            <Icon name="check"/>{__("Close")}
                          </Button>
                        </Modal.Actions>
                        </Modal>
                      </Table.Cell>
                      <Table.Cell width={12}/>
                    </Table.Row>
                    <Table.Row>
                      <Table.Cell>{__("Uptime")}</Table.Cell>
                      <Table.Cell collapsing/>
                      <Table.Cell>
                        {getCameraUptimeText(mediaAspectWithStats?.uptime)}
                      </Table.Cell>
                    </Table.Row>
                    <Table.Row>
                        <Table.Cell>{__("Average FPS")}</Table.Cell>
                        <Table.Cell collapsing/>
                        <Table.Cell>
                          {mediaAspectWithStats?.averageFPS || "N/A"}
                        </Table.Cell>
                    </Table.Row>
                    <Table.Row>
                        <Table.Cell>{__("Average GOP")}</Table.Cell>
                        <Table.Cell collapsing/>
                        <Table.Cell>
                          {mediaAspectWithStats?.averageGOP || "N/A"}
                        </Table.Cell>
                    </Table.Row>
                    <Table.Row>
                        <Table.Cell>{__("Average Inter-arrival Jitter, ms")}</Table.Cell>
                        <Table.Cell collapsing/>
                        <Table.Cell>
                          {mediaAspectWithStats?.averageInterArrivalJitter || "N/A"}
                        </Table.Cell>
                    </Table.Row>
                    <Table.Row>
                        <Table.Cell>{__("RTP Packet Lost")}</Table.Cell>
                        <Table.Cell collapsing/>
                        <Table.Cell>
                          {mediaAspectWithStats?.rtpPacketLost || "N/A"}
                        </Table.Cell>
                    </Table.Row>
                    <Table.Row>
                        <Table.Cell>{__("Video codec")}</Table.Cell>
                        <Table.Cell collapsing/>
                        <Table.Cell>
                          {getVideoCodec(mediaAspectWithStats)}
                        </Table.Cell>
                    </Table.Row>
                    <Table.Row>
                        <Table.Cell>{__("Audio codec")}</Table.Cell>
                        <Table.Cell collapsing/>
                        <Table.Cell>
                          {mediaAspectWithStats?.audioCodec || "N/A"}
                        </Table.Cell>
                    </Table.Row>
                    {isDemoDevice &&
                    <Table.Row>
                    <Table.Cell>{__("Capture Frame Rate, FPS")}</Table.Cell>
                      <Table.Cell collapsing/>
                      <Table.Cell>
                      <Dropdown
                        selection
                        placeholder={__("Choose...")}
                        options={captureFrameRateOptions}
                        value={captureFrameRate}
                        onChange={(e, { value }) => {
                          typeof value === "number" && onFrameRateChange(value);
                        }}
                    />
                      </Table.Cell>
                    </Table.Row>}
                  </Table.Body>
                </Table>
            </>
          }
        </Tab.Pane>
    },
    {
      menuItem: __("Media"),
      pane: (
        <Tab.Pane key="media" className="CreateUpdateCamera__tab_media">
          {!!mediaError && <Message error content={mediaError}/>}

          <div className="CreateUpdateCamera-Media">
            {!probeData && !!device &&
            <div className="CreateUpdateCamera-Snapshot">
              {activeTab === 1 &&
              <StreamSnapshot
                enabled={avatarOnline}
                deviceId={device.id}
                url={mediaAspectUrl}
                load={!!device && !probeData && activeTab === 1}
                platformId={device.platform?.id}
              />}
            </div>}

            <div className="CreateUpdateCamera-MediaAspects">
              <Table celled compact>
                <Table.Header>
                  <Table.Row>
                    <Table.HeaderCell/>
                    {mediaAspects.some(aspect => !!aspect.snapshot) &&
                    <Table.HeaderCell>{__("Image")}</Table.HeaderCell>}
                    <Table.HeaderCell>{__("Media")}</Table.HeaderCell>
                    <Table.HeaderCell>{__("Edge Archive")}</Table.HeaderCell>
                    <Table.HeaderCell>{__("Cloud Archive")}</Table.HeaderCell>
                  </Table.Row>
                </Table.Header>

                <Table.Body>
                  {mediaAspects.map((aspect) => {
                    if (aspect.__typename !== "DFA_Media" || aspect.type !== DeviceFunctionalAspectType.Media || aspect.streamType === MediaStreamType.Audio) {
                      return null;
                    }
                    return (
                      <Table.Row key={aspect.id} className={classNames({"DisabledMediaAspect": isCloudDirect && aspect.codec === "hevc"})}>
                        <Table.Cell collapsing>
                          <Radio
                            checked={aspect.enabled}
                            disabled={!isSupportedDevice || (isCloudDirect && aspect.codec === "hevc")}
                            onChange={(e, data) => changeMediaAspect(aspect.id, aspect => {
                              aspect.enabled = !!data.checked;
                            })}
                          />
                        </Table.Cell>
                        {mediaAspects.some(aspect => !!aspect.snapshot) &&
                        <Table.Cell collapsing>
                          {!!aspect.snapshot &&
                          <Popup
                            trigger={
                              <div className="CreateUpdateCamera-CellSnapshot">
                                <img
                                  src={`data:image;base64, ${aspect.snapshot}`}
                                  alt=""
                                />
                              </div>}>
                            <img
                              src={`data:image;base64, ${aspect.snapshot}`}
                              alt=""
                            />
                          </Popup>}
                        </Table.Cell>}
                        <Table.Cell>
                          <div className="CreateUpdateCamera-MediaAspectName">
                            <Checkbox
                              className="CreateUpdateCamera-VideoAspectCheckbox"
                              label={`${__("Video")}: ${aspect.name}`}
                              checked={aspect.enabled}
                              />
                          </div>
                          {audioAspects.length > 0 && getAudioAspectForVideoStream(aspect) &&
                            <div className="CreateUpdateCamera-AudioAspect">
                              <Checkbox
                                className="CreateUpdateCamera-AudioAspectCheckbox"
                                label={`${__("Audio")}: ${getAudioAspectForVideoStream(aspect)?.name || ""}`}
                                checked={getAudioAspectForVideoStream(aspect)?.enabled}
                                disabled={!aspect.enabled}
                                onChange={(e, data) => changeMediaAspect(getAudioAspectForVideoStream(aspect)?.id || "", aspect => {
                                  aspect.enabled = !!data.checked;
                                })}
                              />
                            </div>
                          }
                        </Table.Cell>
                        <Table.Cell className="CreateUpdateCamera__cell_edgeArchive">
                          {aspect.streamType === MediaStreamType.Video && aspect.enabled && !isCloudDirect ?
                            <Dropdown
                              options={edgeArchiveOptions}
                              value={aspect.edgeArchive}
                              disabled={!isSupportedDevice || (isCloudDirect && aspect.codec === "hevc")}
                              onChange={(e, {value}) => changeMediaAspect(aspect.id, aspect => {
                                if (aspect.__typename === "DFA_Media" && value) {
                                  aspect.edgeArchive = value as MediaArchiveOption;
                                  switch (aspect.edgeArchive) {
                                    case MediaArchiveOption.None:
                                      aspect.cloudArchive = MediaArchiveOption.None;
                                      break;
                                    case MediaArchiveOption.Events:
                                    case MediaArchiveOption.Schedule:
                                      if (aspect.cloudArchive !== MediaArchiveOption.None) {
                                        aspect.cloudArchive = aspect.edgeArchive;
                                      }
                                      break;
                                  }
                                }
                              })}
                            /> :
                            __("N/A")}
                        </Table.Cell>
                        <Table.Cell className="CreateUpdateCamera__cell_cloudArchive">
                          {aspect.streamType === MediaStreamType.Video && aspect.enabled ?
                            <Dropdown
                              options={cloudArchiveOptions}
                              value={aspect.cloudArchive}
                              disabled={!isSupportedDevice || (isCloudDirect && aspect.codec === "hevc")}
                              onChange={(e, {value}) => changeMediaAspect(aspect.id, aspect => {
                                if (aspect.__typename === "DFA_Media" && value) {
                                  aspect.cloudArchive = value as MediaArchiveOption;
                                  switch (aspect.cloudArchive) {
                                    case MediaArchiveOption.Continuous:
                                      aspect.edgeArchive = MediaArchiveOption.Continuous;
                                      break;
                                    case MediaArchiveOption.Events:
                                    case MediaArchiveOption.Schedule:
                                      if (aspect.edgeArchive !== MediaArchiveOption.Continuous) {
                                        aspect.edgeArchive = aspect.cloudArchive;
                                      }
                                      break;
                                  }
                                }
                              })}
                            /> :
                            __("N/A")}
                        </Table.Cell>
                      </Table.Row>
                    );
                  })}

                  {mediaAspects.length === 0 &&
                  <Table.Row>
                    <Table.Cell colSpan={5} textAlign="center">
                      {__("Media is not discovered")}
                    </Table.Cell>
                  </Table.Row>}
                </Table.Body>
              </Table>
            </div>

              <>
                <Header as="h4" />
                <Table compact>
                  <Table.Body>
                  <Table.Row>
                  <Table.Cell width={4}>{__("Force RTP over TCP")}</Table.Cell>
                    <Table.Cell collapsing/>
                    <Table.Cell width={12}>
                      <Checkbox
                        toggle
                        checked={rtpOverTcp}
                        className="CheckboxField__rtpOverTcp"
                        onChange={(e, { checked }) => onRtpValuesChanged({ property: "rtpOverTcp", value: !!checked })}
                      />
                    </Table.Cell>
                  </Table.Row>
                <Table.Row>
                  <Table.Cell>{__("Follow RTP time")}</Table.Cell>
                  <Table.Cell collapsing/>
                  <Table.Cell>
                    <Checkbox
                      toggle
                      checked={rtpTime}
                      className="CheckboxField__rtpTime"
                      onChange={(e, { checked }) => onRtpValuesChanged({ property: "rtpTime", value: !!checked })}
                    />
                  </Table.Cell>
                </Table.Row>
                {device &&
                <>
                  <Table.Row>
                      <Table.Cell>{__("Consumed Space, MB")}</Table.Cell>
                      <Table.Cell collapsing/>
                      <Table.Cell>
                        {mediaAspectWithStats?.consumedSpace || "N/A"}
                      </Table.Cell>
                  </Table.Row>
                    <Table.Row>
                      <Table.Cell>{__("Space Usage, MB/day")}</Table.Cell>
                      <Table.Cell collapsing/>
                      <Table.Cell>
                        {mediaAspectWithStats?.spaceUsageDay || "N/A"}
                      </Table.Cell>
                    </Table.Row>
                    <Table.Row>
                      <Table.Cell>{__("Space Usage, MB/hr")}</Table.Cell>
                      <Table.Cell collapsing/>
                      <Table.Cell>
                        {mediaAspectWithStats?.spaceUsageHour || "N/A"}
                      </Table.Cell>
                    </Table.Row>
                </> }
                  </Table.Body>
                </Table>
              </>

          </div>
        </Tab.Pane>
      )
    },
    {
      menuItem: !device ? null : __("Sharing"),
      pane: (
        <Tab.Pane key="sharing" className="CreateUpdateCamera__tab_sharing">
          { device &&
            <SharingStream
              deviceId={device.id}
              sharedStream={sharedStream}
              onChange={onSharingStreamChange}
              autoUpdate={activeTab === 2}
            /> }
        </Tab.Pane>
      )
    },
    {
      menuItem: __("Storage"),
      pane: (
        <Tab.Pane key="storage" className="CreateUpdateCamera__tab_storage">
          <AutoForm
            ref={storageFormRef}
            schema={storageSchema}
            values={storageValues}
            onChange={() => updateIsChanged()}
            onValidChange={() => updateIsValid()}
          >
            <AutoLayout/>
          </AutoForm>
        </Tab.Pane>
      )
    },
    {
      menuItem: __("PTZ"),
      pane: (
        <Tab.Pane key="ptz" className="CreateUpdateCamera__tab_ptz">
          <AutoForm
            ref={ptzFormRef}
            schema={ptzSchema}
            values={ptzValues}
            onChange={() => updateIsChanged()}
            onValidChange={() => updateIsValid()}
          >
            <AutoLayout/>
          </AutoForm>
        </Tab.Pane>
      )
    },
    {
      menuItem: !device ? null : __("Audit"),
      pane: (
        <Tab.Pane key="audit" className="CreateUpdateCamera__tab_audit">
          {device && (activeTab === 5 || auditTabWasActive) &&
          <div className="audit">
            <div className="event-control">
              <div className="event-list-control">
                <Header size="tiny">{__("Events: {{name}}", {name: device.name})}</Header>
                <Button
                  className="export-audit"
                  size="mini"
                  onClick={() => setGetCsvFile(true)}>
                  <Icon name="download"/>
                  {__("Export CSV")}
                </Button>
              </div>
              <div className="event-info-control">
                <Header size="tiny">{__("Event details")}</Header>
              </div>
            </div>
            <div className="audit-content">
              <div className="event-list">
                <div className="events">
                  <EventList witnessesList={[device.id]} exportCsvFile={getCsvFile}/>
                </div>
              </div>
              <div className="event-info">
                <div className="info">
                  <EventInfoComponent isCell/>
                </div>
              </div>
            </div>
          </div>}
        </Tab.Pane>
      )
    },
    {
      menuItem: !accessConfig?.limitedGISAccess ? __("GIS") : null,
      pane: (
        <Tab.Pane key="geoMap" className="CreateUpdateCamera__tab_geoMap">
          {(activeTab === 6 || gisTabWasActive) &&
          <>
            <div className="geoMap-inputs">
              <AutoForm
                ref={positionFormRef}
                className="CreateUpdateCamera-TemplateForm"
                schema={positionSchema}
                values={positionValues}
                onChange={onGISChange}>
                  <div className="CreateUpdateCamera-FormLayout">
                    <div className="CreateUpdateCamera-FormColumn1">
                      <AutoLayout names={["lat"]}/>
                    </div>
                    <div className="CreateUpdateCamera-FormColumn2">
                      <AutoLayout names={["lng"]}/>
                    </div>
                  </div>
              </AutoForm>
            </div>
            <GEOMap id={device?.id} setPosition={setNewPosition} position={position ?? undefined} isView={false}/>
          </> }
        </Tab.Pane>
      )
    },
    {
      menuItem: !accessConfig?.limitedSetsAccess ? __("Sets") : null,
      pane: (
        <Tab.Pane key="sets" className="CreateUpdateCamera__tab_sets">
          <ControlSetPanels deviceSets={deviceSets} setDeviceSets={setDeviceSets} />
        </Tab.Pane>
      )
    },
    {
      menuItem: accessConfig?.limitedReportsAccess ? null : device ? __("Report") : null,
      pane: (
        <Tab.Pane className="CameraReport" key="CameraReport">
          {device && <Report deviceId={device.id} reportType={Type.camera}/>}
        </Tab.Pane>
      )
    }
  ];
  /* eslint-enable react/jsx-indent */

  return (
    <Segment className="CreateUpdateCamera">
      <WithQueryStatus
        error={avatarError || zoneError || storageError || devTemplError}
        loading={avatarLoading || zoneLoading || storageLoading || devTemplLoading}
      >
        {probeLoading && <Loading/>}
        {(createLoading || updateLoading || refetching) && <Loading text={__("Updating...")}/>}

        <div className="CreateUpdateCamera-TopButtons">
          <Button onClick={() => cancel()}>
            <Icon name="arrow left"/>{__("Back")}
          </Button>
          {!device && activeTab > 0 &&
          <>
            <Button disabled={activeTab <= 0} onClick={() => onActiveTabChange(activeTab === 3 ? activeTab - 2 : activeTab - 1)}>
              <Icon name="arrow alternate circle left"/>{__("Back")}
            </Button>
            {activeTab < 4 &&
            <Button className="next-btn" disabled={!!mediaError} positive onClick={() => onActiveTabChange(activeTab === 1 ? activeTab + 2 : activeTab + 1)}>
              <Icon name="arrow alternate circle right"/>{__("Next")}
            </Button>}
          </>}
          {activeTab === 0 && !isDiscoveryDevice && isSupportedDevice &&
          <Popup
            trigger={
              <Button className="play-test-btn" positive onClick={() => onProbe()}>
                <Icon name="play"/>{__("Probe")}
              </Button>}
            content={__("Camera configuration was changed and should be retested.")}
            position="right center"
            open={!device && !!probeData && paramsChanged}
                />}
          {(!!device || activeTab >= 4) &&
          <Button
            positive
            className="final-create-btn"
            disabled={(!isChanged && !!device) || paramsChanged || !isChanged || !isValid}
            onClick={() => onCreateUpdate()}
          >
            <Icon name={!device ? "plus" : "check"}/>{!device ? __("Add") : __("Save")}
          </Button>}
        </div>

        {!!createError && <Message error content={createError.message} />}
        {!!updateError && <Message error content={updateError.message} />}

        <Tab
          panes={panes}
          className="CreateUpdateCamera-Tab"
          renderActiveOnly={false}
          activeIndex={activeTab}
          onTabChange={(e, {activeIndex}) => onActiveTabChange(activeIndex)}
        />

        <SameDevicesModal
          deviceType="camera"
          names={sameCamerasName || []}
          open={cameraExist}
          okButtonTittle={!device ? __("Add anyway") : __("Save anyway")}
          onCancel={cancelCreating}
          onClose={() => setCameraExist(false)}
          onOk={() => createUpdateAnyway()}
        />

        <Modal open={templateFormOpen} size="tiny" onClose={onTemplateFormClose}>
          <Header>{__("Get URL From")}</Header>
          <Modal.Content>
            <AutoForm
              ref={templateFormRef}
              className="CreateUpdateCamera-TemplateForm"
              schema={templateSchema}
              values={templateValues}
              onChange={onTemplateFormChange}>
              <AutoLayout/>
            </AutoForm>
          </Modal.Content>
          <Modal.Actions>
            <Button positive onClick={onTemplateFormClose}>
              <Icon name="check"/>{__("OK")}
            </Button>
            <Button negative onClick={() => setTemplateFormOpen(false)}>
              <Icon name="cancel"/>{__("Cancel")}
            </Button>
          </Modal.Actions>
        </Modal>
      </WithQueryStatus>
    </Segment>
  );
};

export default CreateUpdateCamera;

function getCameraUptimeText(upTime?: number | null): string {
  if (!upTime) {
    return __("N/A");
  }
  let time = upTime;
  const days = Math.trunc(time / (24 * 60 * 60));
  time %= 24 * 60 * 60;
  const hours = Math.trunc(time / (60 * 60));
  time %= 60 * 60;
  const minutes = Math.trunc(time / 60);
  const seconds = time % 60;
  let text = `${hours.toString().padStart(2, "0")}:${minutes.toString().padStart(2, "0")}:${seconds.toString().padStart(2, "0")}`;
  if (days) {
    text = `${days} day(s), ` + text;
  }
  return text;
}

function getVideoCodec(aspect: Dfa_Media | undefined): string {
  if (!aspect?.videoCodec) {
    return "N/A";
  }
  return `${aspect.videoCodec}${aspect.videoSize ? "/" + aspect.videoSize : ""}`;
}

function isDfaMedia(aspect: Device["aspects"][0]): aspect is Dfa_Media {
  return aspect.__typename === "DFA_Media" && aspect.type === DeviceFunctionalAspectType.Media;
}
