import React, { useEffect, useState, useRef, useMemo } from "react";
import { useNavigate, useParams, useLocation } from "react-router-dom";
import { Segment, Ref, Icon, Popup, Button, Modal, Header, Portal } from "semantic-ui-react";
import { Table, Column, AutoSizer } from "react-virtualized";
import { FontAwesomeIcon } from "@fortawesome/react-fontawesome";
import {
  useDevicesByAspectTypesShortQuery,
  DeviceFunctionalAspectType,
  useLabelsQuery,
  Label,
  useDeleteDeviceMutation,
  Dfa_Avatar,
} from "@generated/graphql";
import WithQueryStatus from "components/WithQueryStatus";
import ListFilter from "components/ListFilter";
import CreateUpdateCameraWrapper from "./CreateUpdateCamera/CreateUpdateCameraWrapper";
import ImportCameraList from "./ImportCameraList";
import VaeManagementWrapper from "./VaeManagement/VaeManagementWrapper";
import CameraDiscovery from "./CameraDiscovery";
import { WidgetProps } from "components/Widgets";
import Loading from "components/Loading";
import Snapshot from "components/Snapshot";
import {useDeviceActions, Device, healthStatusText, DeviceShort} from "@core/actions";
import {Log} from "@solid/libs/log";
import {__} from "@solid/libs/i18n";
import {UUID} from "@solid/types";
import { useAccessability } from "@core/store/actions/accessability";
import { RouteParams } from "@core/types";
import Help from "components/Help";
import markdown from "./help.md";

import "./style.css";

export enum DeviceDialog {
  Create,
  Update,
  Vae,
  Discovery
}

enum AdminCameraListRoute {
  Create = "/view/camera/add",
  Discovery = "/view/camera/discovery",
  Import = "/view/camera/import",
  Update = "/view/camera/edit",
  Vae = "/view/camera/vae"
}

type AdminCameraListProps = WidgetProps & {
  link?: Device;
  showFilter?: boolean,
  onDialogOpen?: (dialog: DeviceDialog) => void;
  onDialogClose?: () => void;
};

const AdminCameraList = ({ cellProps, setCellProps, link, showFilter, onDialogOpen, onDialogClose }: AdminCameraListProps) => {
  const { config: accessConfig } = useAccessability();
  const navigate = useNavigate();
  const { pathname } = useLocation();
  const { viewId = "", cameraId = "", linkId = ""} = useParams<RouteParams>();
  const { loading, error, data } = useDevicesByAspectTypesShortQuery({ variables: { types: [{ type: DeviceFunctionalAspectType.Media }] } });
  const { data: labelData, error: labelError, loading: labelLoading } = useLabelsQuery();
  const [deleteDevice, { data: deleteData, error: deleteError, loading: deleteLoading }] = useDeleteDeviceMutation();
  const { deleteCachedDevice } = useDeviceActions({ skipSubscription: true });
  const [filterVisible, setFilterVisible] = useState(false);
  const [devices, setDevices] = useState<DeviceShort[]>([]);
  const [searchText, setSearchText] = useState("");
  const [labels, setLabels] = useState<Label[]>([]);
  const [importOpen, setImportOpen] = useState(false);
  const [selectedId, setSelectedId] = useState<UUID>("");
  const selectedDevice = selectedId ? devices.find(device => device.id === selectedId) : undefined;
  const [createUpdateOpen, setCreateUpdateOpen] = useState(false);
  const [deleteConfirmOpen, setDeleteConfirmOpen] = useState(false);
  const [vaeOpen, setVaeOpen] = useState(false);
  const [discoveryOpen, setDiscoveryOpen] = useState(false);
  const rootRef = useRef<HTMLDivElement>(null);
  const [createdDeviceIdSet, setCreatedDeviceIdSet] = useState(new Set<UUID>());
  const [snapshotId, setSnapshotId] = useState("");
  const snapshotPopupRef = useRef<HTMLDivElement>(null);
  const rowClickCoord = useRef<{ x: number, y: number }>({ x: 0, y: 0 });
  const imageRef = useRef<HTMLImageElement>(null);
  const isCameraExist = useMemo(() => { return data?.devicesByAspectTypes.some(device => device.id === cameraId); }, [cameraId, data]);
  const editModePath = useMemo(() => { return pathname.split(`/${cameraId}`)[0]; }, [pathname]);

  const mountedRef = useRef<boolean>(false);

  useEffect(() => {
    mountedRef.current = true;

    return () => {
      mountedRef.current = false;
    };
  }, []);

  useEffect(() => {
    if (editModePath && cameraId) {
      switch (editModePath) {
        case AdminCameraListRoute.Update:
          isCameraExist && onEditClick(cameraId);
          break;
        case AdminCameraListRoute.Vae:
          isCameraExist && onVaeClick(cameraId);
          break;
        default:
          break;
      }

      return;
    }

    switch (pathname) {
      case AdminCameraListRoute.Create:
        onCreateClick();
        break;
      case AdminCameraListRoute.Discovery:
        onDiscoveryClick();
        break;
      case AdminCameraListRoute.Import:
        onImportClick();
        break;
      default:
        break;
    }
  }, [pathname, isCameraExist, editModePath, cameraId]);

  useEffect(() => {
    if ((viewId || linkId) && !deleteConfirmOpen) {
      createUpdateOpen && setCreateUpdateOpen(false);
      importOpen && setImportOpen(false);
      discoveryOpen && setDiscoveryOpen(false);
      vaeOpen && setVaeOpen(false);
    }
  }, [viewId, linkId]);

  useEffect(() => {
    if (data && data.devicesByAspectTypes) {
      const cameraList = getDevices(searchText, labels);
      setDevices(cameraList);
    }
  }, [data, searchText, labels, createdDeviceIdSet]);

  useEffect(() => {
    if (deleteData?.deleteDevice) {
      deleteCachedDevice(selectedId);
      setSelectedId("");
    }
  }, [deleteData]);

  useEffect(() => {
    if (deleteError) {
      Log.error(`Could not delete device: ${deleteError.message}`);
    }
  }, [deleteError]);

  useEffect(() => {
    if (!setCellProps) {
      return;
    }

    if (createUpdateOpen) {
      const title = selectedDevice ? __("Configure Device") : __("Create Device");
      setCellProps({ title });
      return;
    }

    if (vaeOpen) {
      setCellProps({ title: __("Device Analytics Management") });
      return;
    }

    if (discoveryOpen) {
      setCellProps({ title: __("Devices Discovery") });
      return;
    }

    const title =
      <div className="AdminCameraList-Title">
        <div className="AdminCameraList-TitleName">{cellProps?.title}</div>

        <Popup
          trigger={
            <Icon
              name="search"
              className="AdminCameraList-TitleIcon"
              onClick={() => setFilterVisible(visible => !visible)}
            />}
          content={filterVisible ? __("Hide Filter") : __("Show Filter")}
          position="bottom right"
        />
      </div>;

    setCellProps({ title });
  }, [filterVisible, createUpdateOpen, vaeOpen, discoveryOpen, selectedDevice, setCellProps]);

  useEffect(() => {
    if (imageRef.current) {
      positionSnapshotPopup();
    }
  }, [imageRef]);

  function getCameraState(state: boolean): string {
    return state ? __("On") : __("Off");
  }

  function getDevices(searchText?: string, labels: Label[] = []): DeviceShort[] {
    const text = searchText?.toLocaleUpperCase();
    const list: DeviceShort[] = data
      ?.devicesByAspectTypes
      ?.filter(cam =>
        (!text
          || cam.name.toLocaleUpperCase().includes(text)
          || cam.location?.toLocaleUpperCase().includes(text)
          || (!accessConfig?.limitedZonesAccess && cam.zone?.name?.toLocaleUpperCase().includes(text))
          || cam.healthStatus?.toLocaleUpperCase().includes(text)
          || getCameraState(cam.enabled).toLocaleUpperCase().includes(text)
          || cam.platform?.name?.toLocaleUpperCase().includes(text))
        && (
          labels.length === 0
          || labels.some(label => label.objects.some(object => object.objectId === cam.id))
        )
        && (
          !link || cam.platform?.id === link.id
        ))
      .sort((a, b) => {
        if ((createdDeviceIdSet.has(a.id) && createdDeviceIdSet.has(b.id)) || (!createdDeviceIdSet.has(a.id) && !createdDeviceIdSet.has(b.id)))
        {
          return a.name.localeCompare(b.name, undefined, { sensitivity: "base" });
        }
        return createdDeviceIdSet.has(a.id) ? -1 : 1;
      })
      ?? [];
    return list;
  }

  /* eslint-disable react/no-unused-prop-types */
  function getActionsContent({ rowData }: { rowData: DeviceShort }) {
    return (
      <>
        <Popup
          trigger={
            <Icon name="edit" className="AdminCameraList-IconButton" onClick={(e: React.MouseEvent) => {
              e.stopPropagation();
              onEditClick(rowData.id);
            }}/>
          }
          content={__("Configure")}
        />
        {!accessConfig?.limitedVideoAnalyticsAccess &&
          <Popup
            trigger={
              <Icon name="lightbulb" className="AdminCameraList-IconButton" onClick={(e: React.MouseEvent) => {
                e.stopPropagation();
                onVaeClick(rowData.id);
              }}/>
            }
            content={__("Analytics")}
          />
        }
        <Popup
          trigger={
            <Icon name="trash alternate" className="AdminCameraList-IconButton" onClick={(e: React.MouseEvent) => {
              e.stopPropagation();
              onDeleteClick(rowData.id);
            }}/>
          }
          content={__("Delete")}
        />
      </>
    );
  }
  /* eslint-enable react/no-unused-prop-types */

  function onCreateClick(): void {
    setSelectedId("");
    setCreateUpdateOpen(true);
    onDialogOpen && onDialogOpen(DeviceDialog.Create);
    !pathname.includes(AdminCameraListRoute.Create) && navigate(AdminCameraListRoute.Create, { state: { linkId: link?.id } });
  }

  function onEditClick(deviceId: UUID): void {
    setSelectedId(deviceId);
    setCreateUpdateOpen(true);
    onDialogOpen && onDialogOpen(DeviceDialog.Update);
    !pathname.includes(`${AdminCameraListRoute.Update}/${deviceId}`) && navigate(`${AdminCameraListRoute.Update}/${deviceId}`);
  }

  function onVaeClick(deviceId: UUID): void {
    setSelectedId(deviceId);
    setVaeOpen(true);
    onDialogOpen && onDialogOpen(DeviceDialog.Vae);
    !pathname.includes(`${AdminCameraListRoute.Vae}/${deviceId}`) && navigate(`${AdminCameraListRoute.Vae}/${deviceId}`);
  }

  function onDeleteClick(deviceId: UUID): void {
    setSelectedId(deviceId);
    setDeleteConfirmOpen(true);
  }

  function onDiscoveryClick(): void {
    setDiscoveryOpen(true);
    onDialogOpen && onDialogOpen(DeviceDialog.Discovery);
    !pathname.includes(AdminCameraListRoute.Discovery) && navigate(AdminCameraListRoute.Discovery);
  }

  function onDelete(): void {
    setDeleteConfirmOpen(false);
    if (!selectedDevice) {
      return;
    }
    deleteDevice({ variables: { id: selectedDevice.id } });
  }

  function onRowClick({ event, rowData }: { event: React.MouseEvent, index: number, rowData: DeviceShort }): void {
    if (event && rowData) {
      setSnapshotId(rowData.id);
      rowClickCoord.current = { x: event.pageX, y: event.pageY };
    }
  }

  const onImportClick = () => {
    !pathname.includes(AdminCameraListRoute.Import) && navigate(AdminCameraListRoute.Import);
    setImportOpen(true);
  };

  function positionSnapshotPopup(): void {
    const popup = snapshotPopupRef.current;
    if (!popup) {
      return;
    }
    popup.setAttribute("style", "left: 0; top: 0");
    const rootElem = document.documentElement;
    const popupRect = popup.getBoundingClientRect();
    let { x, y } = rowClickCoord.current;
    const dX = x + popupRect.width - rootElem.clientWidth;
    const dY = y + popupRect.height - rootElem.clientHeight;
    if (dX > 0) {
      x -= dX;
    }
    if (dY > 0) {
      y -= dY;
    }
    popup.setAttribute("style", `left: ${x}px; top: ${y}px`);
  }

  return !createUpdateOpen && !vaeOpen && !discoveryOpen && !importOpen ?
    <Ref innerRef={rootRef}>
      <Segment className="AdminCameraList">
        <WithQueryStatus
          loading={loading || labelLoading}
          error={error || labelError}>

          {filterVisible &&
          <ListFilter
            filterTextPlaceholder={__("Filter by text")}
            labelData={labelData}
            searchText={searchText}
            labels={labels}
            nameFilter
            labelsFilter
            allowLabelHierarchyEdit
            rootRef={rootRef}
            onSearchTextChange={text => setSearchText(text)}
            onLabelsChange={labels => setLabels(labels)}
          />}

          <div className="AdminCameraList-Header">
            <Button className="create-device" icon="plus" content={__("Add")} onClick={onCreateClick}/>
            {(!link || (link.aspects.length > 0 && (link.aspects[0] as Dfa_Avatar).isLink === true)) &&
              <Button icon="globe" content={__("Discover")} onClick={onDiscoveryClick}/>}
            {!accessConfig?.limitedImportCamerasAccess &&
              <Button
                className="import-devices"
                onClick={() => onImportClick()}>
                <FontAwesomeIcon className="FileImport-Icon" icon="file-import"/>
                {__("Import")}
              </Button>}
          </div>

          {showFilter &&
          <ListFilter
            labelData={labelData}
            searchText={searchText}
            labels={labels}
            nameFilter
            labelsFilter
            allowLabelHierarchyEdit
            rootRef={rootRef}
            onSearchTextChange={text => setSearchText(text)}
            onLabelsChange={labels => setLabels(labels)}
          />}

          <div className="AdminCameraList-Grid">
            <AutoSizer style={{ width: "100%", height: "100%" }}>
              {({ width, height }) => (
                <Table
                  className="AdminCameraList-Table"
                  width={width}
                  height={height}
                  rowHeight={34}
                  headerHeight={58}
                  rowCount={devices.length}
                  rowGetter={({ index }) => devices[index]}
                  rowClassName={({ index }) => devices[index] && createdDeviceIdSet.has(devices[index].id) ? "added" : ""}
                  onRowClick={onRowClick}>
                  <Column
                    dataKey="actions"
                    label=""
                    width={90}
                    flexShrink={0}
                    cellRenderer={getActionsContent}
                  />
                  <Column
                    dataKey="name"
                    label={__("Name")}
                    width={100}
                    flexGrow={1}
                  />
                  <Column
                    dataKey="location"
                    label={__("Location")}
                    width={100}
                    flexGrow={1}
                  />
                  {!accessConfig?.limitedZonesAccess &&
                  <Column
                    dataKey="zone"
                    label={__("Zone")}
                    width={100}
                    flexGrow={1}
                    cellDataGetter={({ rowData }: { rowData: DeviceShort }) => rowData.zone?.name ?? ""}
                  />}
                  <Column
                    dataKey="enabled"
                    label={__("State")}
                    width={60}
                    cellDataGetter={({ rowData }: { rowData: DeviceShort }) => getCameraState(rowData.enabled)}
                  />
                  <Column
                    dataKey="healthStatus"
                    label={__("Status")}
                    width={80}
                    cellDataGetter={({ rowData }: { rowData: DeviceShort }) => healthStatusText[rowData.healthStatus]}
                  />
                  <Column
                    dataKey="platform"
                    label={__("Avatar")}
                    width={100}
                    flexGrow={1}
                    cellDataGetter={({ rowData }: { rowData: DeviceShort }) => rowData.platform?.name ?? ""}
                  />
                </Table>
              )}
            </AutoSizer>
          </div>

          <Modal open={deleteConfirmOpen} onClose={() => setDeleteConfirmOpen(false)}>
            <Header>{__("Delete Device")}</Header>
            <Modal.Content>{__("Are you sure you want to delete device '{{name}}'?", {name: selectedDevice?.name})}</Modal.Content>
            <Modal.Actions>
              <Button negative onClick={onDelete}>
                <Icon name="trash"/>{__("Delete")}
              </Button>
              <Button onClick={() => setDeleteConfirmOpen(false)}>
                <Icon name="cancel"/>{__("Cancel")}
              </Button>
            </Modal.Actions>
          </Modal>

          <Portal
            open={!!snapshotId}
            onMount={() => snapshotId && positionSnapshotPopup()}
            onClose={() => setSnapshotId("")}>
            <Ref innerRef={snapshotPopupRef}>
              <Segment className="AdminCameraList-Snapshot">
                <Snapshot snapshotId={snapshotId} load={!!snapshotId} />
                <Icon name="close" onClick={() => setSnapshotId("")}/>
              </Segment>
            </Ref>
          </Portal>

          {deleteLoading && <Loading text={__("Updating...")}/>}

          <Help markdown={markdown}/>
        </WithQueryStatus>
      </Segment>
    </Ref> :
    (importOpen ?
      <ImportCameraList/> :
      createUpdateOpen ?
        <CreateUpdateCameraWrapper cameras={data} deviceId={selectedId} onBack={createdIds => {
          if (!mountedRef.current) return;
          navigate(-1);
          !!createdIds && setCreatedDeviceIdSet(new Set<UUID>(createdIds));
          onDialogClose && onDialogClose();
        }}/> :
        (vaeOpen ?
          (selectedDevice ?
            <VaeManagementWrapper deviceId={selectedId} onBack={() => {
              navigate(-1);
              onDialogClose && onDialogClose();
            }}/> :
            null) :
            <CameraDiscovery linkId={link?.id} onBack={createdIds => {
              navigate(-1);
              !!createdIds && setCreatedDeviceIdSet(new Set<UUID>(createdIds));
              onDialogClose && onDialogClose();
            }}/>));
};

export default AdminCameraList;
