import child_process from "child_process";
import path from "path";
import { Context } from "@libs/fuse-box/core/context";
import { FuseLog } from "fuse-box-typechecker/dist/fuseLogger/fuseLog";
import Inspector, { InspectorOptions } from "./inspector";
import { WorkerCommand } from "./worker";

type PluginOptions = {
  name: string;
} & Partial<InspectorOptions>;

const logger = new FuseLog();

class ESLintHelper {
  options: PluginOptions = { name: "no-name" };
  private worker?: child_process.ChildProcess;

  constructor(options?: Partial<PluginOptions>) {
    if (options) {
      this.options = { ...this.options, ...options };
    }
  }

  inspectAndPrint(): Promise<void> {
    return new Promise<void>(async (resolve, reject) => {
      try {
        const inspector = new Inspector();
        await inspector.inspect();
        await inspector.print();
        resolve();
      }
      catch (e) {
        reject(e);
      }
    });
  }

  startInspectAndPrint(): void {
    this.startWorker();
    this.worker?.send({ type: WorkerCommand.InspectAndPrint, options: this.options });
  }

  private startWorker(): void {
    if (this.worker) {
      return;
    }
    this.worker = child_process.fork(path.join(__dirname, "worker.ts"));
    this.worker.on("message", msg => {
      if (msg === "error") {
        logger.echo("<black><bold><bgYellow> WARNING </bgYellow></bold></black> <yellow>- error ESLint</yellow>");
        process.exit(1);
      }
      else {
        logger.echo("<black><bold><bgYellow> WARNING </bgYellow></bold></black> <yellow>", `<yellow>ESLint(${this.options.name}) killing worker</yellow>`);
        this.killWorker();
      }
    });
  }

  private killWorker(): void {
    if (this.worker) {
      this.worker.kill();
    }
  }
}

export function pluginESLint(options?: Partial<PluginOptions>) {
  return (ctx: Context) => {
    ctx.ict?.on("complete", props => {
      const eslint = new ESLintHelper(options);
      ctx["eslint"] = eslint;
      if (ctx.config?.env?.NODE_ENV === "production") {
        logger.info(`ESLint (${eslint.options.name}):`, "inspecting code, please wait...");
        eslint.inspectAndPrint();
      }
      else {
        logger.info(`ESLint (${eslint.options.name}):`, "Starting thread. Will print status soon, please wait...");
        eslint.startInspectAndPrint();
      }
      return props;
    });
    ctx.ict?.on("rebundle", props => {
      const eslint: ESLintHelper = ctx["eslint"];
      if (eslint) {
        logger.info(`ESLint (${eslint.options.name}):`, "Calling thread for new report, please wait...");
        eslint.startInspectAndPrint();
      }
      return props;
    });
  };
}
