export type UUID = string

export type Obj = {
  obj: UUID
  name: string
  type: Type
  attributes: Attributes
  stats? :Attributes
}

export type Camera = Obj
export type Avatar = Obj
export type User = Obj
export type Role = Obj
export type Set = Obj

export type Attributes = {
  [key: string]: string
}

export enum Type {
  avatar = 'avatar',
  camera = 'camera',
  user = 'user',
  role = 'role',
  set = 'set',
  storagepool = 'storagepool',
  sensor = 'sensor'
}

export interface AvatarWithCameras extends Avatar {
  cameras: Camera[]
}

export type Session = {
  isLoggedIn: boolean,
  isGettingInfo: boolean,
  info?: SessionInfo
}

export type RawSession = {
  key: string,
  user_agent: string
}

export type SessionInfo = {
  user: { obj: UUID, name: string },
  role: { obj: UUID, name: string },
  realm: { obj: UUID, name: string }
}

export type UserSession = {
  username: string,
  userid: UUID,
  sessions: RawSession[]
}

/*
let event = {
  "eventId": "63374",
  "from": 1547484513000,
  "to": 1547486318000,
  "when": 1547484518000,
  "object": {
    "objId": "611f5a1e-f5cc-11e8-a107-42010a8e0005",
    "name": null,
    "type": null,
    "subtype": null,
    "tDst": null,
    "tZone": 0.0
  },
  "propertyList": null,
  "witnessList": null,
  "actionList": null,
  "mediasize": null,
  "source": 1,
  "state": 1,
  "priority": 3,
  "message": "MD",
  "eventtype": 1,
  "note": null,
  "tagList": [3, 5, 10, 11],
  "history": null,
  "lastModified": 1547486304084,
  "bgColor": "#f3503f"
};
*/

export type Event = {
  eventId: UUID,
  from: number, // unix time in ms
  to: number, // unix time in ms
  when: number, // unix time in ms
  lastModified: number, // unix time in ms
  message: string,
  objects: {
    objId: UUID,
    name: string
  }[],
  source?: EventSource,
  type?: EventType,
  eventCheck?: boolean, // temporary value
  eventIcon?: string // temporary value
  legacyMeta?: AuditlogLegacyMeta,
  v2?: {
    classifiers: {[key: string] : {value:string, attrs: {[key: string]: string}}}
  },
  category: number,
  snapshots?: {
    type: string,
    default: boolean,
    tags: {[key: string] : string},
    snapshot: string
  }[]
}

export enum EventSource {
  MotionDetector = 1,
  CameraMotionDetector,
  Sensor,
  User,
  AccessControlSystem,
  Analytics,
  Interview
}

export enum EventType {
  Informational = 0,
  Alert,
  Suppress
}

export type Storage = {
  size: number,
  ts: number
}

export type StorageTotal = {
  totalTB: number,
  max: number,
  min: number,
  avg: number,
  percent: number
}

export type StorageTotalPerTimestamp = {
  [timestamp: string]: number
}

export type AuditItemType = {
  name: string,
  templates: { [key: string] : string },
  objs: { id: UUID, name: string, type: string  }[]
}

export type AuditItemTypes = {
  [key: string] : AuditItemType
}

export type GetEventsRule = {
  witness: string[]
  category: number,
  entryType?: string,
  contexts?: string[]
}

export type ApiError = {
  code?: number,
  message: string,
  parameters?: {[key: string]: any},
  extCode?: number
}

export type ListenEventsRequest = {
  realm?: UUID,
  witness: UUID[],
  category: number,
  entryType: string
}

export type ManageSubscriptionRequest = {
  action: string,
  id: string,
  request: ListenEventsRequest
}

export type AuditlogLegacyMeta = {
  summary: string[],
  fields: {
    type: string,
    val1: string,
    val2: string,
    val3: string
  }[]
}

export type AuditlogEntry = {
  realm: string,
  inited_at: string,
  created: string,
  witness: string[],
  context: string,
  entryType: string,
  category: number,
  legacy? : AuditlogLegacyMeta,
  v2?: {
    classifiers: {[key: string] : {value:string, attrs: {[key: string]: string}}}
  },
  snapshots?: {
    type: string,
    default: boolean,
    tags: {[key: string] : string},
    snapshot: string
  }[]
}

export type ListenEventsResponse = {
  result: {
    type: string,
    id: string,
    message: string,
    entries: {
      entries: AuditlogEntry[]
    }
  },
  error?: {
    grpc_code: number,
    http_code: number,
    http_status: string,
    details: any
  }
}

export type FeatureUsage = {
  feature: string,
  name: string,
  displayName: string,
  unitsSubscribed: number,
  unitsUsed: number
}

export type StoragePool = {
  id: string,
  displayName: string,
  features: FeatureUsage[]
}

export type StoragePoolUsage = {
  id: string,
  name: string,
  usage: FeatureUsage[]
}

export type RealmUsage = {
  globalUsage: FeatureUsage[];
  storagePools: StoragePoolUsage[];
}

export type EventCounting = {
  analytics?: number;
  system?: number;
}

export enum EventSampling {
  HOURLY = 'HOURLY',
  DAYLY = 'DAYLY'
}

export type EventReportRow = {
  time: number;
  count: number;
}

export enum Tolerance {
  EXACT = 0,
  FIVE_MIN = 300,
  FIFTEEN_MIN = 15 * 60,
  ONE_HOUR = 3600
}

export type SnapshotQueueEntry = {
  obj: UUID,
  ts: number,
  buffer: string | null,
  listeners: number,
  priority: number,
  lastRequest: number,
  tolerance: Tolerance
}

export type AvatarContentDelivery = {
  avatar_delivery_stat: {
    histogram: {
      [key: string]: {
        ARCHIVE_METADATA?: ContentDeliveryHistogramItem[],
        ARCHIVE_ONDEMAND?: ContentDeliveryHistogramItem[],
        ARCHIVE_SCHEDULED?: ContentDeliveryHistogramItem[],
        LIVE?: ContentDeliveryHistogramItem[],
        LIVE_METADATA?: ContentDeliveryHistogramItem[],
        TOTAL?: ContentDeliveryHistogramItem[]
      }
    },
    last: {
      ARCHIVE_ANALYTICS: number,
      ARCHIVE_METADATA: number,
      ARCHIVE_ONDEMAND: number,
      ARCHIVE_SCHEDULED: number,
      LIVE: number
    }
  },
  details: {
    [key: string]: {name: string}
  },
  TOTAL?: ContentDeliveryHistogramItem[],
  TOTAL_HOURLY?: ContentDeliveryHistogramItem[],
  avatar_id?: UUID
}

export type ContentDeliveryHistogramItem = {
  s: number,
  t: number
}

export type VAEFeature = {
  category: string,
  enabled: boolean,
  engine: string,
  name: string,
  order: number,
  requires: string[]|null,
  title: string
}

export type VAEFeatures = {
  [key: string]: VAEFeature
}

export type Watchlist = {
  watchlist_id: UUID,
  watchlist_type_id: number,
  realm_id: UUID,
  name: string,
  description: string,
  threshold: number,
  action: string,
  ttl_days: number,
  entries: WatchlistEntry[]
}

export type WatchlistEntry = {
  description: string,
  expire_date: number,
  links?: string[],
  entry?: string,
  updated_at: number,
  watchlist_entry_id: UUID,
  watchlist_id: UUID
}


export type GetSnapstotsOpts = {
  default?: boolean,
  tags?: {[key:string]:string},
  type?: string
  return_all? :boolean
}

export type ReportsChart = {
  GRAPH_HEIGHT: number,
  TOTAL_COLOR: string,
  TICKS_NUM: number,
  CHART_COLORS: string[],
  EVENTS_COLOR: string
};

export type EventCSV = {
  id: string,
  date: string,
  source: string,
  description: string
};

export type DeviceEvents = {
  [obj in UUID]: {
    isChecked: boolean;
    events: Event[];
  }
};

export type ArrayElement<A> = A extends readonly (infer T)[] ? T : never;

export type Optional<T, K extends keyof T> = Pick<Partial<T>, K> & Omit<T, K>;
