/**
 * TimeLine interface class for working with visjs Timeline
 *
 * @author Ihor Cherhavyy
 * ------------------------------------------------------------------------------
 */
import {getSnapshotURL, Utils} from "@solid/libs/utils";
import {Timeline, TimelineEventPropertiesResult, TimelineOptions, TimelineTimeAxisOption, TimelineTimeAxisScaleType} from "vis-timeline/esnext";
import moment from "moment-timezone";
import SVGInjector from "svg-injector";
import {__} from "@solid/libs/i18n";
import {CoverageItem, Granularity} from "@solid/types/coverage";
import {UUID} from "@solid/types";

import LoaderIcon from "./img/loader.svg";
import PlusIcon from "./img/add_circle_24px.svg";
import MinusIcon from "./img/remove_circle_24px.svg";

import "vis-timeline/styles/vis-timeline-graph2d.css";
import "@solid/timeline/vistimeline.css";

export type TimeLineParameters = {
	node: HTMLDivElement | null,
	id: string,
	isDebug: boolean,
	type: TimeLineType,
	selection: {
		fixed: boolean,
		sticky?: boolean
	},
	scale: {
		granularity: Granularity
	},
	timezone: string,
	dateFormat: string,
	className: string,
	options: TimelineOptions,
	hideSelection?: boolean,
	showTime?: boolean
};

export type TimeLineDataList = {
	[timestamp: number]: CoverageItem & {
		percentage?: number,
		className: string,
		color?: number,
		borderColor?: number,
		gradient?: string,
		inArchive?: boolean,
		eventCount?: number
	}
};

export type TimeLineData = {
	list: TimeLineDataList,
	isEvent?: boolean,
	granularity: Granularity,
	color?: number,
	borderColor?: number
};

export type TimeLineTrack = {
	obj?: UUID;
	name: string,
	rows: TimeLineData[]
};

export type TimeLineItem = {
	id?: string | number,
	type?: string,
	start: number,
	end: number
	group: number,
	style: string,
	content: string
};

export type TimeLineGroup = {
	id: number,
	content: string,
	className?: string,
	style?: string
};

export type TimeLineInterval = {
	start: number,
	end: number
}

export type TimeLineDateInterval = {
	start: Date,
	end: Date
}

export type TimeLineEventInfo = {
	time: Date,
	html: string,
	snapshot: string
}

export type ImageClickEventArgs = {
	obj: UUID,
	time: number,
	gotoTime: boolean
}

let INTERVAL_FOR_GRANULARITY = {
	second: 1000,
	chunk: 30 * 1000,
	min: 60 * 1000,
	hour: 60 * 60 * 1000,
	day: 24 * 60 * 60 * 1000
};

export enum TimeLineType {
	MULTI_LINE = "MULTI_LINE",
	MULTI_TRACK = "MULTI_TRACK",
	SLIM = "SLIM"
}

// width of time block interval in px
export enum TimeLineBlockWidth {
	MINUTE = 36,
	DAY = 134
}

export class TimeLineError extends Error {
	constructor(...params: any) {
		super(...params);

		if (Error.captureStackTrace) {
			Error.captureStackTrace(this, TimeLineError);
		}
	}
}

type VisData = {
	groups: TimeLineGroup[],
	items: TimeLineItem[]
};

type TrackInfo = {
	index: number,
	obj?: UUID,
	objIndex?: number
};

const maxTrackCount = 9;
const thumbDelay = 500; //ms
const defaultHeight = 46;


/**
 * create new timeline on page
 *
 * @param {Object.<any>} parameters
 * @constructor
 */
export class VisTimeLine {
	_node: HTMLElement | null = null;

	_typeToClass: {[type in TimeLineType]: string} = {
		[TimeLineType.SLIM]: "timeline-slim",
		[TimeLineType.MULTI_LINE]: "timeline-multi-line",
		[TimeLineType.MULTI_TRACK]: "timeline-multi-track",
	}

	_parameters: TimeLineParameters = {
		id: "",
		node: null,
		type: TimeLineType.MULTI_LINE,
		className: this._typeToClass[TimeLineType.MULTI_LINE],
		isDebug: false,
		scale: {
			granularity: Granularity.chunk
		},
		selection: {fixed: false},
		dateFormat: "",
		timezone: "UTC",
		options: {
			stack: false,
			zoomable: false,
			width: '100%',
			height: defaultHeight,
			autoResize: true,
			showCurrentTime: false,
			verticalScroll: false,
			timeAxis: {
				scale: "minute",
				step: 1
			},
			format: {
				majorLabels: {
					hour: "DD.MM.YYYY",
					minute: "DD.MM.YYYY",
					second: "HH:mm",
					millisecond: "mm:ss"
				},
				minorLabels: {
					day: "DD.MM.YYYY",
					hour: 'HH:mm',
					minute: 'HH:mm'
				}
			},
			margin: {
				axis: 0
			},
			moveable: true, // flag for disabling timeline scrolling
			groupHeightMode: "fixed",
			zoomMin: 60 * 1000,
			zoomMax: 24 * 60 * 60 * 1000
		}
	}

	_TIMELINE: Timeline | null = null;
	_selection: TimeLineDateInterval | null = null;
	selection: TimeLineInterval | null = null;
	_limit = {
		start: 0,
		end: Number.MAX_SAFE_INTEGER
	};

	MIN_TIME_CHANGE_INTERVAL: number = 1000; // ms
	MIN_REQUEST_DATA_INTERVAL: number = 30 * 1000; // ms
	_lastRequestedData = {
		start: 0,
		end: 0,
		time: 0
	};

	_dragged: boolean = false;

	_relatedTarget: HTMLElement | null = null;

	_oldWidth: number = -1;
	_centerPosition: number = 0.5;

	_minWidth: number = 150;

	_tracks: TimeLineTrack[] = [];
	_selectedTrack?: TrackInfo;
	_grayedTracks: TrackInfo[] = [];
	_requestedEventTime = { startTime: -1, endTime: -1 };
	_requestedThumbX = -1;
	_requestedThumbY = -1;
	_requestedThumbTrack = -1;
	_mouseInThumbnail = false;
	_trackThumbInitialized = false;
	_mouseInArrow = false;
	_time = 0;
	_thumbTime = 0;
	_zoomPlusDisabled = false;
	_zoomMinusDisabled = false;
	_height: string | number;

	constructor(parameters: {
		node: HTMLDivElement | null,
		type?: TimeLineType,
		isDebug?: boolean,
		options?: TimelineOptions,
		className?: string,
		hideSelection?: boolean,
		showTime?: boolean
	}) {
		this._parameters = {
			...this._parameters,
			node: parameters.node ?? null,
			type: parameters.type ?? this._parameters.type,
			className: (parameters.type ? this._typeToClass[parameters.type] : undefined) ?? this._parameters.className,
			isDebug: parameters.isDebug ?? this._parameters.isDebug,
			hideSelection: parameters.hideSelection ?? this._parameters.hideSelection,
			showTime: parameters.showTime ?? this._parameters.showTime,
			options: {
				...this._parameters.options,
				...parameters.options
			}
		};
		this._height = this._parameters.options.height ?? defaultHeight;
	}

	/**
	 * Init timeline
	 *
	 * @returns {Promise<void>}
	 */
	init(): Promise<void> {
		return new Promise((resolve, reject) => {
			this._node = this._parameters.node;
			if (!this._node) {
				reject();
				return;
			}

			let timelineBlock = this._node;

			// clear wrapper
			while (timelineBlock.firstChild) {
				timelineBlock.removeChild(timelineBlock.firstChild);
			}

			let borderDate = this._getBorderDate(Date.now());
			if (borderDate) {
				this._parameters.options.start = borderDate.start;
				this._parameters.options.end = borderDate.end;
			}

			this._parameters.options.onInitialDrawComplete = () => {
				this._onLoad();
				resolve();
			};

			this._TIMELINE = new Timeline(timelineBlock, [], this._parameters.options);
		});
	}

	/**
	 *
	 * @param {number} time
	 * @param {Function} [callback]
	 * @private
	 */
	_setWindow(time: number, callback?: Function) {
		if (!this._TIMELINE) {
			return;
		}

		let borderDate = this._getBorderDate(time);
		if (borderDate
		    && (borderDate.end - borderDate.start > 0)
		) {
			this._TIMELINE.setWindow(borderDate.start, borderDate.end, {animation: false}, callback ? () => callback() : undefined);
		}
	}

	/**
	 *
	 * @param {number} time
	 * @returns {{start: number, end: number}|null}
	 * @private
	 */
	_getBorderDate(time: number): TimeLineInterval | null {
		const width = this.getWidth();

		let start, end;
		let minutes = width / TimeLineBlockWidth.MINUTE;
		let days = width / TimeLineBlockWidth.DAY;
		switch (this._parameters.options?.timeAxis?.scale) {
			case 'second':
			case 'minute':
				start = time - minutes * this._centerPosition * 60 * 1000;
				end = time + minutes * (1 - this._centerPosition) * 60 * 1000;
				break;
			case 'hour':
				start = time - minutes * this._centerPosition * 60 * 60 * 1000;
				end = time + minutes * (1 - this._centerPosition) * 60 * 60 * 1000;
				break;
			case 'day':
				start = time - days * this._centerPosition * 24 * 60 * 60 * 1000;
				end = time + days * (1 - this._centerPosition) * 24 * 60 * 60 * 1000;
				break;
			default:
				break;
		}

		return start && end ? {
			start: start,
			end: end
		} : null;
	}

	/**
	 * set timeline parameters
	 *
	 * Current method disable timeline scrolling effect as well as setOption vis.js method
	 *
	 * @param {Object.<String, ?(Object|String)>} params
	 */
	setParameters(params: Partial<TimeLineParameters>) {
		if (params.timezone) {
			let getOffset = null;
			const timezone = params.timezone;
			if (params.timezone.toLowerCase() === "local") {
				let offset = (new Date()).getTimezoneOffset() / 60;
				getOffset = function (date: any): moment.Moment {
					return moment(date).utcOffset('+' + offset + ':00');
				};
			} else {
				getOffset = function (date: any): moment.Moment {
					return moment(date).tz(timezone);
				};
			}
			// @ts-ignore
			this._parameters.options.moment = getOffset;
		}

		if (params.dateFormat
		    && this._parameters.options?.format?.majorLabels
		    && typeof this._parameters.options?.format?.majorLabels == "object"
		) {
			this._parameters.options.format.majorLabels.hour = params.dateFormat;
			this._parameters.options.format.majorLabels.minute = params.dateFormat;
		}

		if (params.scale) {
			let scale: string = params.scale.granularity;
			if (scale === Granularity.min) {
				scale = "minute";
			}

			const checkScale = (scale: string): scale is TimelineTimeAxisScaleType => ['second', 'minute', 'hour', 'day', 'month', 'year'].includes(scale);

			if (checkScale(scale)) {
				if (this._parameters.options?.timeAxis?.scale) {
					this._parameters.options.timeAxis.scale = scale;
				}
			}

			this._parameters.options.showMajorLabels = scale !== "day";
		}

		this._TIMELINE?.setOptions(this._parameters.options);
		this._height = this._parameters.options.height ?? defaultHeight;
	}

	/**
	 * create thumbnail on timeline
	 *
	 * @param {string} obj
	 */
	createThumb(obj: UUID) {
		if (!this._node) {
			return;
		}

		let timelineBlock = this._node;
		let timeout: NodeJS.Timeout;
		let div: HTMLElement | null | undefined = timelineBlock.parentNode?.querySelector('.thumbnail-holder');
		let imgTime: HTMLElement | null;
		let img: HTMLImageElement | null;

		if (!div) {
			// create the thumbnail
			div = document.createElement('div');
			div.className = 'thumbnail-holder';
			div.style.display = 'none';

			imgTime = document.createElement('span');
			imgTime.className = 'thumbnail-time';

			img = document.createElement('img');
			img.className = 'thumbnail';
			img.src = LoaderIcon;

			div.appendChild(img);
			div.appendChild(imgTime);
			timelineBlock.before(div);

			this._TIMELINE?.on('mouseMove', (props: TimelineEventPropertiesResult) => {
				if (props.what === 'item' || props.what === 'background') {
					if (!div || !imgTime || !img) {
						return;
					}

					const width = this.getWidth();
					let time = this._getTime(props.x / width);
					if (time > Date.now()) {
						return;
					}

					let halfWidth = 180;
					let centerMousePosition = props.x - halfWidth / 2;
					if (props.x < halfWidth / 2) {
						centerMousePosition = 0;
					}
					if (props.x + halfWidth / 2 > width) {
						centerMousePosition = width - halfWidth;
					}

					div.style.display = 'unset';
					div.style.left = centerMousePosition + 'px';

					let datetime = new Date(time);
					imgTime.innerText = moment(datetime).format("HH:mm:ss");

					img.style.width = halfWidth + 'px';
					img.style.height = halfWidth / 2 + 'px';

					if (timeout !== null) {
						clearTimeout(timeout);
						img.src = LoaderIcon;
					}

					timeout = globalThis.setTimeout(() => {
						if (!img) {
							return;
						}

						img.src = getSnapshotURL(obj, Math.trunc(time / 1000), true);

						timelineBlock?.querySelector('.vis-foreground')?.addEventListener("mouseleave", () => {
							clearTimeout(timeout);

							if (!div) {
								return
							}

							div.style.display = 'none';
						});
					}, thumbDelay);
				}
			});

			timelineBlock.addEventListener("mouseleave", (props: MouseEvent) => {
				if (!div) {
					return;
				}

				div.style.display = 'none';
			});
		}
	};

	initTrackThumb(): void {
		const timeline = this._node;
		if (!timeline || !this._TIMELINE) {
			return;
		}

		if (this._trackThumbInitialized) {
			return;
		}
		this._trackThumbInitialized = true;

		let timeout: NodeJS.Timeout | undefined;
		let removeTimeout: NodeJS.Timeout | undefined;

		const clearThumbTimeout = () => {
			if (timeout) {
				clearTimeout(timeout);
			}
		}

		const removeThumb = () => {
			if (removeTimeout) {
				clearTimeout(removeTimeout);
			}
			const div = document.querySelector('.thumb-track-holder');
			if (div) {
				removeTimeout = globalThis.setTimeout(() => {
					if (!this._mouseInThumbnail) {
						clearThumbTimeout();
						this.hideTrackThumb();
					}
				}, 250);
			}
		}

		const foreground = timeline.querySelector('.vis-foreground') as HTMLDivElement;
		foreground?.addEventListener('mouseleave', (event: MouseEvent) => removeThumb());

		this._TIMELINE.on('mouseMove', (props: TimelineEventPropertiesResult) => {
			if (props.what !== 'item' && props.what !== 'background') {
				return;
			}

			if (this._dragged || this._mouseInArrow) {
				return;
			}

			let time = this._getTimeByX(props.x);
			if (time < 0 || time > Date.now()) {
				return;
			}
			this._thumbTime = time;

			const trackIndex = this._getTrackIndex(props.pageY);
			if (trackIndex < 0) {
				return;
			}
			const obj = this._tracks[trackIndex].obj;
			if (!obj) {
				return;
			}

			let div: HTMLDivElement | null = document.querySelector('.thumb-track-holder');
			const reloadThumb = !div || this._requestedThumbX < 0 || this._requestedThumbTrack < 0 ||
				this._requestedThumbTrack !== trackIndex || Math.abs(this._requestedThumbX - props.pageX) > 5;

			if (reloadThumb) {
				clearThumbTimeout();
			}

			if (div && reloadThumb) {
				div.remove();
				div = null;
			}

			if (!div) {
				div = document.createElement('div');
				div.className = 'thumb-track-holder';

				div.addEventListener('mouseenter', (event: MouseEvent) => {
					this._mouseInThumbnail = true;
					if (removeTimeout) {
						clearTimeout(removeTimeout);
					}
				});

				div.addEventListener('mouseleave', (event: MouseEvent) => {
					this._mouseInThumbnail = false;
					clearThumbTimeout();
					this.hideTrackThumb();
				});

				const imgWrapper = document.createElement('div');
				imgWrapper.className = 'thumbnail-wrapper'

				const imgTime = document.createElement('span');
				imgTime.className = 'thumbnail-time';

				const img = document.createElement('img');
				img.className = 'thumbnail';
				img.src = LoaderIcon;
				img.addEventListener('click', () => {
					this.hideTrackThumb();
					const args: ImageClickEventArgs = { obj, time: this._thumbTime, gotoTime: true };
					this.onImageClick(args);
					if (args.gotoTime) {
						this.setTime(args.time, true);
					}
				});

				imgWrapper.appendChild(img);
				imgWrapper.appendChild(imgTime);
				div.appendChild(imgWrapper);

				document.body.appendChild(div);
			}

			const imgTime: HTMLSpanElement | null = div.querySelector('.thumbnail-time');
			if (imgTime) {
				imgTime.innerText = moment(new Date(time)).format("HH:mm:ss");
			}

			const { count: eventCount, startTime, endTime } = this._getEventCount(trackIndex, props.x);

			const addEventLink = () => {
				const div = document.querySelector('.thumb-track-holder');
				const eventText = div?.querySelector('.thumbnail-event');
				if (eventText) {
					eventText.innerHTML = '';
					const link = document.createElement('a');
					link.innerText = eventCount.toString() + ' ' + __('event(s)');
					link.addEventListener('click', () => {
						this.hideTrackThumb();
						this.onEventClick(startTime, endTime, obj);
					});
					eventText.appendChild(link);
				}
			}

			if (reloadThumb) {
				let eventText: HTMLDivElement | null = div.querySelector('.thumbnail-event');
				if (eventText) {
					eventText.remove();
				}
				if (eventCount > 0) {
					eventText = document.createElement('div');
					eventText.className = 'thumbnail-event';
					div.appendChild(eventText);
					if (eventCount === 1) {
						eventText.innerText = __('Loading event...');
					} else {
						addEventLink();
					}
				}
			}

			const positionThumb = (force?: boolean) => {
				const div: HTMLDivElement | null = document.querySelector('.thumb-track-holder');
				if (div) {
					let pageY = props.pageY;
					if (!force && this._requestedThumbY >= 0 && Math.abs(this._requestedThumbY - props.pageY) <= 10) {
						pageY = this._requestedThumbY;
					}
					else {
						this._requestedThumbY = props.pageY;
					}

					const thumbRect = div.getBoundingClientRect();
					const left = props.pageX - thumbRect.width / 2;
					const top = pageY - thumbRect.height - 8;
					div.style.left = `${left}px`;
					div.style.top = `${top}px`;
				}
			}
			positionThumb();

			if (!reloadThumb) {
				return;
			}

			timeout = globalThis.setTimeout(async () => {
				const div = document.querySelector('.thumb-track-holder');
				if (!div) {
					return;
				}

				this._requestedThumbX = props.pageX;
				this._requestedThumbY = props.pageY;
				this._requestedThumbTrack = trackIndex;

				const img: HTMLImageElement | null = div.querySelector('.thumbnail');
				if (img) {
					img.src = getSnapshotURL(obj, Math.trunc(time / 1000), true);
				}

				if (eventCount === 1) {
					this._requestedEventTime = { startTime, endTime };
					try {
						let info: TimeLineEventInfo | undefined;
						try {
							info = await this.onGetEventInfo(startTime, endTime, obj);
						}
						catch (e) {
							console.error('Error loading event:', e);
						}

						if (this._requestedEventTime.startTime !== startTime || this._requestedEventTime.endTime !== endTime) {
							return;
						}

						const div = document.querySelector('.thumb-track-holder');
						if (!div) {
							return;
						}

						const imgTime: HTMLSpanElement | null = div.querySelector('.thumbnail-time');
						const eventText = div.querySelector('.thumbnail-event');
						const img: HTMLImageElement | null = div.querySelector('.thumbnail');

						if (imgTime && info?.time.getTime()) {
							time = info.time.getTime();
							imgTime.innerText = moment(info.time).format("HH:mm:ss");
						}
						if (eventText && info?.html) {
							eventText.innerHTML = info.html;
						} else {
							addEventLink();
						}
						if (img && info?.snapshot) {
							img.src = `data:image;base64, ${info.snapshot}`;
						}
						positionThumb(true);
					}
					finally {
						if (this._requestedEventTime.startTime === startTime && this._requestedEventTime.endTime === endTime) {
							this._requestedEventTime = { startTime: -1, endTime: -1 };
						}
					}
				}
			}, thumbDelay);
		});

		timeline.addEventListener('mouseleave', (event: MouseEvent) => removeThumb());
	}

	hideTrackThumb(): void {
		this._requestedThumbX = -1;
		this._requestedThumbTrack = -1;
		const div = document.querySelector('.thumb-track-holder');
		if (div) {
			div.remove();
		}
	}

	private _getTrackIndex(pageY: number): number {
		const timeline = this._node;
		if (!timeline) {
			return -1;
		}
		for (let i = 1; i < maxTrackCount + 1; i++) {
			const groupElements = Array.from(timeline.querySelectorAll(`.t${i}`));
			for (const element of groupElements) {
				const rect = element.getBoundingClientRect();
				if (pageY >= rect.top && pageY <= rect.bottom) {
					return i - 1;
				}
			}
		}
		return -1;
	}

	/**
	 * Set timeline scrolling limit
	 *
	 * @param {number} start
	 * @param {number} end
	 */
	setLimit(start: number, end: number) {
		this._limit.start = start;
		this._limit.end = end;
	};

	/**
	 * event handler for timeline load event
	 */
	_onLoad() {
		if (!this._node) {
			return;
		}
		if (!this._TIMELINE) {
			return;
		}

		const timelineBlock = this._node.querySelector('.vis-timeline');
		if (!timelineBlock) {
			return;
		}
		let self = this;

		// add specific class name for each timeline type
		this._node.classList.add(this._parameters.className);

		this._TIMELINE.on('mouseDown', function(props: TimelineEventPropertiesResult & {event: MouseEvent}){
			// if not LEFT mouse button
			if (!(props.event.button === 0)) {
				return;
			}

			self._dragged = true;
			self.hideTrackThumb();
		});
		this._TIMELINE.on('mouseUp', function(props: TimelineEventPropertiesResult & {event: MouseEvent}){
			// if not LEFT mouse button
			if (!(props.event.button === 0)) {
				return;
			}

			self._dragged = false;
		});

		this._TIMELINE.on('rangechange', (props: { start: number, end: number }) => {
      this.onRangeChange();

			this._scaleTimeAxis(props.start, props.end);
			this._zoomPlusDisabled = !!this._parameters.options.zoomMin && props.end - props.start <= this._parameters.options.zoomMin;
			this._zoomMinusDisabled = !!this._parameters.options.zoomMax && props.end - props.start >= this._parameters.options.zoomMax;
			this._addRemoveZoomButtonsClass('disabled', this._zoomPlusDisabled, '.plus');
			this._addRemoveZoomButtonsClass('disabled', this._zoomMinusDisabled, '.minus');
		});

		this._TIMELINE.on('rangechanged', async function(props: TimelineEventPropertiesResult & {byUser: boolean}) {
			if (props.byUser) {
				self._dragged = false;
			}

			let currentCenterTime = self._getCenterTime();

			// limit time to interval
			if (currentCenterTime < self._limit.start) {
				currentCenterTime = self._limit.start;
			}
			if (currentCenterTime > self._limit.end) {
				currentCenterTime = self._limit.end;
			}

			if (props.byUser) {
				// prevent going to future for user
				const currentTime = Date.now();
				if (currentCenterTime > currentTime) {
					currentCenterTime = currentTime;
				}
			}

			try {
				await self._setTime(currentCenterTime, false, props.byUser);
				self._onTimeChange(currentCenterTime, true, props.byUser);
			}
			catch (e) {
				if (e instanceof TimeLineError) {
				} else {
					throw e;
				}
			}
		});

		const onChanged = function () {
			self._resizeHandler();
			self._TIMELINE?.off('changed', onChanged);
		};
		this._TIMELINE.on('changed', onChanged);

		this._TIMELINE.on('click', props => this.onClick(props));

		const panel: HTMLElement | null = this._node.querySelector('.vis-panel.vis-background.vis-vertical');

		const resizeObserver = new ResizeObserver(Utils.throttleToDraw((entries: ResizeObserverEntry[]) => {
			for (const entry of entries) {
				if (self._node === entry.target) {
					self._resizeHandler();
				}
				if (panel === entry.target) {
					self._resizeHandler();
				}
				const selectionSeparator = self._node?.querySelector(".selection-separator");
				if (entry.target === selectionSeparator) {
					self._positionSeparator();
				}
			}
		}));

		if (panel) {
			resizeObserver.observe(panel);
		} else {
			resizeObserver.observe(this._node);
		}

		this._addSeparator()
		// this._addSelectionBars();
		const selectionSeparator: HTMLElement | null = this._node.querySelector(".selection-separator");
		if (selectionSeparator) {
			selectionSeparator.style.left = this._getSeparatorLeft();
		}
		// this.clearSelection();

		if (this._parameters.type !== TimeLineType.SLIM) {
			const addHideClass = () => {
				if (!selectionSeparator) {
					return;
				}

				if (selectionSeparator.childNodes.length <= 2) {
					// if (this.getSelection()) {
					// 	return;
					// }
					selectionSeparator.classList.add('hide');
				}
			};

			const removeHideClass = () => {
				if (!selectionSeparator) {
					return;
				}

				selectionSeparator.classList.remove('hide');
			};

			addHideClass();

			const dataLines = this._node.querySelector('.vis-foreground');
			if (dataLines) {
				dataLines.addEventListener('mouseenter', removeHideClass);
				dataLines.addEventListener('mouseleave', addHideClass);
			}

			if (selectionSeparator) {
				selectionSeparator.addEventListener('mouseenter', removeHideClass);
				selectionSeparator.addEventListener('mouseleave', addHideClass);
			}
		}

		if (this._parameters.options.zoomable) {
			this._addZoomButtons();
		}

		const leftPanel = timelineBlock.querySelector('.vis-panel.vis-left');
		leftPanel?.addEventListener('mouseenter', () => this.hideTrackThumb());

		this._TIMELINE.setGroups([{id: 1, content: ""}]);
	}

	/**
	 *
	 * @returns {number}
	 */
	_getCenterTime(): number {
		return this._getTime(this._centerPosition);
	}

	/**
	 *
	 * @param {number} time
	 * @returns {number}
	 * @private
	 */
	_timeToX(time: number): number {
		if (!this._TIMELINE) {
			return 0;
		}

		let window = this._TIMELINE.getWindow();
		let start = window.start.getTime();
		let end = window.end.getTime();

		return (time - start) / (end - start) * this.getWidth();
	};

	/**
	 * get time on position
	 *
	 * @param {number} position [0,1]
	 * @returns {number}
	 * @private
	 */
	_getTime(position: number): number {
		if (!this._TIMELINE) {
			return 0;
		}

		let window = this._TIMELINE.getWindow();
		let start = window.start.getTime();
		let end = window.end.getTime();

		return start + (end - start) * position;
	};

	/**
	 *
	 * @private
	 */
	_resizeHandler() {
		if (!this._node) {
			return;
		}

		let timelineBlock: HTMLElement | null = this._node.querySelector('.vis-timeline');
		if (!timelineBlock) {
			return;
		}

		const width = this.getWidth();
		if (this._oldWidth < 0) {
			this._oldWidth = width;
			this._setWindow(this._getCenterTime());
		} else
		if (width > 0
		    && width !== this._oldWidth) {
			this._oldWidth = width;
			this._setWindow(this._getCenterTime());
		}
		this._positionSeparator();
	}

	/**
	 * set time to timeline
	 *
	 * @param {number} time Unix time in ms
	 * @param {boolean} isCallback
	 * @param {boolean} [animate]
	 * @param {boolean} [byUser]
	 * @returns {Promise<any>}
	 * @private
	 */
	_setTime(time: number, isCallback: boolean = true, animate: boolean = false, byUser: boolean = false): Promise<void> {
		return new Promise((resolve, reject) => {
			if (!this._TIMELINE) {
				return;
			}

			if (this._dragged) {
				reject(new TimeLineError("User dragged timeline"));
				return;
			}

			// do not move to small interval
			if (Math.abs(time - this._getCenterTime()) < this.MIN_TIME_CHANGE_INTERVAL) {
				resolve();
				return;
			}

			let window = this._TIMELINE.getWindow();
			let start = window.start.getTime();
			let end = window.end.getTime();

			let realCenter = time + (end - start) * (0.5 - this._centerPosition);

			// if (this.getSelection() && this.selection) {
			// 	if (time > this.selection.end) {
			// 		reject(new TimeLineError("Cannot set time to future"));
			// 		return;
			// 	}
			// 	this._setSelectionByTime(this.selection.start, this.selection.end);
			// } else {
			// 	this.selection = this.getSelection();
			// 	// this._setSelectionByTime(this.selection.start, this.selection.end);
			// }

			this._TIMELINE.moveTo(realCenter, {animation: animate}, () => {
				// TODO: remove need of this wait
				// wait for timeline draw new time
				globalThis.setTimeout(resolve, 0);

				this._onTimeChange(this._getCenterTime(), isCallback, byUser);
			});
		});
	};

	/**
	 * set time to timeline
	 *
	 * @param {number} time Unix time in ms
	 * @param {boolean} isCallback
   * @param {boolean} byUser
	 * @returns {Promise<any>}
	 */
	setTime(time: number, isCallback: boolean = true, byUser: boolean = false) {
		return this._setTime(time, isCallback, undefined, byUser);
	};

	/**
	 * @returns {Number} time Unix time in ms
	 */
	getTime(): number {
    if (!this._TIMELINE) {
      console.error("this._TIMELINE is undefined");
      return 0;
		}

		return this._getCenterTime();
	};

	clearData() {
		this._lastRequestedData.start = 0;
		this._lastRequestedData.end = 0;
		this.setData('', []);
	};

	destroy() {
		this.clearData();
		this._TIMELINE?.destroy();
	}

	/**
	 * set data for drawing intervals
	 *
	 * @param {String} line
	 * @param {TimeLineTrack[]} data
	 */
	setData(line: string, data: TimeLineTrack[] = []) {
		const time = performance.now();

		if (!this._TIMELINE) {
			return;
		}

		const log = (message: string, data: TimeLineTrack[]) => {
			const count = (track: TimeLineTrack) => track.rows.map((row) => Object.keys(row.list).length);

			const trackCount = data.map((track) => {
				return count(track);
			});
			const trackCountByTrack = trackCount.map((track) => track.reduce((a, c) => a + c), 0);
			const trackCountAll = trackCountByTrack.reduce((a, c) => a + c, 0);
			console.log(message, data, trackCountAll, trackCountByTrack, trackCount, );
		}
		this._parameters.isDebug &&  log("> 1", data);
		const mergedData: TimeLineTrack[] = this._merge(data);
		this._parameters.isDebug && log("> 2", mergedData);

		const visData: VisData = { groups: [], items: [] };
		const options: Partial<TimelineOptions> = {};

		this._tracks = data;
		for (let i = 0; i < mergedData.length; i++) {
			this._addTrack(visData, mergedData[i], i, this._parameters.type === TimeLineType.MULTI_TRACK);
		}

		if (this._parameters.type == TimeLineType.MULTI_TRACK) {
			options.height = `${defaultHeight + 20 + mergedData.length * 26}px`;
		} else {
			options.height = this._height;
		}

		this._TIMELINE.setData(visData);
		this._TIMELINE.setOptions(options);
		this._parameters.options = { ...this._parameters.options, ...options };

		if (this._parameters.type !== TimeLineType.SLIM) {
			// Visualize data slicing
			const elGroups = Array.from(document.querySelectorAll('.vis-group'));
			elGroups.forEach(function (el) {
				el.classList.add('multi-line');
			});
		}

		if (this._parameters.type === TimeLineType.MULTI_TRACK) {
			if (this._selectedTrack) {
				const index = this._findTrackIndex(this._selectedTrack);
				if (index === this._selectedTrack.index) {
					this._addTrackClass([index], 'highlighted');
				} else {
					this.selectTrack(index);
				}
			}
			const grayedIndices = this._grayedTracks.map(t => this._findTrackIndex(t));
			this.setGrayedTracks(grayedIndices);
		}

		this._addRemoveZoomButtonsClass('visible', mergedData.length > 0);

		this._parameters.isDebug && console.log('VisTimeLine.setData', `${performance.now() - time}ms`);
	};

	private _addTrack(visData: VisData, track: TimeLineTrack, trackIndex: number, isMultiTrack?: boolean): void {
		let data = track.rows;

		let isOneLine = this._parameters.type === TimeLineType.SLIM;

		const eventData = [];
		let lineData: TimeLineData[] = [];

		if (typeof data === 'undefined' || !data.length) {
			data = [{list: {}, granularity: Granularity.chunk}, {list: {}, granularity: Granularity.chunk}];
		}
		if (data.length === 1) {
			data.push({list: {}, granularity: Granularity.chunk});
		}

		for (let i = 0; i < data.length; i++) {
			if (data[i].isEvent) {
				eventData.push(data[i])
			} else {
				lineData.push(data[i])
			}
		}

		for (let i = 0; i < lineData.length; i++) {
			const line = lineData[i];
			const timeline = Object.keys(line.list).map((timestamp, index) => {
				const item = line.list[timestamp];
				return {
					start: Number(timestamp),
					end: Number(item.last) + INTERVAL_FOR_GRANULARITY[line.granularity],
					group: isOneLine ? 1 : visData.groups.length,
					className: item.className,
					style: [
						`height: ${isOneLine ? item.percentage + "%" : "5px"};`,
					].join(''),
					content: ""
				};
			});

			visData.groups.push({
				id: visData.groups.length,
				content: i === 0 ? track.name : '',
				className: isMultiTrack ? (i === 0 ? `track-first-line t${trackIndex + 1}` : `t${trackIndex + 1}`) : undefined,
				style: i !== 0 ? 'display: none;' : undefined
			});

			visData.items = [...visData.items, ...timeline];
		}

		if (eventData.length) {
			const gropedEventDataList = { list: {}, granularity: Granularity.second };
			for (let i = 0; i < eventData.length; i++) {
				Object.assign(gropedEventDataList.list, eventData[i].list)
			}
			const eventLine = Object.keys(gropedEventDataList.list).map((timestamp, index) => {
				const item = gropedEventDataList.list[timestamp];
				return {
					start: Number(timestamp),
					end: Number(item.last) + INTERVAL_FOR_GRANULARITY[gropedEventDataList.granularity],
					group: isOneLine ? 1 : visData.groups.length,
					className: item.className,
					style: [
						`height: ${isOneLine ? item.percentage + "%" : "10px"};`
					].join(''),
					content: ""
				};
			});

			visData.groups.push({
				id: visData.groups.length,
				content: '',
				className: isMultiTrack ? `event-line t${trackIndex + 1}` : 'event-line',
				style: 'display: none'
			});

			visData.items = [...visData.items, ...eventLine]
		}

		if (isOneLine) {
			visData.groups = [{id: 1, content: "", style: "width: 200px;"}];
		}
	}

	/**
	 * search for sequence in data and merge similar items with specifying sequence length
	 *
	 * @param data
	 * @param granularity
	 */
	_merge(data: TimeLineTrack[] = [], granularity: Granularity = Granularity.chunk): TimeLineTrack[] {
		const newData: TimeLineTrack[] = [];

		const similar = (a: CoverageItem, b: CoverageItem): boolean => {
			return Boolean((a.isPartialChunk && b.isPartialChunk
			               || a.isChunk && b.isChunk
			               || a.isEvent && b.isEvent) && ((a.isSelected ?? false) === (b.isSelected ?? false)));
		};

		for (let trackNumber =  0; trackNumber < data.length; trackNumber++) {
			const track = data[trackNumber];
			const newTrack: TimeLineTrack = {name: track.name, rows: []};

			for (let stream = 0; stream < track.rows.length; stream++) {
				const row = track.rows[stream];
				const list = row.list;

				// calculate current granularity minimum block length
				const now = Date.now();
				const granularityWidth = this._timeToX(now + INTERVAL_FOR_GRANULARITY[row.granularity]) - this._timeToX(now); // px
				const minGranularityWidth = 4; // px
				let blockTimeLength = INTERVAL_FOR_GRANULARITY[row.granularity]; // ms
				if (granularityWidth < minGranularityWidth) {
					blockTimeLength = this._getTimeByX(minGranularityWidth) - this._getTimeByX(0);
				}

				const newList = {};
				let length = 0;
				let prevTimestamp = 0;
				let firstInSequence = {
					timestamp: 0,
					element: {}
				};

				// select first, minimum element
				let firstTimestamp = Number.MAX_SAFE_INTEGER;
				let lastTimestamp = Number.MIN_SAFE_INTEGER;
				for (let timestamp in list) {
					if (Number(timestamp) < firstTimestamp) {
						firstTimestamp = Number(timestamp);
					}
					if (Number(timestamp) > lastTimestamp) {
						lastTimestamp = Number(timestamp);
					}
				}
				if (firstTimestamp <= lastTimestamp) {
					firstInSequence = {
						timestamp: Number(firstTimestamp),
						element: {...list[firstTimestamp]}
					};

					let timeStampList = Object.keys(row.list).sort();
					for (let timestamp of timeStampList) {
						const current = list[timestamp];
						const isSimilar = similar(current, firstInSequence.element);
						if (!isSimilar
						    || Number(timestamp) > prevTimestamp + blockTimeLength
						) {
							newList[firstInSequence.timestamp] = {
								...firstInSequence.element,
								last: Number(prevTimestamp),
								length: length
							};
							firstInSequence = {
								timestamp: Number(timestamp),
								element: {...current}
							};
							length = 1;
						} else {
							length++;
						}

						prevTimestamp = Number(timestamp);
					}
					newList[firstInSequence.timestamp] = {
						...firstInSequence.element,
						last: prevTimestamp,
						length: length
					};
				}

				newTrack.rows.push({
					...row,
					list: newList
				});
			}

			newData.push(newTrack);
		}

		return newData;
	}

	updateData() {
		let granularity: Granularity = Granularity.chunk;
		switch (this._parameters.options.timeAxis?.scale) {
			case 'second':
			case 'minute':
				granularity = Granularity.min;
				break;
			case 'hour':
				granularity = Granularity.hour;
				break;
			case 'day':
				granularity = Granularity.day;
				break;
			default:
			break;
		}

		this.onGetData('up', this._lastRequestedData.start, this._lastRequestedData.end, granularity);
	}

	convertGroupToTimelineItems(group: TimeLineData, options: {groupId: string}) {
		let isOneLine = this._parameters.type === TimeLineType.SLIM;

		return Object.keys(group.list).map(function(timestamp) {
			return {
				start: Number(timestamp),
				end: Number(timestamp) + INTERVAL_FOR_GRANULARITY[group.granularity],
				group: options.groupId,
				style: [
					'background: #', group.color?.toString(16), ';',
					'height: ', isOneLine ? group.list[timestamp].percentage : 100, '%'
				].join('')
			};
		});
	};

	_timestampToString(timestamp: number) {
		let date = new Date();
		date.setTime(timestamp);

		let minute = date.getMinutes();
		let hour = date.getHours();

		return `${hour}:${String(minute).padStart(2, '0')}`;
	};

	/**
	 *
	 * @param {number} beginTime
	 * @param {number} endTime
	 * @private
	 */
	// _setSelectionByTime(beginTime: number, endTime: number){
	// 	if (!this._node) {
	// 		return;
	// 	}

	// 	const timelineBlock = this._node;
	// 	const selLeft: HTMLElement | null = timelineBlock.querySelector('.tl-selection.left-selection');
	// 	const selRight: HTMLElement | null = timelineBlock.querySelector('.tl-selection.right-selection');
	// 	const selectionSeparator: HTMLElement | null = timelineBlock.querySelector('.selection-separator');

	// 	if (selLeft && selRight && selectionSeparator) {
	// 		const beginPosition = this._timeToX(beginTime);
	// 		const endPosition = this._timeToX(endTime);

	// 		selLeft.style.width = (selectionSeparator.offsetLeft - beginPosition) + 'px';
	// 		selRight.style.width = (endPosition - selectionSeparator.offsetLeft) + 'px';
	// 		// selRight.style.width = endPosition + 'px';
	// 	}
	// }

	/**
	 * Set selected interval
	 *
	 * @param {Number} beginTime begin of the time interval in Unix time in ms
	 * @param {Number} endTime end of the time interval in Unix time in ms
	 * @param {Boolean} isCallback if true send callback of changing interval, or if false not send
	 */
	// setSelection(beginTime: number, endTime: number, isCallback: boolean) {
	// 	if (this._parameters.type === TimeLineType.SLIM) {
	// 		// not implemented
	// 		return;
	// 	}

	// 	if (!this._node) {
	// 		return;
	// 	}

	// 	let timelineBlock = this._node;
	// 	let selLeft: HTMLElement | null = timelineBlock.querySelector('.tl-selection.left-selection');
	// 	let selRight: HTMLElement | null = timelineBlock.querySelector('.tl-selection.right-selection');

	// 	if (!selLeft || !selRight) {
	// 		return;
	// 	}

	// 	const leftSelLayer = selLeft.querySelector('.left-selection-layer');
	// 	const rightSelLayer = selRight.querySelector('.right-selection-layer');
	// 	const visTime = timelineBlock.querySelectorAll('.vis-minor.vis-text:not(.vis-measure)');
	// 	const visDay = timelineBlock.querySelector('.vis-major:not(.vis-measure) > div')?.textContent;

	// 	let leftDateString = this._timestampToString(beginTime);
	// 	let rightDateString = this._timestampToString(endTime);

	// 	let selSeparator: HTMLElement | null = timelineBlock.querySelector('.selection-separator');

	// 	this._setSelectionByTime(beginTime, endTime);

	// 	let selection = [selLeft, selRight];
	// 	//save selection width in variable
	// 	let selWidth = 0;
	// 	for (let i = 0; i < selection.length; i++) {
	// 		selWidth += selection[i].offsetWidth;
	// 	}

	// 	//append clear selection button
	// 	if (selWidth > 0) {
	// 		if (selSeparator && !selSeparator.querySelector('.clear-selection')) {
	// 			let clearSelection = document.createElement('div');
	// 			clearSelection.className = 'clear-selection';

	// 			selSeparator.classList.remove('hide');
	// 			selSeparator.appendChild(clearSelection);

	// 			let self = this;
	// 			clearSelection.addEventListener('click', function(e){
	// 				if (!selLeft || !selRight) {
	// 					return;
	// 				}

	// 				self._clearSelectionBars(selLeft, selRight);
	// 			});
	// 		}

	// 		if (!selLeft.querySelector('.left-selection-tip') || !selRight.querySelector('.right-selection-tip')) {
	// 			let leftSelTip = document.createElement('div'),
	// 				rightSelTip = document.createElement('div'),
	// 				blockLayer = document.createElement('div');

	// 			blockLayer.className = 'block-layer';

	// 			leftSelTip.className = 'left-selection-tip';
	// 			rightSelTip.className = 'right-selection-tip';

	// 			timelineBlock.appendChild(blockLayer);
	// 			selLeft.appendChild(leftSelTip);
	// 			selRight.appendChild(rightSelTip);
	// 		}

	// 		const startDate = new Date();
	// 		const endDate = new Date();
	// 		startDate.setTime(beginTime);
	// 		endDate.setTime(endTime);

	// 		const leftSelectionTip = selLeft.querySelector('.left-selection-tip');
	// 		if (leftSelectionTip) {
	// 			leftSelectionTip.textContent = `${String(startDate.getHours()).padStart(2, '0')}:${String(startDate.getMinutes()).padStart(2, '0')}:${String(startDate.getSeconds()).padStart(2, '0')}`;
	// 		}

	// 		const rightSelectionTip = selLeft.querySelector('.right-selection-tip');
	// 		if (rightSelectionTip) {
	// 			rightSelectionTip.textContent = `${String(endDate.getHours()).padStart(2, '0')}:${String(endDate.getMinutes()).padStart(2, '0')}:${String(endDate.getSeconds()).padStart(2, '0')}`;
	// 		}

	// 		if (!this.selection) {
	// 			this.selection = {
	// 				start: beginTime,
	// 				end: endTime
	// 			};
	// 		}

	// 		if (visDay) {
	// 			let	selStringL = visDay.concat(' ', leftDateString);
	// 			let selStringR = visDay.concat(' ', rightDateString);
	// 			this._selection = {
	// 				start: new Date(selStringL),
	// 				end: new Date(selStringR)
	// 			}
	// 		}
	// 	}
	// 	isCallback && this._onSelectionChange(beginTime, endTime);
	// };

	/**
	 * get current selection
	 *
	 * @returns {null|{start: number, end: number}}
	 */
	// getSelection(): TimeLineInterval | null {
	// 	if (!this._node) {
	// 		return null;
	// 	}

	// 	let isSelection = this._node.querySelector(".left-selection-tip");
	// 	if (!isSelection) {
	// 		return null;
	// 	}

	// 	const timelineWidth = this.getWidth();
	// 	let center = timelineWidth * this._centerPosition;

	// 	let left = center - (this._node.querySelector(".tl-selection.left-selection") as HTMLElement)?.offsetWidth;
	// 	let right = center + (this._node.querySelector(".tl-selection.right-selection") as HTMLElement)?.offsetWidth;

	// 	let positionLeft = left / timelineWidth;
	// 	let positionRight = right / timelineWidth;

	// 	let positionLeftTime = this._getTime(positionLeft);
	// 	let positionRightTime = this._getTime(positionRight);

	// 	return {
	// 		start: positionLeftTime,
	// 		end: positionRightTime
	// 	};
	// };

	/**
	 * clear selection in timeline
	 */
	// clearSelection() {
	// 	if (!this._node) {
	// 		return;
	// 	}

	// 	let blockLayer = this._node.querySelector('.block-layer');
	// 	if (blockLayer) blockLayer.remove();

	// 	const selectionSeparator = this._node.querySelector('.selection-separator');
	// 	if (selectionSeparator) {
	// 		selectionSeparator.classList.add('hide');
	// 	}

	// 	let leftSelection: HTMLElement | null = this._node.querySelector('.left-selection');
	// 	let rightSelection: HTMLElement | null = this._node.querySelector('.right-selection');
	// 	if (leftSelection && rightSelection) {
	// 		this._clearSelectionBars(leftSelection, rightSelection);

	// 		if (leftSelection.parentNode && leftSelection.parentNode) {
	// 			(leftSelection.parentNode as HTMLElement).classList.add('hide');
	// 		}
	// 	}

	// 	this._onSelectionChange();
	// 	this.selection = null;
	// };

	/**
	 * event handler for selection change event
	 *
	 * @param {number|null} beginTime begin of the time interval in Unix time in ms
	 * @param {number|null} endTime end of the time interval in Unix time in ms
	 */
	// _onSelectionChange(beginTime?: number, endTime?: number) {
	// 	this.onSelectionChange("up", beginTime, endTime);
	// };

	/**
	 * event handler for selection change event
	 *
	 * @param {string} line 'up'|'down'
	 * @param {number|null} beginTime begin of the time interval in Unix time in ms
	 * @param {number|null} endTime end of the time interval in Unix time in ms
	 */
	// onSelectionChange(line: string, beginTime?: number, endTime?: number) {};

	onDebug(message: string, type: string) {
		console.log('onDebug method was not implemented yet!');
	};

	onGetImages(line: string, beginTime: number, endTime: number) {
		console.log('onGetImages method was not implemented yet!');
	};

	onImageClick(args: ImageClickEventArgs) {}

	setImages(line: string, timestampList: []) {
		console.log('setImages method was not implemented yet!');
	};

	clearImages() {
		console.log('clearImages method was not implemented yet!');
	};

	nextImagesPage() {
		console.log('nextImagesPage method was not implemented yet!');
	};

	prevImagesPage() {
		console.log('prevImagesPage method was not implemented yet!');
	};

	// getSelectionParameters() {
	// 	console.log('getSelectionParameters method was not implemented yet!');
	// };

	/**
	 * event handler for time change event
	 *
	 * @param {number} time Unix time in ms
	 * @param {boolean} isCallback
	 * @param {boolean} changedByUser
	 */
	_onTimeChange(time: number, isCallback = true, changedByUser: boolean) {
		if (!this._TIMELINE) {
			return;
		}

		this._time = time;
		this._positionSeparator();

		if (this._parameters.showTime) {
			const timeIndicator = this._node?.querySelector('.timeline-time-indicator');
			if (timeIndicator) {
				timeIndicator.innerHTML = moment(this._time).format("DD.MM.YYYY, HH:mm:ss");
			}
		}

		let window = this._TIMELINE.getWindow();
		let start = window.start.getTime();
		let end = window.end.getTime();

		if (Math.abs(this._lastRequestedData.start - start) >= this.MIN_REQUEST_DATA_INTERVAL
			||
			Math.abs(this._lastRequestedData.end - end) >= this.MIN_REQUEST_DATA_INTERVAL
		) {
			this._lastRequestedData.start = start;
			this._lastRequestedData.end = end;

			let granularity: Granularity = Granularity.chunk;
			switch (this._parameters.options.timeAxis?.scale) {
        case 'second':
				case 'minute':
					granularity = Granularity.min;
					break;
				case 'hour':
					granularity = Granularity.hour;
					break;
				case 'day':
					granularity = Granularity.day;
					break;
				default:
				break;
			}

			this.onGetData('up', start, end, granularity);
		}

		if (isCallback && Math.abs(this._lastRequestedData.time - time) > this.MIN_TIME_CHANGE_INTERVAL) {
			this._lastRequestedData.time = time;
			this.onTimeChange(time, changedByUser);
		}
	};

	/**
	 * event handler for time change event
	 *
	 * @param {number} time Unix time in ms
	 * @param {boolean} changedByUser
	 */
	onTimeChange(time: number, changedByUser: boolean) {};

	/**
	 * event handler for requesting data event
	 *
	 * @param {string} line
	 * @param {number} beginTime begin of the time interval in Unix time in ms
	 * @param {number} endTime end of the time interval in Unix time in ms
	 * @param {string} granularity 'year'|'mon'|'day'|'hour'|'min'|'second'
	 */
	onGetData(line: string, beginTime: number, endTime: number, granularity: Granularity) {};

	/**
	 * event handler for mouse down event
	 */
	onRangeChange() {};

	clearColorScale() {
		console.log('clearColorScale not implemented yet');
	};

	/**
	 * adds vertical center bar
	 */
	_addCenterBar() {
		if (!this._node) {
			return;
		}

		let timelineBlock = this._node;
		let centerBar = document.createElement('div');

		centerBar.className = 'center-bar';
		timelineBlock.appendChild(centerBar);
	};

	/**
	 * adds separator
	 */
	_addSeparator() {
		if (!this._node) {
			return;
		}

		let timelineBlock = this._node;
		let selectionSeparator = document.createElement('div');
		selectionSeparator.className = 'selection-separator';
		const timeIndicator = document.createElement('div');

		if (this._parameters.showTime) {
			selectionSeparator.appendChild(timeIndicator);
		}

		timelineBlock.appendChild(selectionSeparator);
	}

	/**
	 * adds draggable selection block
	 */
	// _addSelectionBars() {
	// 	if (!this._node) {
	// 		return;
	// 	}

	// 	let timelineBlock = this._node;
	// 	let selectionSeparator = document.createElement('div');
	// 	// let leftSelection = document.createElement('div');
	// 	// let rightSelection = document.createElement('div');
	// 	// let leftArrow = document.createElement('div');
	// 	// let leftSelectionLayer = document.createElement('div');
	// 	// let rightArrow = document.createElement('div');
	// 	// let rightSelectionLayer = document.createElement('div');
	// 	const timeIndicator = document.createElement('div');
	// 	let self = this;
	// 	let direction = "";
	// 	let oldX = 0;

	// 	selectionSeparator.className = 'selection-separator';
	// 	// leftSelection.className = 'tl-selection';
	// 	// rightSelection.className = 'tl-selection';
	// 	// leftArrow.className = 'left-selection-arrow';
	// 	// rightArrow.className = 'right-selection-arrow';
	// 	// leftSelectionLayer.className = 'left-selection-layer';
	// 	// rightSelectionLayer.className = 'right-selection-layer';
	// 	timeIndicator.className = 'timeline-time-indicator';

	// 	leftSelection.classList.add('left-selection');
	// 	rightSelection.classList.add('right-selection');

	// 	leftSelection.appendChild(leftArrow);
	// 	rightSelection.appendChild(rightArrow);
	// 	leftSelection.appendChild(leftSelectionLayer);
	// 	rightSelection.appendChild(rightSelectionLayer);

	// 	if (!this._parameters.hideSelection) {
	// 		selectionSeparator.appendChild(leftSelection);
	// 		selectionSeparator.appendChild(rightSelection);
	// 	}

	// 	if (this._parameters.showTime) {
	// 		selectionSeparator.appendChild(timeIndicator);
	// 	}

	// 	timelineBlock.appendChild(selectionSeparator);

	// 	if (this._parameters.hideSelection) {
	// 		return;
	// 	}

	// 	selectionSeparator.addEventListener('mousedown', function(e){
	// 		// if not LEFT mouse button
	// 		if (!(e.button === 0)) {
	// 			return;
	// 		}

	// 		if (e.target === leftArrow || e.target === rightArrow) {
	// 			self._relatedTarget = e.target as HTMLElement;
	// 			oldX = e.pageX;
	// 			const width = self.getWidth()

	// 			leftSelectionLayer.style.width = width * self._centerPosition + 'px';
	// 			leftSelectionLayer.style.height = timelineBlock.offsetHeight + 'px';

	// 			rightSelectionLayer.style.width = width * (1 - self._centerPosition) + 'px';
	// 			rightSelectionLayer.style.height = timelineBlock.offsetHeight + 'px';

	// 			selectionSeparator.addEventListener('mousemove', selectionArrowDrag);
	// 			window.addEventListener('mouseup', selectionArrowDragEnd);
	// 		}
	// 	});

	// 	/*mousemove event function*/
	// 	const selectionArrowDrag = (e: MouseEvent) => {
	// 		if (!this._node) {
	// 			return;
	// 		}

	// 		const getLeftSelectionTipTime = (startSelection: number): string => {
	// 			let start = new Date();
	// 			start.setTime(startSelection);
	// 			return `
	// 					${start.getHours() < 10 ? String(0) + start.getHours() : start.getHours()}:
	// 					${start.getMinutes() < 10 ? String(0) + start.getMinutes() : start.getMinutes()}:
	// 					${start.getSeconds() < 10 ? String(0) + start.getSeconds() : start.getSeconds()}
	// 				`;
	// 		};
	// 		const getRightSelectionTipTime = (endSelection: number): string => {
	// 			let end = new Date();
	// 			end.setTime(endSelection);
	// 			return `
	// 					${end.getHours() < 10 ? String(0) + end.getHours() : end.getHours()}:
	// 					${end.getMinutes() < 10 ? String(0) + end.getMinutes() : end.getMinutes()}:
	// 					${end.getSeconds() < 10 ? String(0) + end.getSeconds() : end.getSeconds()}
	// 				`;
	// 		};

	// 		if (!leftSelection.querySelector('.left-selection-tip') || !rightSelection.querySelector('.right-selection-tip')) {
	// 			let leftSelTip = document.createElement('div'),
	// 				rightSelTip = document.createElement('div'),
	// 				blockLayer = document.createElement('div'),
	// 				currentCenterTime = self._getCenterTime();

	// 			leftSelTip.className = 'left-selection-tip';
	// 			rightSelTip.className = 'right-selection-tip';
	// 			blockLayer.className = 'block-layer';

	// 			leftSelTip.textContent = getLeftSelectionTipTime(currentCenterTime);
	// 			rightSelTip.textContent = getRightSelectionTipTime(currentCenterTime);

	// 			leftSelection.appendChild(leftSelTip);
	// 			rightSelection.appendChild(rightSelTip);
	// 			this._node.appendChild(blockLayer);
	// 		}
	// 		selectionSeparator.classList.remove('hide');
	// 		if (this._relatedTarget === leftArrow || this._relatedTarget === rightArrow) {
	// 			// TODO: draw time of selected interval on drag, not on mouseup
	// 			let offset = 0;

	// 			if (e.pageX < oldX) {
	// 				direction = 'l';
	// 				offset = oldX - e.pageX;
	// 			} else
	// 			if (e.pageX > oldX) {
	// 				direction = 'r';
	// 				offset = e.pageX - oldX;
	// 			}

	// 			oldX = e.pageX;
	// 			let width: string = "";
	// 			const timelineWidth = self.getWidth();
	// 			const parentNode: HTMLElement | null = this._relatedTarget.parentNode as HTMLElement;
	// 			if (direction === 'l') {
	// 				if (self._relatedTarget === leftArrow) {
	// 					if (parentNode.offsetWidth < timelineWidth * self._centerPosition - leftArrow.offsetWidth) {
	// 						width = parentNode.offsetWidth + offset + 'px';
	// 					} else {
	// 						width = timelineWidth * self._centerPosition - leftArrow.offsetWidth + 'px';
	// 					}
	// 				} else
	// 				if (self._relatedTarget === rightArrow) {
	// 					width = parentNode.offsetWidth - offset + 'px';
	// 				}
	// 			} else
	// 			if (direction === 'r') {
	// 				if (self._relatedTarget === leftArrow) {
	// 					width = parentNode.offsetWidth - offset + 'px';
	// 				} else
	// 				if (self._relatedTarget === rightArrow) {
	// 					if (parentNode.offsetWidth < timelineWidth * (1 - self._centerPosition) - rightArrow.offsetWidth) {
	// 						width = parentNode.offsetWidth + offset + 'px';
	// 					} else {
	// 						width = timelineWidth * (1 - self._centerPosition) - rightArrow.offsetWidth + 'px';
	// 					}
	// 				}
	// 			}

	// 			if (self._relatedTarget === leftArrow) {
	// 				let selection = self.getSelection();
	// 				if (selection) {
	// 					(leftSelection.querySelector('.left-selection-tip') as HTMLElement).textContent = getLeftSelectionTipTime(selection.start);
	// 				}
	// 			} else
	// 			if (self._relatedTarget === rightArrow) {
	// 				let selection = self.getSelection();
	// 				if (selection) {
	// 					(rightSelection.querySelector('.right-selection-tip') as HTMLElement).textContent = getRightSelectionTipTime(selection.end);
	// 				}
	// 			}

	// 			if (width !== null) {
	// 				parentNode.style.width = width;
	// 			}
	// 		}
	// 	};

	// 	/*mouseup event function*/
	// 	function selectionArrowDragEnd(e: MouseEvent) {
	// 		if (!self._node) {
	// 			return;
	// 		}

	// 		// if not LEFT mouse button
	// 		if (!(e.button === 0)) {
	// 			return;
	// 		}

	// 		if (self._relatedTarget === leftArrow || self._relatedTarget === rightArrow) {
	// 			// const visDay = self._node.querySelector('.vis-major:not(.vis-measure) > div')?.textContent;
	// 			// const visTime = self._node.querySelectorAll('.vis-minor.vis-text:not(.vis-measure)');
	// 			const container = self._relatedTarget.parentNode?.parentNode;
	// 			const selection = self._node.querySelectorAll('.tl-selection');
	// 			let selWidth = 0;

	// 			if (!container) {
	// 				return;
	// 			}

	// 			for (let i = 0; i < selection.length; i++) {
	// 				selWidth += (selection[i] as HTMLElement).offsetWidth;
	// 			}

	// 			if (!container.querySelector('.clear-selection')) {
	// 				if (selWidth > 0) {
	// 					let clearSelection = document.createElement('div');

	// 					clearSelection.className = 'clear-selection';

	// 					container.appendChild(clearSelection);

	// 					clearSelection.addEventListener('click', function(e){
	// 						self.clearSelection();
	// 					});
	// 				}
	// 			} else {
	// 				if (selWidth === 0) {
	// 					const clearSelectionNode = container.querySelector('.clear-selection');
	// 					if (clearSelectionNode) {
	// 						container.removeChild(clearSelectionNode);
	// 					}

	// 					const blockLayerNode = self._node.querySelector('.block-layer');
	// 					if (blockLayerNode) {
	// 						blockLayerNode.remove();
	// 					}
	// 				}
	// 			}

	// 			/*show blocks with selection boundaries if selection exists*/
	// 			if (selWidth !== 0) {
	// 				const setLeftSelectionTipTime = (startSelection: number) => {
	// 					let start = new Date();
	// 					start.setTime(startSelection);

	// 					(leftSelection.querySelector('.left-selection-tip') as HTMLElement).textContent = `
	// 						${start.getHours() < 10 ? String(0) + start.getHours() : start.getHours()}:
	// 						${start.getMinutes() < 10 ? String(0) + start.getMinutes() : start.getMinutes()}:
	// 						${start.getSeconds() < 10 ? String(0) + start.getSeconds() : start.getSeconds()}
	// 					`;
	// 				};
	// 				const setRightSelectionTipTime = (endSelection: number) => {
	// 					let end = new Date();
	// 					end.setTime(endSelection);
	// 					(rightSelection.querySelector('.right-selection-tip') as HTMLElement).textContent = `
	// 						${end.getHours() < 10 ? String(0) + end.getHours() : end.getHours()}:
	// 						${end.getMinutes() < 10 ? String(0) + end.getMinutes() : end.getMinutes()}:
	// 						${end.getSeconds() < 10 ? String(0) + end.getSeconds() : end.getSeconds()}
	// 					`;
	// 				};

	// 				let selection = self.getSelection();
	// 				if (selection) {
	// 					setLeftSelectionTipTime(selection.start);
	// 					setRightSelectionTipTime(selection.end);

	// 					self._onSelectionChange(selection.start, selection.end);
	// 				}
	// 			} else {
	// 				self._clearSelectionBars(leftSelection, rightSelection);
	// 			}

	// 			self.selection = self.getSelection();
	// 			selectionSeparator.removeEventListener('mousemove', selectionArrowDrag);
	// 			window.removeEventListener('mouseup', selectionArrowDragEnd);
	// 			self._relatedTarget = null;
	// 		}
	// 	}

	// 	leftArrow.addEventListener('mouseenter', () => this._mouseInArrow = true);
	// 	leftArrow.addEventListener('mouseleave', () => this._mouseInArrow = false);
	// 	rightArrow.addEventListener('mouseenter', () => this._mouseInArrow = true);
	// 	rightArrow.addEventListener('mouseleave', () => this._mouseInArrow = false);
	// };

	/**
	 * clear user selection
	 *
	 * @param {Element} selLeft - left selection block
	 * @param {Element} selRight - right selection block
	 */
	// _clearSelectionBars(selLeft: HTMLElement, selRight: HTMLElement) {
	// 	if (!this._node) {
	// 		return;
	// 	}

	// 	this._selection = null;

	// 	const selectionArr = [selLeft, selRight];
	// 	const clearSelection = this._node.querySelector('.clear-selection');

	// 	for (let i = 0; i < selectionArr.length; i++) {
	// 		selectionArr[i].style.width = 0 + 'px';
	// 	}

	// 	if (selLeft.querySelector('.left-selection-tip') || selRight.querySelector('.right-selection-tip')) {
	// 		selLeft.removeChild(selLeft.querySelector('.left-selection-tip') as HTMLElement);
	// 		selRight.removeChild(selRight.querySelector('.right-selection-tip') as HTMLElement);
	// 	}

	// 	if (selLeft.querySelector('.left-selection-layer') || selRight.querySelector('.right-selection-layer')) {
	// 		(selLeft.querySelector('.left-selection-layer') as HTMLElement).style.width = '0px';
	// 		(selLeft.querySelector('.left-selection-layer') as HTMLElement).style.height = '0px';
	// 		(selRight.querySelector('.right-selection-layer') as HTMLElement).style.width = '0px';
	// 		(selRight.querySelector('.right-selection-layer') as HTMLElement).style.height = '0px';
	// 	}

	// 	if (clearSelection) {
	// 		clearSelection.parentNode?.removeChild(clearSelection);
	// 	}
	// };

	onScaleChange() {
		this._setWindow(this._getCenterTime());
	};

	onClick(props: TimelineEventPropertiesResult & {event: MouseEvent}) {
		if (props.what === 'group-label') {
			this._selectTrackByY(props.pageY);
			return;
		}

		if (!props.what) {
			const panel = this._node?.querySelector('.vis-panel.vis-left');
			if (!panel) {
				return;
			}
			const rect = panel.getBoundingClientRect();
			if (props.pageX > rect.left && props.pageX < rect.right &&
				props.pageY > rect.top && props.pageY < rect.bottom) {
				this.selectTrack(-1);
				return;
			}
		}

		if (props.what === 'item' || props.what === 'background') {
			const trackIndex = this._getTrackIndex(props.pageY);
			if (trackIndex < 0) {
				return;
			}
			const { startTime, endTime, count } = this._getEventCount(trackIndex, props.x);
			if (count > 0) {
				this.onEventClick(startTime, endTime, this._tracks[trackIndex].obj);
			}
		}
	}

	private _selectTrackByY(pageY: number): void {
		const trackIndex = this._getTrackIndex(pageY);
		if (trackIndex >= 0) {
			this.selectTrack(trackIndex);
		}
	}

	private _addTrackClass(trackIndices: number[], className: string): void {
		const timeline = this._node;
		if (!timeline) {
			return;
		}
		for (let i = 1; i < maxTrackCount + 1; i++) {
			const groupElements = Array.from(timeline.querySelectorAll(`.t${i}`));
			for (const element of groupElements) {
				if (trackIndices.includes(i - 1)) {
					element.classList.add(className);
				} else {
					element.classList.remove(className);
				}
			}
		}
	}

	private _getTrackObjIndex(trackIndex: number): number | undefined {
		if (trackIndex < 0 || trackIndex >= this._tracks.length) {
			return undefined;
		}
		const track = this._tracks[trackIndex];
		return this._tracks.filter((t, index) => (t.obj === track.obj || (!t.obj && !track.obj)) && index < trackIndex).length;
	}

	private _getTrackInfo(index: number): TrackInfo {
		return { index, obj: this._tracks[index].obj, objIndex: this._getTrackObjIndex(index) };
	}

	private _findTrackIndex(info: TrackInfo): number {
		if (info.index >= 0 && info.index < this._tracks.length
		    && (
			    this._tracks[info.index].obj === info.obj
			    || !this._tracks[info.index].obj && !info.obj
		    )
		    && this._getTrackObjIndex(info.index) === info.objIndex
		) {
			return info.index;
		}
		return this._tracks.findIndex(
			(t, index) => (t.obj === info.obj || (!t.obj && !info.obj)) && info.objIndex === this._getTrackObjIndex(index)
		);
	}

	selectedTrack(): number {
		return this._selectedTrack?.index ?? -1;
	}

	selectTrack(trackIndex: number): void {
		const isValid = trackIndex >= 0 && trackIndex < this._tracks.length;
		if (trackIndex === this._selectedTrack?.index || (!isValid && !this._selectedTrack)) {
			return;
		}
		this._addTrackClass([trackIndex], 'highlighted');
		this._selectedTrack = isValid ? this._getTrackInfo(trackIndex) : undefined;
		this.onTrackSelected(trackIndex);
	}

	selectTrackByObj(obj: UUID): void {
		const trackIndex = this._tracks.findIndex(t => t.obj === obj);
		if (trackIndex >= 0) {
			this.selectTrack(trackIndex);
		}
	}

	onTrackSelected(trackIndex: number): void {}
	onEventClick(startTime: number, endTime: number, obj?: UUID): void {}

	async onGetEventInfo(startTime: number, endTime: number, obj: UUID): Promise<TimeLineEventInfo> {
		return { time: new Date(0), html: '', snapshot: '' };
	}

	setGrayedTracks(trackIndices: number[]): void {
		this._addTrackClass(trackIndices, 'grayed');
		const indexSet = new Set<number>(trackIndices);
		this._grayedTracks = Array.from(indexSet.keys())
			.filter(index => index >= 0 && index < this._tracks.length)
			.map(index => this._getTrackInfo(index));
	}

	getWidth(): number {
		const timeline = this._node;
		if (!timeline) {
			return this._minWidth;
		}
		const panel: HTMLElement | null = timeline.querySelector('.vis-panel.vis-background.vis-vertical');
		return panel ? panel.offsetWidth : timeline.offsetWidth;
	}

	private _getTimeByX(x: number): number {
		return this._getTime(x / this.getWidth());
	}

	private _getEventCount(trackIndex: number, x: number): { startTime: number, endTime: number, count: number } {
		const time = this._getTimeByX(x);
		const result = { startTime: time, endTime: time, count: 0 };
		if (time < 0 || time > Date.now()) {
			return result;
		}

		result.startTime = this._getTimeByX(x - 4);
		result.startTime = result.startTime > 0 ? result.startTime : time;
		result.endTime = this._getTimeByX(x + 4);

		const row = this._tracks[trackIndex].rows.find(r => r.isEvent);
		if (!row) {
			return result;
		}
		result.count = Object.keys(row.list)
			.map(key => Number(key))
			.filter(key => key >= result.startTime && key <= result.endTime)
			.reduce((accumulator, value) => accumulator + (row.list[value].eventCount ?? 1), 0);
		return result;
	}

	private _getSeparatorLeft(): string {
		const timeline = this._node;
		const panel = timeline?.querySelector('.vis-panel.vis-background.vis-vertical');
		if (!timeline || !panel || !this._TIMELINE) {
			return this._centerPosition * 100 + '%';
		}
		const time = this._time || this._getCenterTime();
		const { start, end } = this._TIMELINE.getWindow();
		const pos = (time - start.getTime()) / (end.getTime() - start.getTime());
		const tlRect = timeline.getBoundingClientRect();
		const pnRect = panel.getBoundingClientRect();
		return (pnRect.left - tlRect.left) + pnRect.width * pos + 'px';
	}

	private _positionSeparator(): void {
		const selectionSeparator: HTMLElement | null | undefined = this._node?.querySelector(".selection-separator");
		if (selectionSeparator) {
			selectionSeparator.style.left = this._getSeparatorLeft();
		}
	}

	private _scaleTimeAxis(start: number, end: number): void {
		if (!this._TIMELINE || !this._parameters.options.zoomable) {
			return;
		}
		const tickCount = Math.trunc(this.getWidth() / TimeLineBlockWidth.MINUTE);
		const timeAxis: Required<TimelineTimeAxisOption> = { scale: 'millisecond', step: Math.round((end - start) / tickCount) };
		if (timeAxis.step < 1000) {
			this._roundStep(timeAxis, 1);
		} else
		if (timeAxis.step < 60 * 1000) {
			timeAxis.scale = 'second';
			this._roundStep(timeAxis, 1000, 'minute');
		} else
		if (timeAxis.step < 60 * 60 * 1000) {
			timeAxis.scale = 'minute';
			this._roundStep(timeAxis, 60 * 1000, 'hour');
		} else {
			timeAxis.scale = 'hour';
			this._roundStep(timeAxis, 60 * 60 * 1000);
		}

		this._TIMELINE.setOptions({ timeAxis });
		this._parameters.options = { ...this._parameters.options, timeAxis };
	}

	private _roundStep(timeAxis: Required<TimelineTimeAxisOption>, divider: number,  nextScale?: TimelineTimeAxisScaleType): void {
		timeAxis.step /= divider;
		timeAxis.step = timeAxis.step - Math.trunc(timeAxis.step) <= 0.1 ? Math.trunc(timeAxis.step) : Math.trunc(timeAxis.step) + 1;
		if (timeAxis.step > 100) {
			timeAxis.step += 100 - timeAxis.step % 100;
		} else
		if (timeAxis.step > 30 && nextScale) {
			timeAxis.step = 1;
			timeAxis.scale = nextScale;
		} else
		if (timeAxis.step > 10) {
			timeAxis.step += 10 - timeAxis.step % 10;
		} else
		if (timeAxis.step > 5) {
			timeAxis.step = 10;
		} else
		if (timeAxis.step > 2) {
			timeAxis.step = 5;
		}
	}

	getWidthInMilliseconds(): number {
		if (!this._TIMELINE) {
			return 0;
		}
		const { start, end } = this._TIMELINE.getWindow();
		return end.getTime() - start.getTime();
	}

	private _addZoomButtons(): void {
		const timeline = this._node;
		if (!timeline) {
			return;
		}

		const plusBtn = document.createElement('div');
		plusBtn.className = 'timeline-zoom-button plus';
		plusBtn.addEventListener('click', () => !this._zoomPlusDisabled && this._TIMELINE?.zoomIn(0.5));
		const plusImg = document.createElement('img');
		plusImg.src = PlusIcon;
		plusBtn.appendChild(plusImg);
		timeline.appendChild(plusBtn);

		const minusBtn = document.createElement('div');
		minusBtn.className = 'timeline-zoom-button minus';
		minusBtn.addEventListener('click', () => !this._zoomMinusDisabled && this._TIMELINE?.zoomOut(0.5));
		const minusImg = document.createElement('img');
		minusImg.src = MinusIcon;
		minusBtn.appendChild(minusImg);
		timeline.appendChild(minusBtn);

		const svgs = timeline.querySelectorAll('.timeline-zoom-button > img');
		SVGInjector(svgs);
	}

	private _addRemoveZoomButtonsClass(className: string, add: boolean, selector?: string): void {
		const zoomButtons = this._node?.querySelectorAll('.timeline-zoom-button' + (selector ?? ''));
		if (zoomButtons) {
			for (const btn of zoomButtons) {
				if (add) {
					btn.classList.add(className);
				} else {
					btn.classList.remove(className);
				}
			}
		}
	}
}
